﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_LinkMonitorImpl-hardware.nx.h"

namespace nn { namespace xcd { namespace detail {

LinkMonitorImpl::LinkMonitorImpl() NN_NOEXCEPT :
    m_NwcpManager(),
    m_Activated(false),
    m_SuspendState(SuspendState_Resumed),
    m_IsBleDisabled(false)
{
    // 何もしない
}

LinkMonitorImpl::~LinkMonitorImpl() NN_NOEXCEPT
{
    // 何もしない
}
void LinkMonitorImpl::StartMonitoring(nn::os::LightEventType* pUpdateEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pUpdateEvent);
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    m_pUpdateEvent = pUpdateEvent;

    ::nn::os::InitializeLightEvent(&m_BluetoothEvent, false, ::nn::os::EventClearMode_ManualClear);
    m_BluetoothTask.Activate(&m_BluetoothEvent);
    // BluetoothTask で BTM の初期化をした後に BleCentralTask をアクティベートする
    ::nn::os::InitializeLightEvent(&m_BleEvent, false, ::nn::os::EventClearMode_ManualClear);
    if (m_IsBleDisabled == false)
    {
        m_BleCentralTask.Activate(&m_BleEvent);
    }

    ::nn::os::InitializeLightEvent(&m_UartEvent, false, ::nn::os::EventClearMode_ManualClear);
    m_NwcpManager.StartMonitoring(&m_UartEvent);

    ::nn::os::InitializeLightEvent(&m_UsbHidEvent, false, ::nn::os::EventClearMode_ManualClear);
    m_UsbHidManager.StartMonitoring(&m_UsbHidEvent);

    ::nn::os::InitializeLightEvent(&m_SuspendEvent, false, ::nn::os::EventClearMode_ManualClear);

    GetTaskManager().RegisterPeriodicTask(this);

    m_SuspendState = SuspendState_Resumed;
    m_Activated = true;
}

void LinkMonitorImpl::StopMonitoring() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    ::nn::os::FinalizeLightEvent(&m_SuspendEvent);

    if (m_IsBleDisabled == false)
    {
        m_BleCentralTask.Deactivate();
    }
    ::nn::os::FinalizeLightEvent(&m_BleEvent);

    m_BluetoothTask.Deactivate();
    ::nn::os::FinalizeLightEvent(&m_BluetoothEvent);

    m_NwcpManager.StopMonitoring();
    ::nn::os::FinalizeLightEvent(&m_UartEvent);

    m_UsbHidManager.StopMonitoring();
    ::nn::os::FinalizeLightEvent(&m_UsbHidEvent);

    m_SuspendState = SuspendState_Suspended;
    m_Activated = false;
}

void LinkMonitorImpl::Suspend() NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Suspend Start\n");
    if (m_SuspendState == SuspendState_Resumed)
    {
        m_SuspendState = SuspendState_ToSuspend;
        ::nn::os::InitializeLightEvent(&m_NwcpSuspendCompleteEvent, false, ::nn::os::EventClearMode_ManualClear);
        ::nn::os::InitializeLightEvent(&m_UsbHidSuspendCompleteEvent, false, ::nn::os::EventClearMode_ManualClear);

        // Suspend 処理を開始
        ::nn::os::SignalLightEvent(&m_SuspendEvent);

        // Suspend 処理が完了するまで待つ
        ::nn::os::WaitLightEvent(&m_NwcpSuspendCompleteEvent);
        ::nn::os::WaitLightEvent(&m_UsbHidSuspendCompleteEvent);

        ::nn::os::FinalizeLightEvent(&m_NwcpSuspendCompleteEvent);
        ::nn::os::FinalizeLightEvent(&m_UsbHidSuspendCompleteEvent);

        m_SuspendState = SuspendState_Suspended;
    }
    NN_DETAIL_XCD_INFO("Suspend Complete\n");
}

void LinkMonitorImpl::Resume() NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Resume\n");
    if (m_SuspendState == SuspendState_Suspended)
    {
        m_SuspendState = SuspendState_ToResume;
        ::nn::os::InitializeLightEvent(&m_ResumeCompleteEvent, false, ::nn::os::EventClearMode_ManualClear);

        // Resume 処理を開始
        ::nn::os::SignalLightEvent(&m_SuspendEvent);

        // Resume 処理が完了するまで待つ
        ::nn::os::WaitLightEvent(&m_ResumeCompleteEvent);

        ::nn::os::FinalizeLightEvent(&m_ResumeCompleteEvent);

        m_SuspendState = SuspendState_Resumed;
    }
    else if (m_SuspendState == SuspendState_Resumed)
    {
        // TORIAEZU: FullAwake のときはここを通る
        // 切断時再起動を有効にする
        m_NwcpManager.SetRebootEnabled(true);
    }
    NN_DETAIL_XCD_INFO("Resume Complete\n");
}

void LinkMonitorImpl::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void LinkMonitorImpl::PeriodicEventFunction() NN_NOEXCEPT
{
    if (::nn::os::TryWaitLightEvent(&m_BluetoothEvent))
    {
        ::nn::os::ClearLightEvent(&m_BluetoothEvent);
        ::nn::os::SignalLightEvent(m_pUpdateEvent);

        // Bluetooth のデバイスの一覧を取得
        HidDeviceInfo btDevices[HidDeviceCountMax];
        auto btDeviceCount = m_BluetoothTask.GetDevices(btDevices, HidDeviceCountMax);
        // 接続済み Bluetooth デバイスを UsbManager に通知
        m_UsbHidManager.SetBluetoothDeviceList(btDevices, btDeviceCount);
    }
    else if (::nn::os::TryWaitLightEvent(&m_UartEvent))
    {
        ::nn::os::ClearLightEvent(&m_UartEvent);

        HidDeviceInfo nwcpDevices[NwcpDeviceCountMax];
        auto nwcpDeviceCount = m_NwcpManager.GetDevices(nwcpDevices, NwcpDeviceCountMax);
        HidDeviceInfo bluetoothDevices[HidDeviceCountMax];
        auto bluetoothDeviceCount = m_BluetoothTask.GetDevices(bluetoothDevices, HidDeviceCountMax);

        // Uart に切り替わったら Bluetooth は切断
        for (int i = 0; i < nwcpDeviceCount; i++)
        {
            for (int j = 0; j < bluetoothDeviceCount; j++)
            {
                if (nwcpDevices[i].address == bluetoothDevices[j].address)
                {
                    m_BluetoothTask.DetachDevice(bluetoothDevices[j].deviceHandle);
                }
            }
        }

        nn::os::SignalLightEvent(m_pUpdateEvent);
    }
    else if (::nn::os::TryWaitLightEvent(&m_UsbHidEvent))
    {
        ::nn::os::ClearLightEvent(&m_UsbHidEvent);

        HidDeviceInfo usbDevices[UsbHidDeviceCountMax];
        auto usbDeviceCount = m_UsbHidManager.GetDevices(usbDevices, UsbHidDeviceCountMax);
        HidDeviceInfo bluetoothDevices[HidDeviceCountMax];
        auto bluetoothDeviceCount = m_BluetoothTask.GetDevices(bluetoothDevices, HidDeviceCountMax);

        // USB に切り替わったら Bluetooth は切断
        for (int i = 0; i < usbDeviceCount; i++)
        {
            for (int j = 0; j < bluetoothDeviceCount; j++)
            {
                if (usbDevices[i].address == bluetoothDevices[j].address)
                {
                    m_BluetoothTask.DetachDevice(bluetoothDevices[j].deviceHandle);
                }
            }
        }

        nn::os::SignalLightEvent(m_pUpdateEvent);
    }
    else if (::nn::os::TryWaitLightEvent(&m_SuspendEvent))
    {
        ::nn::os::ClearLightEvent(&m_SuspendEvent);

        if (m_SuspendState == SuspendState_ToSuspend)
        {
            m_BleCentralTask.Suspend();
            m_BluetoothTask.Suspend();
            m_NwcpManager.TriggerSuspend(&m_NwcpSuspendCompleteEvent);
            m_UsbHidManager.TriggerSuspend(&m_UsbHidSuspendCompleteEvent);
            return;
        }
        if (m_SuspendState == SuspendState_ToResume)
        {
            m_UsbHidManager.Resume();
            m_NwcpManager.Resume();
            m_BluetoothTask.Resume();
            m_BleCentralTask.Resume();
            ::nn::os::SignalLightEvent(&m_ResumeCompleteEvent);
            return;
        }
    }
    else if (::nn::os::TryWaitLightEvent(&m_BleEvent))
    {
        ::nn::os::ClearLightEvent(&m_BleEvent);
        ::nn::os::SignalLightEvent(m_pUpdateEvent);
    }

}

size_t LinkMonitorImpl::GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // Nwcp のデバイス一覧を取得
    auto nwcpDeviceCount = m_NwcpManager.GetDevices(pOutValue, deviceCount);
    auto returnCount = nwcpDeviceCount;

    // Nwcp デバイスの Bluetooth アドレスを取得
    ::nn::bluetooth::Address nwcpDeviceAddresses[NwcpDeviceCountMax];
    for (int i = 0; i < nwcpDeviceCount; i++)
    {
        nwcpDeviceAddresses[i] = pOutValue[i].address;
    }

    // Bluetooth のデバイスの一覧を取得
    HidDeviceInfo btDevices[HidDeviceCountMax];
    auto btDeviceCount = m_BluetoothTask.GetDevices(btDevices, HidDeviceCountMax);

    // Uart と Bd アドレスが重複しないもののみ追加
    for (int i = 0; i < btDeviceCount; i++)
    {
        // 上限に到達していたら終了
        if (returnCount == deviceCount)
        {
            break;
        }

        bool isSameDevice = false;
        // BtAddr が一致するか確認
        for (int j = 0; j < nwcpDeviceCount; j++)
        {
            if (btDevices[i].address == pOutValue[j].address)
            {
                isSameDevice = true;
                break;
            }
        }

        if (isSameDevice == false)
        {
            pOutValue[returnCount] = btDevices[i];
            returnCount++;
        }
    }

    HidDeviceInfo usbDevices[UsbHidDeviceCountMax];
    auto usbDeviceCount = m_UsbHidManager.GetDevices(usbDevices, UsbHidDeviceCountMax);

    // Usb デバイスの追加
    for (int i = 0; i < usbDeviceCount; i++)
    {
        // 上限に到達していたら終了
        if (returnCount == deviceCount)
        {
            break;
        }

        pOutValue[returnCount] = usbDevices[i];
        returnCount++;
    }

    return returnCount;
}

size_t LinkMonitorImpl::GetBleDevices(uint32_t *pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // Ble デバイス一覧を取得
    return m_BleCentralTask.GetConnectionHandleList(pOutValue, deviceCount);
}

HidAccessor* LinkMonitorImpl::GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    auto pHidAccessor = m_BluetoothTask.GetHidAccessor(deviceHandle);

    if (pHidAccessor == nullptr)
    {
        pHidAccessor = m_NwcpManager.GetHidAccessor(deviceHandle);
    }

    if (pHidAccessor == nullptr)
    {
        pHidAccessor = m_UsbHidManager.GetHidAccessor(deviceHandle);
    }

    return pHidAccessor;
}

BleHidAccessor* LinkMonitorImpl::GetBleHidAccessor(uint32_t connectionHandle) NN_NOEXCEPT
{
    auto pAccessor = m_BleCentralTask.GetBleHidAccessor(connectionHandle);
    return pAccessor;
}

void LinkMonitorImpl::SetSlotSizeUpdateEvent(DeviceHandle deviceHandle, ::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    m_BluetoothTask.SetSlotSizeEvent(deviceHandle, pEvent);
}

Result LinkMonitorImpl::SetSlotSize(DeviceHandle deviceHandle, int slotSize) NN_NOEXCEPT
{
    NN_RESULT_DO(m_BluetoothTask.SetSlotSize(deviceHandle, slotSize));
    NN_RESULT_SUCCESS;
}

int LinkMonitorImpl::GetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    return m_BluetoothTask.GetSlotSize(deviceHandle);
}

int LinkMonitorImpl::GetTargetSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    return m_BluetoothTask.GetTargetSlotSize(deviceHandle);
}

bool LinkMonitorImpl::IsChangingSlotSize(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    return m_BluetoothTask.IsChangingSlotSize(deviceHandle);
}

bool LinkMonitorImpl::IsRightNwcpAttached() NN_NOEXCEPT
{
    return m_NwcpManager.IsRightNwcpAttached();
}

bool LinkMonitorImpl::IsLeftNwcpAttached() NN_NOEXCEPT
{
    return m_NwcpManager.IsLeftNwcpAttached();
}

void LinkMonitorImpl::SetNwcpEnabled(bool enabled) NN_NOEXCEPT
{
    return m_NwcpManager.SetNwcpEnabled(enabled);
}

bool LinkMonitorImpl::IsConsolePowered() NN_NOEXCEPT
{
    return m_NwcpManager.IsConsolePowered();
}

void LinkMonitorImpl::SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    m_NwcpManager.SetRailUpdateEvent(pEvent);
}

void LinkMonitorImpl::GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);

    m_NwcpManager.GetRailUpdateEventType(pOutEventType, pOutAddress);
}

AwakeTriggerReason LinkMonitorImpl::GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT
{
    return m_NwcpManager.GetAwakeTriggerReasonForLeftRail();
}

AwakeTriggerReason LinkMonitorImpl::GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT
{
    return m_NwcpManager.GetAwakeTriggerReasonForRightRail();
}

bool LinkMonitorImpl::IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    return m_UsbHidManager.IsUsbHidSupported(deviceInfo);
}

Result LinkMonitorImpl::AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(m_UsbHidManager.AddDevice(index, deviceInfo));
    NN_RESULT_SUCCESS;
}

Result LinkMonitorImpl::RemoveUsbHidDevice(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(m_UsbHidManager.RemoveDevice(index));
    NN_RESULT_SUCCESS;
}

Result LinkMonitorImpl::SetUsbHidInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(m_UsbHidManager.SetInputReport(index, pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t LinkMonitorImpl::GetUsbHidOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
{
    return m_UsbHidManager.GetOutputReport(index, pOutBuffer, length);
}

void LinkMonitorImpl::SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT
{
    return m_UsbHidManager.SetFullKeyUsbEnabled(enabled);
}

bool LinkMonitorImpl::IsUsbConnected(DeviceHandle handle) NN_NOEXCEPT
{
    // Usb のデバイスの一覧を取得
    HidDeviceInfo usbDevices[UsbHidDeviceCountMax];
    auto usbDeviceCount = m_UsbHidManager.GetDevices(usbDevices, UsbHidDeviceCountMax);

    for (int i = 0; i < usbDeviceCount; i++)
    {
        if (usbDevices[i].deviceHandle == handle)
        {
            // 既に Usb デバイスとして登録されていたら true を返す
            return true;
        }
    }

    // Bluetooth のデバイスの一覧を取得
    HidDeviceInfo btDevices[HidDeviceCountMax];
    auto btDeviceCount = m_BluetoothTask.GetDevices(btDevices, HidDeviceCountMax);

    // Usb と Bluetooth で Bd アドレスが重複するかチェック
    for (int i = 0; i < btDeviceCount; i++)
    {
        if (btDevices[i].deviceHandle == handle)
        {
            return m_UsbHidManager.IsUsbConnected(btDevices[i].address);
        }
    }

    return false;
}

void LinkMonitorImpl::UpdateUsbHidDeviceLists() NN_NOEXCEPT
{
    return m_UsbHidManager.UpdateUsbHidDeviceLists();
}


void LinkMonitorImpl::ReInitializePendingUsbDevices() NN_NOEXCEPT
{
    return m_UsbHidManager.ReInitializePendingDevices();
}

int LinkMonitorImpl::GetMaxBluetoothLinks() NN_NOEXCEPT
{
    return m_BluetoothTask.GetMaxLinks();
}

Result LinkMonitorImpl::SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT
{
    return m_UsbHidManager.SetFirmwareUpdatingDevice(handle);
}

void LinkMonitorImpl::UnsetFirmwareUpdatingDevice() NN_NOEXCEPT
{
    m_UsbHidManager.UnsetFirmwareUpdatingDevice();
}

void LinkMonitorImpl::DisableBle() NN_NOEXCEPT
{
    m_IsBleDisabled = true;
}

void LinkMonitorImpl::SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT
{
    m_BleCentralTask.SetIsPalmaAllConnectable(connectable);
}

void LinkMonitorImpl::SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
{
    m_BleCentralTask.SetIsPalmaPairedConnectable(connectable);
}

Result LinkMonitorImpl::PairPalma(const ::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    NN_RESULT_DO(m_BleCentralTask.PairPalma(address));
    NN_RESULT_SUCCESS;
}

bool LinkMonitorImpl::IsFiftyConnected() NN_NOEXCEPT
{
    return m_NwcpManager.IsFiftyConnected();
}

PadState LinkMonitorImpl::GetFiftyPadState() NN_NOEXCEPT
{
    return m_NwcpManager.GetFiftyPadState();
}

Result LinkMonitorImpl::GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT
{
    return m_UsbHidManager.GetKuinaVersion(pOutMcuVersionData,index);
}

Result LinkMonitorImpl::RequestKuinaVersion(int index) NN_NOEXCEPT
{
    return m_UsbHidManager.RequestKuinaVersion(index);
}

Result LinkMonitorImpl::SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT
{
    return m_UsbHidManager.SetKuinaToFirmwareUpdateMode(index);
}


}}} // namespace nn::xcd::detail
