﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>
#include <nn/xcd/xcd_FirmwareTypes.h>

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   FW イメージを読み込むクラス
 */
class FirmwareUpdaterStream
{
    NN_DISALLOW_COPY(FirmwareUpdaterStream);
    NN_DISALLOW_MOVE(FirmwareUpdaterStream);

public:
    FirmwareUpdaterStream() NN_NOEXCEPT;

    /**
     * @brief   ストリームのセットアップ
     */
    virtual nn::Result Setup(const FirmwareImage& image) NN_NOEXCEPT;

    /**
     * @brief   FW イメージのサイズを取得
     */
    virtual size_t GetImageSize() const NN_NOEXCEPT { return m_Image.imageSize; }

    /**
     * @brief   読み込み済みのデータサイズを取得
     */
    size_t GetTotalReadBytes() const NN_NOEXCEPT { return m_ReadBytes; }

    /**
     * @brief   残りデータサイズを取得
     */
    size_t GetRemainBytes() const NN_NOEXCEPT { return GetImageSize() - GetTotalReadBytes(); }

    /**
     * @brief   終端まで読み込み済みか判定
     */
    bool IsEof() const NN_NOEXCEPT { return GetTotalReadBytes() >= GetImageSize(); }

    /**
     * @brief   FW イメージを読み込む
     */
    virtual nn::Result Read(uint8_t* pOutBuffer, size_t* pOutReadLength, size_t readLength) NN_NOEXCEPT
    {
        return ReadImpl(pOutBuffer, pOutReadLength, readLength, GetTotalReadBytes());
    }

    /**
     * @brief   読み込み位置を巻き戻す
     */
    void Rewind() NN_NOEXCEPT { m_ReadBytes = 0; }

protected:
    nn::Result ReadImpl(uint8_t* pOutBuffer, size_t* pOutReadLength, size_t readLength, size_t offset) NN_NOEXCEPT;

private:
    FirmwareImage       m_Image;        //!< FW イメージの情報
    size_t              m_ReadBytes;    //!< 読み込んだデータ量
};

}}}  // nn::xcd::detail
