﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Aardvark ラッパーライブラリの型や定数の定義です。
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace xcd { namespace detail { namespace aardvark {

/**
 * @brief   Aardvark デバイスのハンドルを表す型です。
 */
typedef int DeviceHandle;

/**
 * @brief   Aardvark の動作モード設定です。
 */
enum Mode
{
    Mode_GpioOnly   = 0x00,     //!< GPIO のみ
    Mode_SpiAndGpio = 0x01,     //!< SPI + GPIO
    Mode_GpioAndI2c = 0x02,     //!< GPIO + I2C
    Mode_SpiAndI2c  = 0x03,     //!< SPI + I2C
    Mode_Query      = 0x80      //!< 現在のモードの取得
};

/**
 * @brief   SPI の動作モードです。
 */
enum SpiMode
{
    SpiMode_Master,     //!< マスター
    SpiMode_Slave       //!< スレーブ
};

/**
 * @brief   SPI のシリアルクロックの極性です。
 */
enum SpiPolarity
{
    SpiPolarity_ActiveLow  = 0,     //!< 負論理 (Active Low)
    SpiPolarity_ActiveHigh = 1      //!< 正論理 (Active High)
};

/**
 * @brief   SPI のシリアルクロックの位相です。
 */
enum SpiPhase
{
    SpiPhase_SampleSetup = 0,
    SpiPhase_SetupSample = 1
};

/**
 * @brief   SPI のビット伝送順です。
 */
enum SpiBitOrder
{
    SpiBitOrder_MsbFirst = 0,   //!< MSB から順に伝送
    SpiBitOrder_LsbFirst = 1    //!< LSB から順に伝送
};

/**
 * @brief   SPI の SS (Slave Select) ピンの極性です。
 */
enum SpiMasterSsPolarity
{
    SpiMasterSsPolarity_ActiveLow  = 0,     //!< 0 が入力されると有効
    SpiMasterSsPolarity_ActiveHigh = 1      //!< 1 が入力されると有効
};

/**
 * @brief   SPI の設定を格納する構造体です。
 */
struct SpiConfig
{
    SpiPolarity         polarity;           //!< クロック極性
    SpiPhase            phase;              //!< クロック位相
    SpiBitOrder         bitOrder;           //!< ビット伝送順
};

/**
 * @brief   GPIO のピン定義です。
 */
enum GpioPins : nn::Bit8
{
    GpioPins_Scl  = 0x01,   //!< SCL (I2C Clock)
    GpioPins_Sda  = 0x02,   //!< SDA
    GpioPins_Miso = 0x04,   //!< MISO
    GpioPins_Sck  = 0x08,   //!< SCK (SPI Clock)
    GpioPins_Mosi = 0x10,   //!< MOSI
    GpioPins_Ss   = 0x20,   //!< SS

    GpioPins_None = 0x00,
    GpioPins_All  = 0xFF
};

enum TeraGpioPins : nn::Bit8
{
    TeraGpioPins_Reset      = GpioPins_Sda,   //!< リセットピン。High でリセット解除
    TeraGpioPins_SystemBoot = GpioPins_Scl,   //!< ブートモードピン。High で System boot

    TeraGpioPins_None       = GpioPins_None
};

}}}}
