﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cctype>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/wlan/wlan_Ssid.h>

namespace nn {
namespace wlan {

Ssid::Ssid() NN_NOEXCEPT
{
    m_Length = 0;
    std::memset(&m_Ssid[0], 0x00, SsidLengthMax);
}

Ssid::Ssid(const Bit8* pSsid, size_t length) NN_NOEXCEPT
{
    // SsidLengthMaxを超えていたら、間引く
    if( length > SsidLengthMax )
    {
        length = SsidLengthMax;
    }
    Set(pSsid, length);
}

Ssid::Ssid(const char* pSsidString) NN_NOEXCEPT
{
    m_Length = static_cast<size_t>(nn::util::Strnlen(pSsidString, static_cast<int>(SsidLengthMax)));

    // まず0埋め
    std::memset(&m_Ssid[0], 0x00, SsidLengthMax);
    // SSID埋め込み
    std::memcpy(&m_Ssid[0], pSsidString, m_Length);
}

bool Ssid::Set(const Bit8* pSsid, size_t length) NN_NOEXCEPT
{
    if( length > SsidLengthMax )
    {
        NN_SDK_LOG("[Warning]nn::wlan::Ssid.Set Invalid length.\n");
        return false;

        // SsidLengthMaxを超えていたら、間引く
        length = SsidLengthMax;
    }

    m_Length = length;

    // まず0埋め
    std::memset(&m_Ssid[0], 0x00, SsidLengthMax);
    // SSID埋め込み
    std::memcpy(&m_Ssid[0], pSsid, length);

    return true;
}

bool Ssid::Set(const char* pSsidString) NN_NOEXCEPT
{
    size_t length;
    length = static_cast<size_t>(nn::util::Strnlen(pSsidString, static_cast<int>(SsidLengthMax + 1)));

    if( length > SsidLengthMax )
    {
        NN_SDK_LOG("[Warning]nn::wlan::Ssid.Set Invalid length.\n");
        return false;
    }

    m_Length = length;

    // まず0埋め
    std::memset(&m_Ssid[0], 0x00, SsidLengthMax);
    // SSID埋め込み
    std::memcpy(&m_Ssid[0], pSsidString, length);

    return true;
}

char* Ssid::GetHexString(char outputSsidString[SsidHexStringLengthMax]) const NN_NOEXCEPT
{
    uint32_t dstIndex = 0;
    for(uint32_t i = 0; i < m_Length && dstIndex < SsidHexStringLengthMax; ++i)
    {
        if (std::isprint(m_Ssid[i]))
        {
            outputSsidString[dstIndex++] = m_Ssid[i];
        }
        else
        {
            dstIndex += nn::util::TSNPrintf(&outputSsidString[dstIndex], SsidHexStringLengthMax - dstIndex,
                                            "\\x%02X", m_Ssid[i]);
        }
    }
    outputSsidString[dstIndex] = '\0';

    return outputSsidString;
}


} // end of namespace wlan
} // end of namespace nn

