﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>
#include "visrv_Config.h"

namespace nn{ namespace visrv{

    // System event handling
    namespace detail{
        static const int SystemEventWaitEntryCountMax = 32;

        struct SystemEventWaitEntry
        {
            nn::os::MultiWaitHolderType* pHolder;
            nn::os::SystemEventType* pEvent;
            void (*onSignaledCallback)(void*);
            void* onSignaledParameter;
        };

        class SystemEventWaitList
        {
        public:
            SystemEventWaitList() NN_NOEXCEPT
            {
                std::memset(m_EntryList, 0, sizeof(m_EntryList));
            }

            void Initialize() NN_NOEXCEPT
            {
                std::memset(m_EntryList, 0, sizeof(m_EntryList));
            }

            void Set(int index, const SystemEventWaitEntry& entry) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(entry.pHolder);
                NN_SDK_REQUIRES_NOT_NULL(entry.pEvent);
                NN_SDK_REQUIRES_NOT_NULL(entry.onSignaledCallback);
                NN_SDK_REQUIRES_EQUAL(m_EntryList[index].pHolder, nullptr);

                m_EntryList[index] = entry;
            }

            void Remove(int index) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_RANGE(index, 0, SystemEventWaitEntryCountMax);
                std::memset(&m_EntryList[index], 0, sizeof(m_EntryList[index]));
            }

            SystemEventWaitEntry& Get(int index) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_RANGE(index, 0, SystemEventWaitEntryCountMax);
                NN_SDK_REQUIRES_NOT_NULL(m_EntryList[index].pHolder);
                return m_EntryList[index];
            }

        private:
            SystemEventWaitEntry m_EntryList[SystemEventWaitEntryCountMax];
        };
    }

    //
    class ServerManager
    {
    public:
        static const int SessionCountMax = ClientProcessCountMax;
        static const int PortCount = 3; // vi:u, vi:s, vi:m

    public:
        struct DisplayServerManagerOption
        {
            static const int SubDomainCountMax = 32;
            static const int ObjectInSubDomainCountMax = 128;
            static const bool CanDeferInvokeRequest = true;
        };

        class DisplayServerManager
            : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCount, DisplayServerManagerOption>
        {
        public:
            DisplayServerManager() NN_NOEXCEPT;

            nn::Result ProcessMyInvokeRequest(nn::os::MultiWaitHolderType* pSession) NN_NOEXCEPT;

            // セッションを Defer する場合には、セッション中からこの関数を呼び出してハンドラを設定しておく。
            void SetDeferExecutionHandler(void (*pHandler)(DisplayServerManager* pManager, nn::os::MultiWaitHolderType* pSession, void* pUserPtr), void* pUserPtr) NN_NOEXCEPT;
        private:
            void (*m_pDeferExecutionHandler)(DisplayServerManager* pManager, nn::os::MultiWaitHolderType* pSession, void* pUserPtr);
            void* m_pDeferExecutionUserPtr;
        };

    public:
        ServerManager() NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        DisplayServerManager* GetDisplayServerManager() NN_NOEXCEPT;

        void Run() NN_NOEXCEPT;

    private:
        // system event handler
        void RegisterSystemEventWait(
            int slotIndex,
            nn::os::MultiWaitHolderType* pHolder,
            nn::os::SystemEventType* pEvent,
            void (*onSignaledCallback)(void*),
            void* onSignaledParameter
        ) NN_NOEXCEPT;
        void UnregisterSystemEventWait(int index) NN_NOEXCEPT;

    private:
        std::aligned_storage<sizeof(DisplayServerManager), NN_ALIGNOF(DisplayServerManager)>::type m_DisplayServerManagerStorage;
        DisplayServerManager* m_pDisplayServerManager;

        detail::SystemEventWaitList m_SystemEventWaitList;

    };

    extern ServerManager g_ServerManager;

}}
