﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_VicTaskQueue.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>

namespace nn{ namespace visrv{ namespace vic{

    IVicTask::~IVicTask() NN_NOEXCEPT
    {
    }

    const int VicTaskQueue::Size;
    const int VicTaskQueue::Capacity;

    VicTaskQueue::VicTaskQueue() NN_NOEXCEPT
    {
        std::memset(m_TaskList, 0, sizeof(m_TaskList));
        m_Head = 0;
        m_Tail = 0;
        std::memset(&m_Mutex, 0, sizeof(m_Mutex));
        std::memset(&m_ConditionVariable, 0, sizeof(m_ConditionVariable));
    }

    void VicTaskQueue::Initialize() NN_NOEXCEPT
    {
        nn::os::InitializeMutex(&m_Mutex, false, 0);
        nn::os::InitializeConditionVariable(&m_ConditionVariable);
        m_Head = 0;
        m_Tail = 0;
    }

    void VicTaskQueue::Finalize() NN_NOEXCEPT
    {
        nn::os::FinalizeConditionVariable(&m_ConditionVariable);
        nn::os::FinalizeMutex(&m_Mutex);
    }

    void VicTaskQueue::Enqueue(IVicTask* pTask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTask);
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT { nn::os::UnlockMutex(&m_Mutex); };

            for(;;)
            {
                int nextTail = (m_Tail + 1) % Size;
                if(nextTail != m_Head)
                {
                    break;
                }
                nn::os::WaitConditionVariable(&m_ConditionVariable, &m_Mutex);
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(EnqueueImpl(pTask));
        }
        nn::os::SignalConditionVariable(&m_ConditionVariable);
    }

    void VicTaskQueue::Dequeue(IVicTask** ppOutTask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppOutTask);
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            for(;;)
            {
                if(m_Head != m_Tail)
                {
                    break;
                }
                nn::os::WaitConditionVariable(&m_ConditionVariable, &m_Mutex);
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(DequeueImpl(ppOutTask));
        }
        nn::os::SignalConditionVariable(&m_ConditionVariable);
    }

    nn::Result VicTaskQueue::EnqueueImpl(IVicTask* pTask) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(m_Head, 0, Size);
        NN_SDK_ASSERT_RANGE(m_Tail, 0, Size);

        int nextTail = (m_Tail + 1) % Size;
        NN_RESULT_THROW_UNLESS(nextTail != m_Head, nn::vi::ResultNotReady());

        NN_SDK_ASSERT(m_TaskList[m_Tail] == nullptr);
        m_TaskList[m_Tail] = pTask;
        m_Tail = nextTail;
        NN_RESULT_SUCCESS;
    }

    nn::Result VicTaskQueue::DequeueImpl(IVicTask** ppOutTask) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(m_Head, 0, Size);
        NN_SDK_ASSERT_RANGE(m_Tail, 0, Size);

        NN_RESULT_THROW_UNLESS(m_Head != m_Tail, nn::vi::ResultNotReady());

        NN_SDK_ASSERT_NOT_NULL(m_TaskList[m_Head]);
        *ppOutTask = m_TaskList[m_Head];
        m_TaskList[m_Head] = nullptr;
        m_Head = (m_Head + 1) % Size;
        NN_RESULT_SUCCESS;
    }

}}}
