﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_VicMemoryPool.h"

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include "visrv_VicConfig.h"
#include "../visrv_Log.h"

namespace nn{ namespace visrv{ namespace vic{

    size_t VicMemoryPool::GetRequiredAlignment() NN_NOEXCEPT
    {
        return MemoryPoolRequiredAlignment;
    }

    size_t VicMemoryPool::GetRequiredUnitSize() NN_NOEXCEPT
    {
        return MemoryPoolRequiredUnitSize;
    }

    VicMemoryPool::VicMemoryPool() NN_NOEXCEPT
        : m_hMemory(0)
        , m_pMemory(nullptr)
        , m_Size(0)
    {
    }

    nn::Result VicMemoryPool::Initialize(VicModule* pModule, void* pMemory, size_t size) NN_NOEXCEPT
    {
        //auto tick0 = nn::os::GetSystemTick();
        //NN_UTIL_SCOPE_EXIT {
        //    auto tick1 = nn::os::GetSystemTick();
        //    NN_SDK_LOG("Init Pool %lldus\n", (tick1 - tick0).ToTimeSpan().GetMicroSeconds());
        //};
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES(pModule->IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pMemory);
        NN_SDK_REQUIRES_ALIGNED(pMemory, GetRequiredAlignment());
        NN_SDK_REQUIRES_GREATER(size, 0u);
        NN_SDK_REQUIRES_EQUAL(size % GetRequiredUnitSize(), 0u);

        size_t alignment = GetRequiredAlignment();
        NvRmDeviceHandle hDevice = pModule->GetDevice();
        NvRmMemKind kind = NvRmMemKind_Generic_16Bx2; // TORIAEZU:


        NVRM_DEFINE_MEM_HANDLE_ATTR(memAttributes);
        NVRM_MEM_HANDLE_SET_ATTR(
            memAttributes,
            static_cast<NvU32>(alignment),
            NvOsMemAttribute_WriteBack,
            static_cast<NvU32>(size),
            NVRM_MEM_TAG_DDK2D_MISC
        );
        NVRM_MEM_HANDLE_SET_KIND_ATTR(memAttributes, kind);
        NVRM_MEM_HANDLE_SET_VA_ATTR(memAttributes, reinterpret_cast<NvU64>(pMemory));

        NvError err = NvRmMemHandleAllocAttr(hDevice, &memAttributes, &m_hMemory);
        if(err != NvSuccess)
        {
            NN_VISRV_LOG_ERR("NvRmHandleAllocAttr failed(%d)\n", err);
        }
        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            nn::vi::ResultOperationFailed()
        );
        m_pMemory = pMemory;
        m_Size = size;

        NN_RESULT_SUCCESS;
    }

    void VicMemoryPool::Finalize() NN_NOEXCEPT
    {
        //auto tick0 = nn::os::GetSystemTick();
        //NN_UTIL_SCOPE_EXIT {
        //    auto tick1 = nn::os::GetSystemTick();
        //    NN_SDK_LOG("Finl Pool %lldus\n", (tick1 - tick0).ToTimeSpan().GetMicroSeconds());
        //};
        NN_SDK_REQUIRES(IsInitialized());
        NvRmMemHandleFree(m_hMemory);
        m_hMemory = 0;
        m_pMemory = nullptr;
        m_Size = 0;
    }

    bool VicMemoryPool::IsInitialized() const NN_NOEXCEPT
    {
        return m_hMemory != 0;
    }

    NvRmMemHandle VicMemoryPool::GetHandle() const NN_NOEXCEPT
    {
        return m_hMemory;
    }

    void* VicMemoryPool::GetPointer() const NN_NOEXCEPT
    {
        return m_pMemory;
    }

    uintptr_t VicMemoryPool::GetAddress() const NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(m_pMemory);
    }

    size_t VicMemoryPool::GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

}}}
