﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include "visrv_NativeType.h"

#include "detail/visrv_SyncpointEventManager.h"
#include "detail/visrv_SyncpointWaiterQueue.h"

namespace nn{ namespace visrv{ namespace native{
    class SyncpointWaiter;
    class SyncpointEntry;

    namespace detail
    {
        enum SyncpointEntryState
        {
            SyncpointEntryState_Free,
            SyncpointEntryState_Queued,
            SyncpointEntryState_Waiting,
            SyncpointEntryState_Expired,
            SyncpointEntryState_Canceled,
        };
    }


    class SyncpointEntry
    {
        NN_DISALLOW_COPY(SyncpointEntry);
        NN_DISALLOW_MOVE(SyncpointEntry);
    public:
        SyncpointEntry() NN_NOEXCEPT;
        explicit SyncpointEntry(const NvRmFence& fence) NN_NOEXCEPT;
        // fence 中の Syncpoint はこのオブジェクトにコピーされます。
        explicit SyncpointEntry(const android::sp<android::Fence>& fence) NN_NOEXCEPT;

        // 待ち中だった場合、キャンセルします。
        ~SyncpointEntry() NN_NOEXCEPT;

        // 空の状態に戻します。
        // 待ち中だった場合、キャンセルします。
        void Reset() NN_NOEXCEPT;

        // 新しく fence を設定します。
        // 待ち中だった場合、キャンセルします。
        void Reset(const NvRmFence& fence) NN_NOEXCEPT;

        // 新しく fence を設定します。
        // 待ち中だった場合、キャンセルします。
        // fence が nullptr だった場合、 fence なしになります。
        void Reset(const android::sp<android::Fence>& fence) NN_NOEXCEPT;

        // Syncpoint が Expire するまで待ちます
        void Wait() NN_NOEXCEPT;

        // Syncpoint がシグナルしているかチェックします
        bool TryWait() NN_NOEXCEPT;

        // SyncpointWaiter での待ちをキャンセルします
        void Cancel() NN_NOEXCEPT;

        // Syncpoint が Expire するのを待つための MultiWaitHolder を初期化します
        void InitializeWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

        void FinalizeWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    private:
        friend class SyncpointWaiter;

        // Queued にしてからキューに入れる、以降はワーカーが書き換える。
        // Expired または Canceled になった後は書き換えない。
        detail::SyncpointEntryState m_State;

        // 初期化時に設定して以降変更しない
        native::NativeRmSync m_Sync;

        // ワーカーがシグナルする。
        nn::os::EventType m_ExpiredEvent;

        // 必ず要求元で操作する。
        SyncpointWaiter* m_pWaiter;

        // Waiter が使用する。
        struct {
            int nextFence;
            int waitSlot;
        } m_WaiterPrivate;
    };


    class SyncpointWaiter
    {
    public:
        static const uintptr_t DequeueableTag      = 0x01;
        static const uintptr_t CancelRequestTag    = 0x02;
        static const uintptr_t ExpiringEventTagMin = 0x7000;
        static const uintptr_t ExpiringEventTagMax = ExpiringEventTagMin + detail::SyncpointEventManager::Size - 1;

    public:
        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void Enqueue(SyncpointEntry* pEntry) NN_NOEXCEPT;
        void Cancel(SyncpointEntry* pEntry) NN_NOEXCEPT;

        void Run() NN_NOEXCEPT;

    private:
        void ProcessDequeueImpl() NN_NOEXCEPT;
        void ProcessExpiredImpl(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        void ProcessCancelRequestImpl() NN_NOEXCEPT;
        // 待つ fence がなくなったら true を返す
        bool KickNextFenceWaitImpl(SyncpointEntry* pEntry) NN_NOEXCEPT;

        // EventManager の空きを見て Queue の Dequeue イベントを MultiWait にリンクするか決める。
        void UpdateDequeueableLinkImpl() NN_NOEXCEPT;

    private:
        detail::SyncpointWaiterQueue m_Queue;
        detail::SyncpointEventManager m_EventManager;

        // キャンセル要求元が取る Mutex。
        nn::os::MutexType m_CancelRequestMutex;
        nn::os::EventType m_CancelRequestEvent;
        nn::os::EventType m_CancelCompleteEvent;
        SyncpointEntry*   m_pCancelRequestEntry;

        nn::os::MultiWaitType        m_MultiWait;
        nn::os::MultiWaitHolderType  m_DequeueableWaitHolder;
        nn::os::MultiWaitHolderType  m_CancelRequestWaitHolder;

        bool m_IsDequeueableLinked;
    };

    extern SyncpointWaiter g_SyncpointWaiter;

}}}
