﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_BinderTable.h"

#include <nn/nn_SdkAssert.h>
#include <algorithm>

namespace nn{ namespace visrv{ namespace native{

    void BinderEntry::Initialize(
        const android::sp<android::IBinder>& pIBinder,
        BinderClassType binderClass,
        int32_t driverHandle,
        nn::vi::DisplayId displayId,
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES(pIBinder != nullptr);
        m_pIBinder = pIBinder;
        m_BinderClass = binderClass;
        m_DriverHandle = driverHandle;
        m_DisplayId = displayId;
        m_LayerId = layerId;
        m_RemoteStrongReferenceCount = 0;
        m_RemoteWeakReferenceCount = 0;
    }

    void BinderEntry::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pIBinder.clear();
        m_BinderClass = 0;
        m_DriverHandle = 0;
        m_DisplayId = 0;
        m_LayerId = 0;
        m_RemoteStrongReferenceCount = 0;
        m_RemoteWeakReferenceCount = 0;
    }

    bool BinderEntry::IsInitialized() const NN_NOEXCEPT
    {
        return m_pIBinder != nullptr;
    }

    BinderClassType BinderEntry::GetBinderClass() const NN_NOEXCEPT
    {
        return m_BinderClass;
    }

    int32_t BinderEntry::GetDriverHandle() const NN_NOEXCEPT
    {
        return m_DriverHandle;
    }

    nn::vi::DisplayId BinderEntry::GetDisplayId() const NN_NOEXCEPT
    {
        return m_DisplayId;
    }

    nn::vi::LayerId BinderEntry::GetLayerId() const NN_NOEXCEPT
    {
        return m_LayerId;
    }

    int BinderEntry::GetRemoteStrongReferenceCount() const NN_NOEXCEPT
    {
        return m_RemoteStrongReferenceCount;
    }

    int BinderEntry::GetRemoteWeakReferenceCount() const NN_NOEXCEPT
    {
        return m_RemoteWeakReferenceCount;
    }

    void BinderEntry::AddRemoteStrongReferenceCount(int delta) NN_NOEXCEPT
    {
        m_RemoteStrongReferenceCount += delta;
    }

    void BinderEntry::AddRemoteWeakReferenceCount(int delta) NN_NOEXCEPT
    {
        m_RemoteWeakReferenceCount += delta;
    }

    BinderEntry* BinderTable::AddEntry(
        const android::sp<android::IBinder>& pIBinder,
        BinderClassType binderClass,
        int32_t driverHandle,
        nn::vi::DisplayId displayId,
        nn::vi::LayerId layerId
        ) NN_NOEXCEPT
    {
        BinderEntry* pEntry = nullptr;
        for(int i = 0; i < Capacity; i++)
        {
            if(!m_EntryList[i].IsInitialized())
            {
                pEntry = &m_EntryList[i];
                break;
            }
        }
        if(pEntry == nullptr)
        {
            return nullptr;
        }
        pEntry->Initialize(pIBinder, binderClass, driverHandle, displayId, layerId);
        return pEntry;
    }

    BinderEntry* BinderTable::FindEntryByDriverHandle(int32_t driverHandle) NN_NOEXCEPT
    {
        BinderEntry* pEntry = nullptr;
        for(int i = 0; i < Capacity; i++)
        {
            auto p = &m_EntryList[i];
            if(p->IsInitialized() && p->GetDriverHandle() == driverHandle)
            {
                pEntry = p;
                break;
            }
        }
        return pEntry;
    }

    //void BinderTable::RemoveHandlesOnDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT
    //{
    //    for(int i = 0; i < Capacity; i++)
    //    {
    //        auto p = &m_EntryList[i];
    //        if(p->IsInitialized() && p->GetDisplayId() == displayId)
    //        {
    //            p->Finalize();
    //        }
    //    }
    //}

    void BinderTable::RemoveHandlesOnLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; i++)
        {
            auto p = &m_EntryList[i];
            if(p->IsInitialized() && p->GetLayerId() == layerId)
            {
                p->Finalize();
            }
        }
    }

    bool BinderTable::IsEmpty() const NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; i++)
        {
            if(m_EntryList[i].IsInitialized())
            {
                return false;
            }
        }
        return true;
    }

}}}
