﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Result.h>
#include "visrv_DefaultDisplay-spec.NX.h"
#include "visrv_IPhysicalDisplay.h"
#include "visrv_IDisplayStrategy.h"

namespace
{
    // std::make_index_sequence unavailable...
    template <std::size_t... List>
    struct Sequence
    {
    };

    template <std::size_t...>
    struct SequenceGenerator;

    template <std::size_t Index, std::size_t... List>
    struct SequenceGenerator<Index, List...>
    {
        using Result = typename SequenceGenerator<Index - 1, Index - 1, List...>::Result;
    };

    template <std::size_t... List>
    struct SequenceGenerator<0, List...>
    {
        using Result = Sequence<List...>;
    };

    template <std::size_t End>
    struct MakeSequence
    {
        using Result = typename SequenceGenerator<End - 1, End - 1>::Result;
    };

    template <typename... Args, std::size_t... Indicies>
    nn::Result VisitorImpl(nn::visrv::master::detail::IPhysicalDisplay* display,
                           nn::Result(nn::visrv::master::detail::IPhysicalDisplay::*fn)(Args...),
                           std::tuple<Args...> args,
                           Sequence<Indicies...>)
    {
        NN_SDK_ASSERT_NOT_NULL(display);
        NN_SDK_ASSERT_NOT_NULL(fn);

        return (display->*fn)(std::get<Indicies>(args)...);
    }

    template <typename Tuple>
    nn::Result Visitor(nn::visrv::master::detail::IPhysicalDisplay* display, void* userData) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(display);
        NN_SDK_ASSERT_NOT_NULL(userData);

        auto& tuple = *static_cast<Tuple*>(userData);
        return VisitorImpl(display, std::get<0>(tuple), std::get<1>(tuple), std::get<2>(tuple));
    }

    template <typename... Args>
    nn::Result Invoke(nn::visrv::master::detail::IDisplayStrategy* strategy,
                      nn::Result (nn::visrv::master::detail::IPhysicalDisplay::*fn)(Args...),
                      Args... args) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(strategy);
        NN_SDK_ASSERT_NOT_NULL(fn);

        auto forward = std::make_tuple(fn, std::make_tuple(args...), typename MakeSequence<sizeof...(Args)>::Result{});
        return strategy->Visit(Visitor<decltype(forward)>, &forward);
    }
}

namespace nn { namespace visrv { namespace master { namespace detail {

    DefaultDisplay::DefaultDisplay(const detail::PlatformDisplayInfoSet& infoSet,
                                   Allocator* pAllocator,
                                   detail::IDisplayStrategy* pStrategy)
        : Display(infoSet, pAllocator, pStrategy)
    {
    }

    nn::Result DefaultDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
    {
        NN_UNUSED(underscan);

        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutUnderscan);

        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::SetAlpha(float alpha) NN_NOEXCEPT
    {
        NN_UNUSED(alpha);

        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
    {
        NN_UNUSED(state);

        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::SetLayerStack(nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        NN_UNUSED(id);

        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::GetLayerStack(nn::vi::LayerStackType* pOutId) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutId);

        return nn::vi::ResultNotSupported();
    }

    bool DefaultDisplay::IsHotplugEventSupported() const NN_NOEXCEPT
    {
        return false;
    }

    nn::Result DefaultDisplay::GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutState);

        return nn::vi::ResultNotSupported();
    }

    bool DefaultDisplay::IsModeChangedEventSupported() const NN_NOEXCEPT
    {
        return false;
    }

    nn::Result DefaultDisplay::SetCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
    {
        return Invoke(m_pStrategy, &IPhysicalDisplay::SetCmuMode, mode);
    }

    nn::Result DefaultDisplay::SetContrastRatio(float ratio) NN_NOEXCEPT
    {
        return Invoke(m_pStrategy, &IPhysicalDisplay::SetContrastRatio, ratio);
    }

    nn::Result DefaultDisplay::GetGamma(float* pOutGamma) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutGamma);
        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::SetGamma(float gamma) NN_NOEXCEPT
    {
        NN_UNUSED(gamma);
        return nn::vi::ResultNotSupported();
    }

    nn::Result DefaultDisplay::SetViewport(int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(x, 0);
        NN_SDK_ASSERT_GREATER_EQUAL(y, 0);
        NN_SDK_ASSERT_GREATER(width, 0);
        NN_SDK_ASSERT_GREATER(height, 0);

        return Invoke(m_pStrategy, &IPhysicalDisplay::SetViewport, x, y, width, height);
    }

    nn::Result DefaultDisplay::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID) NN_NOEXCEPT
    {
        return Invoke(m_pStrategy, &IPhysicalDisplay::GetCompositorErrorInfo, pOutErrorInfo, pOutLength, errorID);
    }

}}}}
