﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectCopyImageMapDefer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nvgr.h>
#include "../vic/visrv_VicOperation.h"

#include "../visrv_ServerManager.h"
#include "visrv_IndirectLayer.h"
#include "../native/visrv_SyncpointWaiter.h"

namespace nn{ namespace visrv{ namespace indirect{

    IndirectMapCopyImageContextTable g_IndirectMapCopyImageContextTable;

    IndirectMapCopyImageContext::IndirectMapCopyImageContext() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_Parameter()
        , m_OutResult(nn::ResultSuccess())
        , m_OutImageSize()
        , m_OutImageStride()
        , m_CompleteEvent()
        , m_CompleteEventHolder()
        , m_MemoryPool()
        , m_ImageBuffer()
    {
    }

    nn::Result IndirectMapCopyImageContext::Initialize(const Parameter& param) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        bool isSuccess = false;

        nn::os::InitializeEvent(&m_CompleteEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_CompleteEventHolder, &m_CompleteEvent);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                nn::os::FinalizeMultiWaitHolder(&m_CompleteEventHolder);
                nn::os::FinalizeEvent(&m_CompleteEvent);
            }
        };

        m_Parameter = param;
        m_OutResult = nn::ResultSuccess();
        m_OutImageSize = 0;
        m_OutImageStride = 0;
        m_IsInitialized = true;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void IndirectMapCopyImageContext::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        nn::os::FinalizeMultiWaitHolder(&m_CompleteEventHolder);
        nn::os::FinalizeEvent(&m_CompleteEvent);
    }

    bool IndirectMapCopyImageContext::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    IndirectMapCopyImageContextParameter IndirectMapCopyImageContext::GetParameter() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_Parameter;
    }

    nn::Result IndirectMapCopyImageContext::GetOutResult() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_OutResult;
    }

    size_t IndirectMapCopyImageContext::GetOutImageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_OutImageSize;
    }

    size_t IndirectMapCopyImageContext::GetOutImageStride() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_OutImageStride;
    }

    nn::os::EventType* IndirectMapCopyImageContext::GetCompleteEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return &m_CompleteEvent;
    }

    nn::os::MultiWaitHolderType* IndirectMapCopyImageContext::GetCompleteEventHolder() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return &m_CompleteEventHolder;
    }

    void IndirectMapCopyImageContext::Execute() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        m_OutResult = ExecuteImpl();
    }

    nn::Result IndirectMapCopyImageContext::ExecuteImpl() NN_NOEXCEPT
    {
        nn::Result result = nn::ResultSuccess();
        NvRmFence fence = { NVRM_INVALID_SYNCPOINT_ID, 0 };

        // sMMU マップ作成
        result = InitializeImageBuffer(m_Parameter.pBuffer, m_Parameter.bufferSize, m_Parameter.width, m_Parameter.height);
        if(result.IsFailure())
        {
            goto KICKERROR;
        }

        // VIC をキック
        result = KickVicCopy(&fence);
        if(result.IsFailure())
        {
            goto KICKERROR;
        }

        // ReleaseFence が取れれば Source は不要
        UnlockSourceBuffer(fence);

        // VIC が終わるまで待つ
        WaitSyncpoint(fence);

        // sMMU マップ破棄
        FinalizeImageBuffer();

        // Destination 不要
        UnlockDestinationBuffer();

        // 処理完了
        SignalCompleteEvent();
        NN_RESULT_SUCCESS;

    KICKERROR:
        UnlockSourceBuffer({ NVRM_INVALID_SYNCPOINT_ID, 0 });
        FinalizeImageBuffer();
        UnlockDestinationBuffer();
        SignalCompleteEvent();
        return result;
    }

    nn::Result IndirectMapCopyImageContext::KickVicCopy(NvRmFence* pOutFence) NN_NOEXCEPT
    {
        vic::VicCopyOption option = {};
        vic::VicCopySingleDestinationInfo dst = {};
        vic::VicCopySingleSourceInfo src = {};

        pOutFence->SyncPointID = NVRM_INVALID_SYNCPOINT_ID;
        pOutFence->Value = 0;

        // option
        {
            option.backgroundColor = nn::util::Color4f(0, 0, 0, 0);
            option.filter = vic::VicCopyFilter_Bilinear;
            option.alphaMode = vic::VicCopyAlphaMode_Source;
        }
        // destination
        {
            dst.pSurface = m_ImageBuffer.GetSurface();
            dst.region.x = 0;
            dst.region.y = 0;
            dst.region.width  = m_Parameter.width;
            dst.region.height = m_Parameter.height;
        }
        // source
        {
            auto pReadyBuffer = m_Parameter.pSourceGraphicBuffer;
            auto pSrcNativeBuffer = pReadyBuffer->getNativeBuffer();
            const NvRmSurface* pSrcSurfaceList = nullptr;
            size_t srcSurfaceCount = 0;
            nvgr_get_surfaces(pSrcNativeBuffer->handle, &pSrcSurfaceList, &srcSurfaceCount);
            NN_RESULT_THROW_UNLESS(srcSurfaceCount == 1, nn::vi::ResultOperationFailed());
            src.pSurface = &pSrcSurfaceList[0];
            src.region.x      = static_cast<int>(pSrcNativeBuffer->width  * m_Parameter.sourceRectX);
            src.region.y      = static_cast<int>(pSrcNativeBuffer->height * m_Parameter.sourceRectY);
            src.region.width  = static_cast<int>(pSrcNativeBuffer->width  * m_Parameter.sourceRectWidth);
            src.region.height = static_cast<int>(pSrcNativeBuffer->height * m_Parameter.sourceRectHeight);
        }

        // kick
        NN_RESULT_DO(vic::VicOperation::CopySingle(pOutFence, dst, &vic::g_VicModule, src, option));
        NN_RESULT_SUCCESS;
    }

    nn::Result IndirectMapCopyImageContext::InitializeImageBuffer(void* pBuffer, size_t bufferSize, int width, int height) NN_NOEXCEPT
    {
        bool isSuccess = false;

        vic::VicImageBuffer::InfoType info = {};
        info.width = width;
        info.height = height;
        info.format = vic::VicImageFormat_Rgba8;
        size_t imageSize = vic::VicImageBuffer::GetRequiredMemorySize(info);

        size_t poolUnit = vic::VicMemoryPool::GetRequiredUnitSize();
        size_t poolSize = ((imageSize + poolUnit - 1) / poolUnit) * poolUnit;
        size_t poolAlignment = vic::VicMemoryPool::GetRequiredAlignment();
        NN_RESULT_THROW_UNLESS(reinterpret_cast<uintptr_t>(pBuffer) % poolAlignment == 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(bufferSize >= poolSize, nn::vi::ResultInvalidRange());

        NN_RESULT_DO(m_MemoryPool.Initialize(&vic::g_VicModule, pBuffer, poolSize));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_MemoryPool.Finalize();
            }
        };

        NN_RESULT_DO(m_ImageBuffer.Initialize(&vic::g_VicModule, info, &m_MemoryPool, 0, imageSize));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_ImageBuffer.Finalize();
            }
        };

        m_OutImageSize   = imageSize;
        m_OutImageStride = m_ImageBuffer.GetStride();
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void IndirectMapCopyImageContext::FinalizeImageBuffer() NN_NOEXCEPT
    {
        m_ImageBuffer.Finalize();
        m_MemoryPool.Finalize();
    }


    void IndirectMapCopyImageContext::UnlockSourceBuffer(const NvRmFence& fence) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Parameter.pSourceBufferStage);
        NN_SDK_REQUIRES_NOT_NULL(m_Parameter.pSourceBufferLockCounter);
        NN_SDK_REQUIRES_NOT_NULL(m_Parameter.pSourceGraphicBuffer);

        m_Parameter.pSourceBufferStage->UpdateVicReleaseFence(fence);
        m_Parameter.pSourceGraphicBuffer.clear();
        m_Parameter.pSourceBufferLockCounter->ReleaseLock();

        m_Parameter.pSourceBufferStage       = nullptr;
        m_Parameter.pSourceBufferLockCounter = nullptr;
    }

    void IndirectMapCopyImageContext::UnlockDestinationBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Parameter.pDestinationBoundLockCounter);
        m_Parameter.pDestinationBoundLockCounter->ReleaseLock();

        m_Parameter.pDestinationBoundLockCounter = nullptr;
    }

    void IndirectMapCopyImageContext::WaitSyncpoint(const NvRmFence& fence) NN_NOEXCEPT
    {
        native::SyncpointEntry e(fence);
        native::g_SyncpointWaiter.Enqueue(&e);
        e.Wait();
    }

    void IndirectMapCopyImageContext::SignalCompleteEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        nn::os::SignalEvent(&m_CompleteEvent);
    }


    //---------------------------------------------------
    const int IndirectMapCopyImageContextTable::Capacity;

    void IndirectMapCopyImageContextTable::Initialize() NN_NOEXCEPT
    {
        std::memset(m_Memory, 0, sizeof(m_Memory));
        std::memset(m_DeferringInfo, 0, sizeof(m_DeferringInfo));
        std::memset(m_StateList, 0, sizeof(m_StateList));
        m_CurrentContextIndex = -1;
    }

    void IndirectMapCopyImageContextTable::Finalize() NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; i++)
        {
            NN_SDK_REQUIRES_EQUAL(m_StateList[i], State_Free);
        }
    }

    nn::Result IndirectMapCopyImageContextTable::Acquire(IndirectMapCopyImageContext** pOut) NN_NOEXCEPT
    {
        int index = -1;
        for(int i = 0; i < Capacity; i++)
        {
            if(m_StateList[i] == State_Free)
            {
                index = i;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(index >= 0, nn::vi::ResultResourceLimit());

        auto p = m_Memory + sizeof(IndirectMapCopyImageContext) * index;
        auto pContext = new (p) IndirectMapCopyImageContext();
        NN_SDK_ASSERT_NOT_NULL(pContext);

        m_StateList[index] = State_Used;
        *pOut = pContext;
        NN_RESULT_SUCCESS;
    }

    void IndirectMapCopyImageContextTable::Release(IndirectMapCopyImageContext* p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);

        int index = GetIndexOf(p);
        NN_SDK_ASSERT_EQUAL(m_StateList[index], State_Used);

        p->~IndirectMapCopyImageContext();
        m_StateList[index] = State_Free;
    }

    IndirectMapCopyImageContext* IndirectMapCopyImageContextTable::GetCurrentContext() NN_NOEXCEPT
    {
        if(m_CurrentContextIndex < 0)
        {
            return nullptr;
        }
        NN_SDK_ASSERT_RANGE(m_CurrentContextIndex, 0, Capacity);
        return reinterpret_cast<IndirectMapCopyImageContext*>(m_Memory + sizeof(IndirectMapCopyImageContext) * m_CurrentContextIndex);
    }

    int IndirectMapCopyImageContextTable::GetIndexOf(const IndirectMapCopyImageContext* p) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);
        NN_SDK_REQUIRES_ALIGNED(p, NN_ALIGNOF(IndirectMapCopyImageContext));
        NN_SDK_REQUIRES_RANGE(reinterpret_cast<uintptr_t>(p), reinterpret_cast<uintptr_t>(m_Memory), reinterpret_cast<uintptr_t>(m_Memory + sizeof(m_Memory)));

        int index = static_cast<int>( (reinterpret_cast<uintptr_t>(p) - reinterpret_cast<uintptr_t>(m_Memory)) / sizeof(IndirectMapCopyImageContext) );
        NN_SDK_ASSERT_RANGE(index, 0, Capacity);
        return index;
    }

    void IndirectMapCopyImageContextTable::DeferExecution(IndirectMapCopyImageContext* p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(p);

        int index = GetIndexOf(p);
        auto pHolder = p->GetCompleteEventHolder();
        pHolder->userData = static_cast<uintptr_t>(MultiWaitIndex_IndirectMapCopyImageMin + index);
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitIndex_IndirectMapCopyImageMin, MultiWaitIndex_IndirectMapCopyImageMax);

        auto pDeferringInfo = &m_DeferringInfo[index];
        NN_SDK_ASSERT(pDeferringInfo->pSession == nullptr);
        NN_SDK_ASSERT(pDeferringInfo->pServerManager == nullptr);

        auto pServerManager = g_ServerManager.GetDisplayServerManager();
        pServerManager->SetDeferExecutionHandler(DeferExecutionHandler, pDeferringInfo);
        pServerManager->AddUserWaitHolder(pHolder);
    }

    void IndirectMapCopyImageContextTable::ResumeExecution(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_REQUIRES_MINMAX(pHolder->userData, MultiWaitIndex_IndirectMapCopyImageMin, MultiWaitIndex_IndirectMapCopyImageMax);

        int index = static_cast<int>(pHolder->userData - MultiWaitIndex_IndirectMapCopyImageMin);
        NN_SDK_ASSERT_EQUAL(m_StateList[index], State_Used);

        auto deferringInfo = m_DeferringInfo[index];
        NN_SDK_ASSERT_NOT_NULL(deferringInfo.pSession);
        NN_SDK_ASSERT_NOT_NULL(deferringInfo.pServerManager);

        m_DeferringInfo[index].pServerManager = nullptr;
        m_DeferringInfo[index].pSession = nullptr;

        m_CurrentContextIndex = index;
        NN_ABORT_UNLESS_RESULT_SUCCESS(deferringInfo.pServerManager->ProcessMyInvokeRequest(deferringInfo.pSession));
        m_CurrentContextIndex = -1;
    }

    void IndirectMapCopyImageContextTable::DeferExecutionHandler(ServerManager::DisplayServerManager* pManager, nn::os::MultiWaitHolderType* pSession, void* pUserPtr) NN_NOEXCEPT
    {
        auto pDeferringInfo = reinterpret_cast<DeferringInfo*>(pUserPtr);
        pDeferringInfo->pServerManager = pManager;
        pDeferringInfo->pSession = pSession;
    }

}}}
