﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include "../../visrv_Config.h"
#include "../../local/visrv_LocalManagedLayer.h"
#include "../../local/visrv_LocalIndirectLayer.h"
#include "../../util/visrv_TQueue.h"
#include "../../util/visrv_TIntegerContainer.h"
#include "../../native/visrv_SyncpointWaiter.h"
#include "../../visrv_ResourceIdManagement.h"

namespace nn{ namespace visrv{
    class ServerManager;
}}

namespace nn{ namespace visrv{ namespace client{
    class ClientObject;
}}}

namespace nn{ namespace visrv{ namespace fbshare{
    class SharedBuffer;
}}}

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{
    class SharedFrameBuffer;
    class SharedLowLevelLayerManager;
    class SharedClientLayer;

    enum SharedLowLevelLayerState
    {
        SharedLowLevelLayerState_Free,
        SharedLowLevelLayerState_Bound,
        SharedLowLevelLayerState_Connected,
    };

    enum SharedLowLevelLayerAttachState
    {
        SharedLowLevelLayerAttachState_Detached,
        SharedLowLevelLayerAttachState_Attached,
    };

    class SharedLowLevelLayer
    {
        NN_DISALLOW_COPY(SharedLowLevelLayer);
        NN_DISALLOW_MOVE(SharedLowLevelLayer);
    public:
        static const int FrameBufferCountMax = SharedFrameBufferCountMaxPerSharedBuffer;
        static const uintptr_t DisplayDequeueableLocalTag = 0;
        static const uintptr_t FrameBufferReleasedLocalTagMin = 1;
        static const uintptr_t FrameBufferReleasedLocalTagMax = FrameBufferReleasedLocalTagMin + FrameBufferCountMax - 1;
        static const int MultiWaitTagCount = FrameBufferReleasedLocalTagMax + 1;

    private:
        typedef util::TQueue<int, FrameBufferCountMax, false> DisplayIndexQueue;
        typedef util::TIntegerContainer<int, FrameBufferCountMax, -1, util::ContainerThreadSafetyOption_SingleThread> WaitingIndexContainer;
        typedef util::TIntegerContainer<int, FrameBufferCountMax, -1, util::ContainerThreadSafetyOption_SingleThread> IdleIndexContainer;

        struct DisplayReleaseFence
        {
            native::SyncpointEntry entry;
            nn::os::MultiWaitHolderType holder;
            bool isWaiting;
        };

    public:
        SharedLowLevelLayer() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;
        bool IsBound() const NN_NOEXCEPT;
        bool IsConnected() const NN_NOEXCEPT;
        bool IsAttached() const NN_NOEXCEPT;

        SharedLowLevelLayerState GetState() const NN_NOEXCEPT;
        SharedLowLevelLayerAttachState GetAttachState() const NN_NOEXCEPT;

        ResourceId GetLayerResourceId() const NN_NOEXCEPT;

        // @pre IsBound
        nn::sf::NativeHandle GetSynchronizedEventHandle() const NN_NOEXCEPT;

        // @pre IsConnected
        int GetFrameBufferCount() const NN_NOEXCEPT;

        // @pre IsConnected
        SharedFrameBuffer* GetFrameBuffer(int index) NN_NOEXCEPT;

        void SetupMultiWaitAdaptor(uintptr_t tagBase) NN_NOEXCEPT;
        void DispatchMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

        // SharedLowLevelLayerManager から呼ばれる
        // @pre State == Free
        // @post State == Bound
        // @post AttachState == Detached
        nn::Result InitializeForManagedLayer(
            client::ClientObject* pClientObject,
            nn::vi::LayerId layerId,
            const char* displayName,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

        // SharedLowLevelLayerManager から呼ばれる
        // @pre State == Free
        // @post State == Bound
        // @post AttachState == Detached
        nn::Result InitializeForIndirectLayer(
            client::ClientObject* pClientObject,
            nn::vi::IndirectProducerHandleType handle,
            nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

        // SharedLowLevelLayerManager から呼ばれる
        // @pre State == Bound
        // @pre !IsAttached
        // @post State == Free
        void Finalize() NN_NOEXCEPT;

        // @pre State == Bound
        // @post State == Connected
        nn::Result Connect(SharedBuffer* pSharedBuffer, SharedFrameBuffer* pFrameBufferList, int frameBufferCount) NN_NOEXCEPT;

        // @pre State == Connected
        // @pre !IsAttached
        // @post State == Bound
        void Disconnect() NN_NOEXCEPT;

        // @pre State == Connected
        // @pre AttachState == Detached
        // @post AttachState == Attached
        nn::Result Attach(SharedClientLayer* pClientLayer) NN_NOEXCEPT;

        // @pre AttachState == Attached
        // @post AttachState == Detached
        void Detach() NN_NOEXCEPT;

        // @pre IsConnected
        bool IsFrameBufferIdle(int frameBufferIndex) const NN_NOEXCEPT;

        // @pre IsConnected
        SharedBuffer* GetConnectedSharedBuffer() NN_NOEXCEPT;

        // @pre IsAttached
        detail::SharedClientLayer* GetAttachedClientLayer() NN_NOEXCEPT;


        // 指定したインデックスのバッファを低レベルレイヤに送る。
        // AcquireFence は先にバッファに設定しておくこと。
        // @pre IsConnected
        nn::Result Enqueue(int frameBufferIndex) NN_NOEXCEPT;

        // レイヤが同期状態かを検査して同期状態であれば表示中のバッファのインデックスを取得する。
        // 表示中のレイヤがない場合は -1 が取得される。
        // @pre IsBound
        nn::Result CheckSynchronized(int* pOutDisplayedFrameBufferIndex) const NN_NOEXCEPT;

        void Dump() const NN_NOEXCEPT;

    private:
        // @pre IsConnected
        void Dequeue() NN_NOEXCEPT;

        // @pre IsConnected
        void MoveToIdle(int frameBufferIndex) NN_NOEXCEPT;

        // @pre IsBound
        bool IsSynchronized() const NN_NOEXCEPT;

        // @pre IsBound
        void UpdateSynchronizedEvent() NN_NOEXCEPT;

        // @pre IsConnected
        // 現在提出中のバッファに設定されている各 LayerStack の積集合を計算する。
        // *pOutValue に計算した値を代入する。提出中のバッファがない場合は 0 が代入される。
        // 提出中のバッファの枚数を返す。提出中のバッファがない場合は 0 を返す。
        int CalculateLayerStackForPresentedBuffers(nn::vi::LayerStackFlagType* pOutValue) const NN_NOEXCEPT;

    private:
        SharedLowLevelLayerState       m_State;
        SharedLowLevelLayerAttachState m_AttachState;

        struct MultiWaitAdaptor
        {
        public:
            MultiWaitAdaptor() NN_NOEXCEPT;
            uintptr_t GetLocalTag(uintptr_t globalTag) const NN_NOEXCEPT;
            void LinkMultiWait(nn::os::MultiWaitHolderType* pHolder, uintptr_t localTag) NN_NOEXCEPT;
            void UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        public:
            uintptr_t tagBase;
        } m_MultiWaitAdaptor;

        struct BindingStatus
        {
            client::ClientObject*       pOwnerClient;
            local::ILocalLayer*         pLayer;
            local::LocalManagedLayer    managedLayer;
            local::LocalIndirectLayer   indirectLayer;
            nn::os::SystemEventType     displayDequeueableEvent;
            nn::os::MultiWaitHolderType displayDequeueableHolder;

            nn::os::SystemEventType     synchronizedEvent;
        } m_BindingStatus;

        struct ConnectionStatus
        {
            SharedBuffer* pSharedBuffer;

            int                   frameBufferCount;
            SharedFrameBuffer*    pFrameBufferList[FrameBufferCountMax];

            DisplayReleaseFence         bufferReleasedFence[FrameBufferCountMax];
            nn::os::MultiWaitHolderType bufferReleasedHolder;

            DisplayIndexQueue     displayIndexQueue;
            WaitingIndexContainer waitingIndexContainer;
            IdleIndexContainer    idleIndexContainer;

            nn::vi::LayerStackFlagType currentLayerStacks;
        } m_ConnectionStatus;

        struct AttachmentStatus
        {
            SharedClientLayer* pClientLayer;
        } m_AttachmentStatus;

    };

}}}}
