﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedLowLevelLayer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include "../../visrv_Macro.h"
#include "../../visrv_Log.h"
#include "../../visrv_ServerManager.h"
#include "visrv_SharedFrameBuffer.h"
#include "visrv_SharedClientLayer.h"

#define NN_VISRV_LOG_FBSHARE_DEV(...) //NN_VISRV_LOG_FBSHARE("[dev]" __VA_ARGS__)

#define NN_VISRV_LOG_FBSHARE_DEV_TRACE(...)      //NN_VISRV_LOG_FBSHARE_DEV(__VA_ARGS__)
#define NN_VISRV_LOG_FBSHARE_DEV_LAYERSTACK(...) //NN_VISRV_LOG_FBSHARE_DEV(__VA_ARGS__)
#define NN_VISRV_LOG_FBSHARE_DEV_DUMP(...)       //NN_VISRV_LOG_FBSHARE_DEV(__VA_ARGS__)

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{

    SharedLowLevelLayer::MultiWaitAdaptor::MultiWaitAdaptor() NN_NOEXCEPT
        : tagBase(-1)
    {
    }

    uintptr_t SharedLowLevelLayer::MultiWaitAdaptor::GetLocalTag(uintptr_t globalTag) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        return globalTag - this->tagBase;
    }

    void SharedLowLevelLayer::MultiWaitAdaptor::LinkMultiWait(nn::os::MultiWaitHolderType* pHolder, uintptr_t localTag) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        NN_SDK_REQUIRES_RANGE(localTag, 0, static_cast<uintptr_t>(SharedLowLevelLayer::MultiWaitTagCount));
        pHolder->userData = localTag + tagBase;
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitIndex_SharedLowLevelLayerMin, MultiWaitIndex_SharedLowLevelLayerMax);
        g_ServerManager.GetDisplayServerManager()->AddUserWaitHolder(pHolder);
    }

    void SharedLowLevelLayer::MultiWaitAdaptor::UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        nn::os::UnlinkMultiWaitHolder(pHolder);
    }

    //---------------------------------------------

    SharedLowLevelLayer::SharedLowLevelLayer() NN_NOEXCEPT
        : m_State(SharedLowLevelLayerState_Free)
        , m_AttachState(SharedLowLevelLayerAttachState_Detached)
        , m_BindingStatus()
        , m_ConnectionStatus()
        , m_AttachmentStatus()
    {
    }

    bool SharedLowLevelLayer::IsInitialized() const NN_NOEXCEPT
    {
        return IsBound();
    }

    bool SharedLowLevelLayer::IsBound() const NN_NOEXCEPT
    {
        return m_State != SharedLowLevelLayerState_Free;
    }

    bool SharedLowLevelLayer::IsConnected() const NN_NOEXCEPT
    {
        return m_State == SharedLowLevelLayerState_Connected;
    }

    bool SharedLowLevelLayer::IsAttached() const NN_NOEXCEPT
    {
        return m_AttachState == SharedLowLevelLayerAttachState_Attached;
    }

    SharedLowLevelLayerState SharedLowLevelLayer::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    SharedLowLevelLayerAttachState SharedLowLevelLayer::GetAttachState() const NN_NOEXCEPT
    {
        return m_AttachState;
    }

    ResourceId SharedLowLevelLayer::GetLayerResourceId() const NN_NOEXCEPT
    {
        if(m_State == SharedLowLevelLayerState_Free)
        {
            return 0;
        }
        return m_BindingStatus.pLayer->GetLayerResourceId();
    }

    int SharedLowLevelLayer::GetFrameBufferCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        return m_ConnectionStatus.frameBufferCount;
    }

    SharedFrameBuffer* SharedLowLevelLayer::GetFrameBuffer(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_RANGE(index, 0, m_ConnectionStatus.frameBufferCount);
        return m_ConnectionStatus.pFrameBufferList[index];
    }

    bool SharedLowLevelLayer::IsFrameBufferIdle(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_RANGE(index, 0, m_ConnectionStatus.frameBufferCount);
        return m_ConnectionStatus.idleIndexContainer.Contains(index);
    }

    SharedBuffer* SharedLowLevelLayer::GetConnectedSharedBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        return m_ConnectionStatus.pSharedBuffer;
    }

    detail::SharedClientLayer* SharedLowLevelLayer::GetAttachedClientLayer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        return m_AttachmentStatus.pClientLayer;
    }

    //--------------------------------------------------------------------

    nn::Result SharedLowLevelLayer::InitializeForManagedLayer(
        client::ClientObject* pClientObject,
        nn::vi::LayerId layerId,
        const char* displayName,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Initializing SharedLowLevelLayer layer for ManagedLayer #%lld\n", layerId);
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, SharedLowLevelLayerState_Free);
        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(m_BindingStatus.managedLayer.Initialize(pClientObject, layerId, displayName, aruid));
        NN_VISRV_PROCESS_ROLLBACK(m_BindingStatus.managedLayer.Finalize());

        NN_RESULT_DO(m_BindingStatus.managedLayer.GetAcquirableEvent(&m_BindingStatus.displayDequeueableEvent));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_BindingStatus.displayDequeueableEvent));

        nn::os::InitializeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder, &m_BindingStatus.displayDequeueableEvent);
        NN_VISRV_PROCESS_ROLLBACK(nn::os::FinalizeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder));

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_BindingStatus.synchronizedEvent, nn::os::EventClearMode_ManualClear, true));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_BindingStatus.synchronizedEvent));
        nn::os::SignalSystemEvent(&m_BindingStatus.synchronizedEvent); // initially signaled

        m_BindingStatus.pOwnerClient = pClientObject;
        m_BindingStatus.pLayer = &m_BindingStatus.managedLayer;
        m_State = SharedLowLevelLayerState_Bound;
        m_AttachState = SharedLowLevelLayerAttachState_Detached;

        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedLowLevelLayer::InitializeForIndirectLayer(
        client::ClientObject* pClientObject,
        nn::vi::IndirectProducerHandleType handle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Initializing SharedLowLevelLayer layer for IndirectProducer #%lld\n", handle);
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, SharedLowLevelLayerState_Free);
        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(m_BindingStatus.indirectLayer.Initialize(pClientObject, handle, aruid));
        NN_VISRV_PROCESS_ROLLBACK(m_BindingStatus.indirectLayer.Finalize());

        NN_RESULT_DO(m_BindingStatus.indirectLayer.GetAcquirableEvent(&m_BindingStatus.displayDequeueableEvent));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_BindingStatus.displayDequeueableEvent));

        nn::os::InitializeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder, &m_BindingStatus.displayDequeueableEvent);
        NN_VISRV_PROCESS_ROLLBACK(nn::os::FinalizeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder));

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_BindingStatus.synchronizedEvent, nn::os::EventClearMode_ManualClear, true));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_BindingStatus.synchronizedEvent));
        nn::os::SignalSystemEvent(&m_BindingStatus.synchronizedEvent); // initially signaled

        m_BindingStatus.pOwnerClient = pClientObject;
        m_BindingStatus.pLayer = &m_BindingStatus.indirectLayer;
        m_State = SharedLowLevelLayerState_Bound;
        m_AttachState = SharedLowLevelLayerAttachState_Detached;

        NN_VISRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    void SharedLowLevelLayer::Finalize() NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Finalizing SharedLowLevelLayer layer #%lld\n", GetLayerResourceId());
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, SharedLowLevelLayerState_Bound);
        NN_SDK_REQUIRES(!IsAttached());

        nn::os::SignalSystemEvent(&m_BindingStatus.synchronizedEvent); // ensure all waiter unblocked
        nn::os::DestroySystemEvent(&m_BindingStatus.synchronizedEvent);

        if(m_BindingStatus.pLayer == &m_BindingStatus.managedLayer)
        {
            nn::os::FinalizeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder);
            nn::os::DestroySystemEvent(&m_BindingStatus.displayDequeueableEvent);
            m_BindingStatus.managedLayer.Finalize();
        }
        else if(m_BindingStatus.pLayer == &m_BindingStatus.indirectLayer)
        {
            nn::os::FinalizeMultiWaitHolder(&m_BindingStatus.displayDequeueableHolder);
            nn::os::DestroySystemEvent(&m_BindingStatus.displayDequeueableEvent);
            m_BindingStatus.indirectLayer.Finalize();
        }
        else
        {
            NN_ABORT("SharedLowLevelLayer::m_pLayer is invalid value");
        }

        m_BindingStatus.pOwnerClient = nullptr;
        m_BindingStatus.pLayer = nullptr;
        m_State  = SharedLowLevelLayerState_Free;
    }

    nn::sf::NativeHandle SharedLowLevelLayer::GetSynchronizedEventHandle() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsBound());
        auto h = nn::os::GetReadableHandleOfSystemEvent(&m_BindingStatus.synchronizedEvent);
        return nn::sf::NativeHandle(h, false);
    }

    nn::Result SharedLowLevelLayer::CheckSynchronized(int* pOutDisplayedFrameBufferIndex) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());

        NN_RESULT_THROW_UNLESS(IsSynchronized(), nn::vi::ResultNotReady());

        int index = -1;
        if(!m_ConnectionStatus.displayIndexQueue.IsEmpty())
        {
            auto pHead = m_ConnectionStatus.displayIndexQueue.GetHead();
            NN_SDK_ASSERT_NOT_NULL(pHead);
            index = *pHead;
        }
        else if(m_ConnectionStatus.waitingIndexContainer.GetCount() > 0)
        {
            index = m_ConnectionStatus.waitingIndexContainer.GetHead();
        }

        *pOutDisplayedFrameBufferIndex = index;
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------

    void SharedLowLevelLayer::SetupMultiWaitAdaptor(uintptr_t tagBase) NN_NOEXCEPT
    {
        m_MultiWaitAdaptor.tagBase = tagBase;
    }

    void SharedLowLevelLayer::DispatchMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        auto localTag = m_MultiWaitAdaptor.GetLocalTag(pHolder->userData);
        NN_SDK_REQUIRES_RANGE(localTag, 0, static_cast<uintptr_t>(MultiWaitTagCount));

        if(localTag == DisplayDequeueableLocalTag)
        {
            Dequeue();
            m_MultiWaitAdaptor.LinkMultiWait(pHolder, DisplayDequeueableLocalTag); // DisplayDequeueable は常時リンクしておく
        }
        else if(localTag >= FrameBufferReleasedLocalTagMin && localTag <= FrameBufferReleasedLocalTagMax)
        {
            int frameBufferIndex = static_cast<int>(localTag - FrameBufferReleasedLocalTagMin);
            NN_ABORT_UNLESS_RANGE(frameBufferIndex, 0, m_ConnectionStatus.frameBufferCount);
            MoveToIdle(frameBufferIndex);
        }
        else
        {
            NN_VISRV_LOG_FBSHARE_ERR("[l]unknown local tag %lld\n", localTag);
        }
    }

    nn::Result SharedLowLevelLayer::Connect(SharedBuffer* pBuffer, SharedFrameBuffer* pFrameBufferList, int frameBufferCount) NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Connecting SharedLowLevelLayer layer #%d\n", GetLayerResourceId());
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, SharedLowLevelLayerState_Bound);
        NN_RESULT_THROW_UNLESS(pFrameBufferList != nullptr, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(frameBufferCount >= 0 && frameBufferCount <= FrameBufferCountMax, nn::vi::ResultInvalidRange());

        NN_VISRV_PROCESS_START();

        // すべてのレイヤスタックから抜く
        NN_VISRV_LOG_FBSHARE_DEV_LAYERSTACK("connecting: remove from all layerstack\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_BindingStatus.pLayer->SetLayerStackFlags(0));

        // 接続
        NN_RESULT_DO(m_BindingStatus.pLayer->Connect());
        NN_VISRV_PROCESS_ROLLBACK(m_BindingStatus.pLayer->Disconnect());
        m_MultiWaitAdaptor.LinkMultiWait(&m_BindingStatus.displayDequeueableHolder, DisplayDequeueableLocalTag);
        NN_VISRV_PROCESS_ROLLBACK(m_MultiWaitAdaptor.UnlinkMultiWait(&m_BindingStatus.displayDequeueableHolder));

        NN_RESULT_DO(m_BindingStatus.pLayer->SetBufferCount(frameBufferCount));
        int setCount = 0;
        for(int i = 0; i < frameBufferCount; i++)
        {
            auto& fb = pFrameBufferList[i];
            auto pBuffer = fb.GetGraphicBuffer();
            NN_RESULT_DO(m_BindingStatus.pLayer->SetBuffer(i, pBuffer));
            setCount++;
            // 即座に Dequeue
            int outIndex = -1;
            android::sp<android::Fence> outFence;
            NN_RESULT_DO(m_BindingStatus.pLayer->Acquire(&outIndex, &outFence, fb.GetWidth(), fb.GetHeight()));
            NN_SDK_ASSERT_EQUAL(outIndex, i);
            NN_SDK_ASSERT(!outFence->isValid());
            // RequestBuffer して使えるようにする
            NN_RESULT_DO(m_BindingStatus.pLayer->RequestBuffer(outIndex));
        }
        // 挿した分のバッファを外す。
        NN_VISRV_PROCESS_ROLLBACK({
            for(int i = 0; i < setCount; i++)
            {
                (void)m_BindingStatus.pLayer->SetBuffer(i, nullptr);
            }
        });

        // 全バッファ Dequeue したのでイベントはクリアされているはず
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(33));
        NN_SDK_ASSERT(!nn::os::TryWaitSystemEvent(&m_BindingStatus.displayDequeueableEvent));

        // バッファのリストをコピー
        m_ConnectionStatus.frameBufferCount = frameBufferCount;
        std::memset(m_ConnectionStatus.pFrameBufferList, 0, sizeof(m_ConnectionStatus.pFrameBufferList));
        for(int i = 0; i < frameBufferCount; i++)
        {
            m_ConnectionStatus.pFrameBufferList[i] = &pFrameBufferList[i];
        }

        // フェンスを作成
        for(int i = 0; i < frameBufferCount; i++)
        {
            auto& e = m_ConnectionStatus.bufferReleasedFence[i];
            e.entry.InitializeWaitHolder(&e.holder);
            e.isWaiting = false;
        }

        // 空きインデックスに登録
        m_ConnectionStatus.displayIndexQueue.Initialize();
        m_ConnectionStatus.waitingIndexContainer.Initialize();
        m_ConnectionStatus.idleIndexContainer.Initialize();
        for(int i = 0; i < frameBufferCount; i++)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.idleIndexContainer.Register(i));
        }

        // イベントの更新
        UpdateSynchronizedEvent();

        m_ConnectionStatus.pSharedBuffer = pBuffer;
        m_ConnectionStatus.currentLayerStacks = 0;
        NN_VISRV_PROCESS_SUCCESS();
        m_State = SharedLowLevelLayerState_Connected;
        NN_RESULT_SUCCESS;
    }

    void SharedLowLevelLayer::Disconnect() NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Disconnecting SharedLowLevelLayer layer #%d\n", GetLayerResourceId());
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, SharedLowLevelLayerState_Connected);
        NN_SDK_REQUIRES(!IsAttached());

        // すべてのレイヤスタックから抜く
        NN_VISRV_LOG_FBSHARE_DEV_LAYERSTACK("disconnecting: remove from all layerstack\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_BindingStatus.pLayer->SetLayerStackFlags(0));

        // スロットをクリア
        //   クリアしないと Preallocated フラグが残ったままになる
        for(int i = 0; i < m_ConnectionStatus.frameBufferCount; i++)
        {
            (void)m_BindingStatus.pLayer->SetBuffer(i, nullptr);
        }

        // 切断
        // NOTE: 切断時に DequeueableEvent はシグナルされる
        m_BindingStatus.pLayer->Disconnect();
        m_MultiWaitAdaptor.UnlinkMultiWait(&m_BindingStatus.displayDequeueableHolder);

        // キューをクリア
        m_ConnectionStatus.displayIndexQueue.Finalize();
        m_ConnectionStatus.waitingIndexContainer.Finalize();
        m_ConnectionStatus.idleIndexContainer.Finalize();

        // フェンスを破棄
        for(int i = 0; i < m_ConnectionStatus.frameBufferCount; i++)
        {
            auto& e = m_ConnectionStatus.bufferReleasedFence[i];
            if(e.isWaiting)
            {
                m_MultiWaitAdaptor.UnlinkMultiWait(&e.holder);
                e.isWaiting = false;
            }
            e.entry.FinalizeWaitHolder(&e.holder);
            e.entry.Reset();
        }

        // バッファの所有権を放棄
        auto myLayerId = GetLayerResourceId();
        for(int i = 0; i < m_ConnectionStatus.frameBufferCount; i++)
        {
            auto& p = m_ConnectionStatus.pFrameBufferList[i];
            if(p->GetPresentedLowLevelLayerId() == myLayerId)
            {
                p->SetPresentedLowLevelLayerId(0);
            }
        }

        // バッファのリストを破棄
        m_ConnectionStatus.frameBufferCount = 0;
        std::memset(m_ConnectionStatus.pFrameBufferList, 0, sizeof(m_ConnectionStatus.pFrameBufferList));

        m_ConnectionStatus.pSharedBuffer = nullptr;
        m_ConnectionStatus.currentLayerStacks = 0;

        m_State = SharedLowLevelLayerState_Bound;
    }

    nn::Result SharedLowLevelLayer::Attach(SharedClientLayer* pClientLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedLowLevelLayerAttachState_Detached);

        m_AttachmentStatus.pClientLayer = pClientLayer;

        m_AttachState = SharedLowLevelLayerAttachState_Attached;

        // イベントの更新
        UpdateSynchronizedEvent();
        NN_RESULT_SUCCESS;
    }

    void SharedLowLevelLayer::Detach() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedLowLevelLayerAttachState_Attached);
        NN_SDK_ASSERT(IsConnected());

        m_AttachmentStatus.pClientLayer = nullptr;

        m_AttachState = SharedLowLevelLayerAttachState_Detached;

        // イベントの更新
        UpdateSynchronizedEvent();
    }

    nn::Result SharedLowLevelLayer::Enqueue(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_RANGE(frameBufferIndex, 0, m_ConnectionStatus.frameBufferCount);

        NN_VISRV_PROCESS_START();
        auto pBuffer = m_ConnectionStatus.pFrameBufferList[frameBufferIndex];
        auto param = pBuffer->GetContentParameter();

        // 送り先に空きがあるかチェック
        NN_RESULT_THROW_UNLESS(!m_ConnectionStatus.displayIndexQueue.IsFull(), nn::vi::ResultNotReady());

        // ディスプレイに表示中のバッファはエンキューできない
        auto layerId = pBuffer->GetPresentedLowLevelLayerId();
        NN_RESULT_THROW_UNLESS(layerId == 0, nn::vi::ResultNotReady());

        // バッファを出す前に新しいレイヤスタックを設定
        nn::vi::LayerStackFlagType newStacks = 0;
        if(CalculateLayerStackForPresentedBuffers(&newStacks) > 0)
        {
            newStacks &= param.layerStacks;
        }
        else
        {
            newStacks = param.layerStacks;
        }
        if(newStacks != m_ConnectionStatus.currentLayerStacks)
        {
            NN_VISRV_LOG_FBSHARE_DEV_LAYERSTACK("enqueueing: layerstack changing %X -> %X\n", m_ConnectionStatus.currentLayerStacks, newStacks);
            NN_RESULT_DO(m_BindingStatus.pLayer->SetLayerStackFlags(newStacks));
        }
        NN_VISRV_PROCESS_ROLLBACK(
            if(newStacks != m_ConnectionStatus.currentLayerStacks)
            {
                (void)m_BindingStatus.pLayer->SetLayerStackFlags(m_ConnectionStatus.currentLayerStacks);
            }
        );

        // FrameBuffer を Idle 状態から外す
        NN_RESULT_DO(m_ConnectionStatus.idleIndexContainer.Unregister(frameBufferIndex));
        NN_VISRV_PROCESS_ROLLBACK(NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.idleIndexContainer.Register(frameBufferIndex)));

        // Display に送る
        NN_RESULT_DO(m_BindingStatus.pLayer->Present(frameBufferIndex, pBuffer->GetAcquireFence(), param.cropRegion, param.scalingMode, param.transform, param.presentInterval));

        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Enqueued: buffer %d to layer #%d\n", frameBufferIndex, GetLayerResourceId());
        NN_VISRV_PROCESS_SUCCESS();

        // 表示中キューに追加
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.displayIndexQueue.TryEnqueue(frameBufferIndex));
        // バッファの表示中レイヤを設定
        pBuffer->SetPresentedLowLevelLayerId(GetLayerResourceId());
        // 設定したレイヤスタックを更新
        m_ConnectionStatus.currentLayerStacks = newStacks;

        // イベントの更新
        UpdateSynchronizedEvent();
        NN_RESULT_SUCCESS;
    }

    void SharedLowLevelLayer::Dequeue() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsConnected());

        int deqIndex = -1;
        int index = -1;
        android::sp<android::Fence> fence;

        // 次に Dequeue されるべきインデックスを取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.displayIndexQueue.TryDequeue(&deqIndex));
        auto pBuf = m_ConnectionStatus.pFrameBufferList[deqIndex];

        // Dequeue
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_BindingStatus.pLayer->Acquire(&index, &fence, pBuf->GetWidth(), pBuf->GetHeight()));
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Dequeued: buffer %d from layer #%d\n", index, GetLayerResourceId());
        NN_ABORT_UNLESS_EQUAL(index, deqIndex);
        NN_ABORT_UNLESS_RANGE(index, 0, m_ConnectionStatus.frameBufferCount);

        // DisplayRelease 待ちリストに追加
        m_ConnectionStatus.waitingIndexContainer.Register(index);

        // DisplayRelease イベントの待機
        m_ConnectionStatus.pFrameBufferList[index]->SetReleaseFence(fence);
        {
            auto& e = m_ConnectionStatus.bufferReleasedFence[index];
            NN_SDK_ASSERT(!e.isWaiting);
            // 待ち開始
            e.entry.Reset(fence);
            native::g_SyncpointWaiter.Enqueue(&e.entry);
            m_MultiWaitAdaptor.LinkMultiWait(&e.holder, FrameBufferReleasedLocalTagMin + index);
            e.isWaiting = true;
        }

        // NOTE:
        //   Dequeue しただけではディスプレイから除外されていないので
        //   LayerStack の変更は行わない。

        if(IsAttached())
        {
            m_AttachmentStatus.pClientLayer->NotifyBufferDequeued(index);
        }
    }

    void SharedLowLevelLayer::MoveToIdle(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES_RANGE(frameBufferIndex, 0, m_ConnectionStatus.frameBufferCount);
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("Released: buffer %d from layer #%d\n", frameBufferIndex, GetLayerResourceId());

        NN_SDK_ASSERT(
            m_ConnectionStatus.pFrameBufferList[frameBufferIndex]->GetPresentedLowLevelLayerId() == 0 ||
            m_ConnectionStatus.pFrameBufferList[frameBufferIndex]->GetPresentedLowLevelLayerId() == GetLayerResourceId()
        );

        m_ConnectionStatus.bufferReleasedFence[frameBufferIndex].entry.Reset();
        m_ConnectionStatus.bufferReleasedFence[frameBufferIndex].isWaiting = false;
        m_ConnectionStatus.pFrameBufferList[frameBufferIndex]->SetPresentedLowLevelLayerId(0);
        m_ConnectionStatus.pFrameBufferList[frameBufferIndex]->SetReleaseFence(android::Fence::NO_FENCE);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.waitingIndexContainer.Unregister(frameBufferIndex));
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ConnectionStatus.idleIndexContainer.Register(frameBufferIndex));

        // レイヤスタックの更新
        nn::vi::LayerStackFlagType newStacks = 0;
        CalculateLayerStackForPresentedBuffers(&newStacks);
        if(newStacks != m_ConnectionStatus.currentLayerStacks)
        {
            NN_VISRV_LOG_FBSHARE_DEV_LAYERSTACK("releasing: layerstack changing %X -> %X\n", m_ConnectionStatus.currentLayerStacks, newStacks);
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_BindingStatus.pLayer->SetLayerStackFlags(newStacks));
            m_ConnectionStatus.currentLayerStacks = newStacks;
        }

        if(IsAttached())
        {
            m_AttachmentStatus.pClientLayer->NotifyBufferIdle();
        }

        // イベントの更新
        UpdateSynchronizedEvent();
    }

    bool SharedLowLevelLayer::IsSynchronized() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        NN_VISRV_LOG_FBSHARE_DEV_TRACE("signal LowLevelLayerSynchronizedEvent for attached layer #%lld\n", GetLayerResourceId());
        Dump();
        return
            !IsAttached() &&
            (m_ConnectionStatus.displayIndexQueue.GetCount() + m_ConnectionStatus.waitingIndexContainer.GetCount() <= 1);
    }

    void SharedLowLevelLayer::UpdateSynchronizedEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        if(IsAttached())
        {
            // レイヤのアタッチ／デタッチの責任は manager にあるが、
            // アタッチ中のレイヤの SynchronizedEvent を manager が無限待ちした場合、他の誰もレイヤのデタッチができないため
            // IsSynchronized == true が成立することがなくデッドロックする。
            // デッドロックを回避するためにアタッチ中のレイヤの SynchronizedEvent は常にシグナル状態にしておく。
            // manager はこのイベントを待った後で CheckSynchronized() を呼ぶはずで、その際に ResultNotReady が返される。
            NN_VISRV_LOG_FBSHARE_DEV_TRACE("signal LowLevelLayerSynchronizedEvent for attached layer #%lld\n", GetLayerResourceId());
            nn::os::SignalSystemEvent(&m_BindingStatus.synchronizedEvent);
        }
        else if(IsSynchronized())
        {
            NN_VISRV_LOG_FBSHARE_DEV_TRACE("signal LowLevelLayerSynchronizedEvent #%lld\n", GetLayerResourceId());
            nn::os::SignalSystemEvent(&m_BindingStatus.synchronizedEvent);
        }
        else
        {
            NN_VISRV_LOG_FBSHARE_DEV_TRACE("clear LowLevelLayerSynchronizedEvent #%lld\n", GetLayerResourceId());
            nn::os::ClearSystemEvent(&m_BindingStatus.synchronizedEvent);
        }
    }

    int SharedLowLevelLayer::CalculateLayerStackForPresentedBuffers(nn::vi::LayerStackFlagType* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());

        int count = 0;
        nn::vi::LayerStackFlagType stacks = nn::vi::ValidLayerStackFlags;

        m_ConnectionStatus.displayIndexQueue.Foreach([&](int index){
            auto param = m_ConnectionStatus.pFrameBufferList[index]->GetContentParameter();
            stacks &= param.layerStacks;
            count++;
        });

        m_ConnectionStatus.waitingIndexContainer.Foreach([&](int index){
            auto param = m_ConnectionStatus.pFrameBufferList[index]->GetContentParameter();
            stacks &= param.layerStacks;
            count++;
        });

        if(count == 0)
        {
            stacks = 0;
        }

        *pOutValue = stacks;
        return count;
    }

    void SharedLowLevelLayer::Dump() const NN_NOEXCEPT
    {
        NN_VISRV_LOG_FBSHARE_DEV_DUMP("SharedLowLevelLayer #%lld\n", GetLayerResourceId());
        switch(m_State)
        {
        case SharedLowLevelLayerState_Free:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  State: Free\n");
            break;
        case SharedLowLevelLayerState_Bound:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  State: Bound\n");
            break;
        case SharedLowLevelLayerState_Connected:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  State: Connected\n");
            break;
        default:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  State: unknown(%d)\n", m_State);
        }

        switch(m_AttachState)
        {
        case SharedLowLevelLayerAttachState_Detached:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  AttachState: Detached\n");
            break;
        case SharedLowLevelLayerAttachState_Attached:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  AttachState: Attached\n");
            break;
        default:
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  AttachState: unknown(%d)\n", m_AttachState);
        }

        auto dumpLayer = [&](int index){
            auto pLayer = m_ConnectionStatus.pFrameBufferList[index];
            auto param = pLayer->GetContentParameter();
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("      %02d: stacks=%X\n", index, param.layerStacks);
            NN_UNUSED(param);
        };

        NN_VISRV_LOG_FBSHARE_DEV_DUMP("  IsInitialized(): %s\n", IsInitialized() ? "true" : "false");
        NN_VISRV_LOG_FBSHARE_DEV_DUMP("  IsBound()      : %s\n", IsBound() ? "true" : "false");
        NN_VISRV_LOG_FBSHARE_DEV_DUMP("  IsConnected()  : %s\n", IsConnected() ? "true" : "false");
        NN_VISRV_LOG_FBSHARE_DEV_DUMP("  IsAttached()   : %s\n", IsAttached() ? "true" : "false");
        if(IsConnected())
        {
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  FrameBuffers\n");
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("    count: %d\n", m_ConnectionStatus.frameBufferCount);
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("    display: (%d)\n", m_ConnectionStatus.displayIndexQueue.GetCount());
            m_ConnectionStatus.displayIndexQueue.Foreach(dumpLayer);
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("    waiting: (%d)\n", m_ConnectionStatus.waitingIndexContainer.GetCount());
            m_ConnectionStatus.waitingIndexContainer.Foreach(dumpLayer);
            NN_VISRV_LOG_FBSHARE_DEV_DUMP("    idle   : (%d)\n", m_ConnectionStatus.idleIndexContainer.GetCount());
            m_ConnectionStatus.idleIndexContainer.Foreach(dumpLayer);

            NN_VISRV_LOG_FBSHARE_DEV_DUMP("  LayerStack = %X\n", m_ConnectionStatus.currentLayerStacks);
        }
    }

}}}}
