﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/vi/vi_Result.h>
#include "vi_Display.h"
#include "detail/vi_Common.h"

nn::vi::Display::Display(const detail::PlatformDisplayInfo& info, Allocator* pAllocator, const detail::IModeFilter* pFilter) NN_NOEXCEPT
    : m_Info(info)
    , m_pAllocator(pAllocator)
    , m_pFilter(pFilter)
    , m_Lock(false)
    , m_OpenCount(0)
{
    NN_SDK_ASSERT_NOT_NULL(pAllocator);
}

nn::vi::Display::~Display() NN_NOEXCEPT
{
    if( IsOpen() )
    {
        ForceClose();
    }
}

bool nn::vi::Display::IsOpen() const NN_NOEXCEPT
{
    return m_OpenCount != 0;
}

nn::Result nn::vi::Display::Open() NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    if( !IsOpen() )
    {
        result = Initialize();
    }

    if( result.IsSuccess() )
    {
        ++m_OpenCount;
    }

    return result;
}

void nn::vi::Display::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_OpenCount > 0);

    --m_OpenCount;

    if( !IsOpen() )
    {
        ForceClose();
    }
}

void nn::vi::Display::ForceClose() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    LayerList::iterator i = m_Layers.begin();

    while( i != m_Layers.end() )
    {
        LayerList::iterator temp = i++;
        temp->~Layer();
        m_pAllocator->Free(&*temp);
    }
}

nn::Result nn::vi::Display::CreateLayer(Layer** pOutLayer, int width, int height, PixelFormat format, LayerSettings settings) NN_NOEXCEPT
{
    *pOutLayer = nullptr;

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if( m_Info.GetPolicy().HasLayerLimit() &&
        m_Layers.size() >= m_Info.GetPolicy().GetLayerCountMax() )
    {
        return nn::vi::ResultOperationFailed();
    }

    if( !m_Info.ValidateLayerDimensions(width, height) )
    {
        return nn::vi::ResultInvalidDimensions();
    }

    if( !m_Info.ValidatePixelFormat(format) )
    {
        return nn::vi::ResultInvalidPixelFormat();
    }

    void* block = reinterpret_cast<Layer*>(m_pAllocator->Allocate());
    NN_SDK_ASSERT_NOT_NULL(block);

    Layer* pLayer = new (block) Layer(this);

    nn::Result result = pLayer->Initialize(width, height, format, settings);
    if( result.IsSuccess() )
    {
        m_Layers.push_back(*pLayer);
        *pOutLayer = pLayer;
    }
    else
    {
        pLayer->~Layer();
        m_pAllocator->Free(pLayer);
    }

    return result;
}

const nn::vi::detail::PlatformDisplayInfo& nn::vi::Display::GetInfo() const NN_NOEXCEPT
{
    return m_Info;
}

nn::Result nn::vi::Display::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    for( LayerList::iterator i = m_Layers.begin(); i != m_Layers.end(); ++i )
    {
        // TODO: investigate performance of opening transaction outside this loop
        result = i->SetVisibility(isEnabled);

        if( result.IsFailure() )
        {
            break;
        }
    }

    return result;
}

void nn::vi::Display::DestroyLayer(Layer* pLayer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    m_Layers.erase(m_Layers.iterator_to(*pLayer));
    pLayer->~Layer();
    m_pAllocator->Free(pLayer);
}

int nn::vi::Display::ListModes(DisplayModeInfo* pOutModes, int modeCountMax) const NN_NOEXCEPT
{
    return ListModes(pOutModes, modeCountMax, m_pFilter);
}

nn::Result nn::vi::Display::SetMode(const DisplayModeInfo* pMode) NN_NOEXCEPT
{
    return SetMode(pMode, m_pFilter);
}

nn::os::Mutex& nn::vi::Display::GetLayerListLock() const NN_NOEXCEPT
{
    return m_Lock;
}

const nn::vi::Display::LayerList& nn::vi::Display::GetLayerList() const NN_NOEXCEPT
{
    return m_Layers;
}
