﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Result.h>
#include "vi_EdidDisplay.h"
#include "vi_Edid.h"
#include "vi_IModeFilter.h"

nn::vi::detail::EdidDisplay::EdidDisplay() NN_NOEXCEPT
    : m_Handle{ nullptr, 0 }
{
    memset(&m_Data, 0, sizeof(m_Data));
}

nn::Result nn::vi::detail::EdidDisplay::Open() NN_NOEXCEPT
{
    // make sure there's no padding, otherwise this trick won't work
    NN_STATIC_ASSERT(sizeof(nn::settings::system::Edid) == 256);

    nn::settings::system::GetEdid(&m_Data);

    if( nn::edid::OpenEdid(&m_Handle, &m_Data, sizeof(nn::settings::system::Edid)) == nn::edid::Error_None )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

void nn::vi::detail::EdidDisplay::Close() NN_NOEXCEPT
{
    nn::edid::CloseEdid(&m_Handle);
}

nn::Result nn::vi::detail::EdidDisplay::GetHotplugState(HotplugState* pOutState) const NN_NOEXCEPT
{
    // this isn't a real display
    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::GetMode(DisplayModeInfo* pOutMode) const NN_NOEXCEPT
{
    // this isn't a real display
    return nn::vi::ResultOperationFailed();
}

int nn::vi::detail::EdidDisplay::ListModes(DisplayModeInfo* pOutModes, int modeCountMax, const IModeFilter* pFilter) const NN_NOEXCEPT
{
    DisplayModeVisitorData data = { 0, pFilter, pOutModes, modeCountMax };

    nn::edid::VisitDisplayModes(&m_Handle, DisplayModeVisitor, &data);

    return data.writeCount;
}

nn::Result nn::vi::detail::EdidDisplay::SetMode(const DisplayModeInfo* pMode, const IModeFilter* pFilter) NN_NOEXCEPT
{
    return nn::ResultSuccess();
}

bool nn::vi::detail::EdidDisplay::DisplayModeVisitor(const nn::edid::DisplayModeInfo* pMode,
                                                     const nn::edid::DisplayTimingInfo* pTimingInfo,
                                                     const nn::edid::ImageSizeInfo* pSizeInfo,
                                                     void* pUserData) NN_NOEXCEPT
{
    DisplayModeVisitorData* pData = static_cast<DisplayModeVisitorData*>(pUserData);

    if( pData->writeCount >= pData->modeCountMax )
    {
        return false;
    }

    DisplayModeInfo mode;
    if( detail::ConvertDisplayMode(&mode, pMode) && pData->pFilter->IsValid(mode) )
    {
        bool isDuplicate = false;

        for( int i = 0; i < pData->writeCount; ++i )
        {
            if( pData->pOutModes[i].width == mode.width &&
                pData->pOutModes[i].height == mode.height &&
                pData->pOutModes[i].refreshRate == mode.refreshRate &&
                pData->pOutModes[i].mode == mode.mode )
            {
                isDuplicate = true;
                break;
            }
        }

        if( !isDuplicate )
        {
            pData->pOutModes[pData->writeCount] = mode;
            ++pData->writeCount;
        }
    }

    return true;
}

int nn::vi::detail::EdidDisplay::ListRgbRanges(RgbRange* pOutRanges, int rgbCountMax) const NN_NOEXCEPT
{
    nn::edid::DisplayInfo info;
    nn::edid::GetDisplayInfo(&info, &m_Handle);

    return WriteCeFormatRgbRange(pOutRanges, rgbCountMax, info.isRgbQuantizationSelectable);
}

nn::Result nn::vi::detail::EdidDisplay::GetRgbRange(RgbRange* pOutRange) const NN_NOEXCEPT
{
    NN_UNUSED(pOutRange);

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::EdidDisplay::SetRgbRange(RgbRange range) NN_NOEXCEPT
{
    NN_UNUSED(range);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
{
    NN_UNUSED(underscan);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
{
    NN_UNUSED(pOutUnderscan);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::SetAlpha(float alpha) NN_NOEXCEPT
{
    NN_UNUSED(alpha);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::SetPowerState(PowerState state) NN_NOEXCEPT
{
    NN_UNUSED(state);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::SetLayerStack(LayerStack id) NN_NOEXCEPT
{
    NN_UNUSED(id);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::SetCmuLuma(float luma) NN_NOEXCEPT
{
    NN_UNUSED(luma);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
{
    NN_UNUSED(pOutLuma);

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::detail::EdidDisplay::GetHotplugEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT
{
    NN_UNUSED(pOutEvent);

    return nn::vi::ResultNotSupported();
}
