﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/vi/vi_Result.h>
#include "vi_DisplayManager.h"

nn::vi::detail::DisplayManager::DisplayManager(void* displaySegmentStartAddress, size_t displaySegmentSize,
                                               void* layerSegmentStartAddress, size_t layerSegmentSize,
                                               PlatformDisplayInfo* platformDisplays, int platformDisplayCount) NN_NOEXCEPT
    : m_LayerAllocator(layerSegmentStartAddress, layerSegmentSize, NN_ALIGNOF(Layer))
    , m_DisplayFactory(displaySegmentStartAddress, displaySegmentSize)
    , m_PlatformDisplays(platformDisplays)
    , m_PlatformDisplayCount(platformDisplayCount)
    , m_Lock(false)
{
}

nn::vi::detail::DisplayManager::~DisplayManager() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    DisplayList::iterator i = m_OpenDisplays.begin();

    while( i != m_OpenDisplays.end() )
    {
        DisplayList::iterator temp = i++;
        // not pretty, but need the pointer...
        m_DisplayFactory.Destroy(&*temp);
    }
}

nn::vi::Display* nn::vi::detail::DisplayManager::Open(const char* name) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    // check to see if the display is already open by the process
    for( DisplayList::iterator i = m_OpenDisplays.begin(); i != m_OpenDisplays.end(); ++i )
    {
        if( strncmp(name, i->GetInfo().GetName(), DisplayInfo::NameLengthMax) == 0 )
        {
            i->Open();
            return &*i;
        }
    }

    // check in named platform displays first
    for( int i = 0; i < m_PlatformDisplayCount; ++i )
    {
        if( strncmp(name, m_PlatformDisplays[i].GetName(), DisplayInfo::NameLengthMax) == 0 )
        {
            Display* pDisplay = m_DisplayFactory.Create(m_PlatformDisplays[i], &m_LayerAllocator);

            if( pDisplay != nullptr )
            {
                if( pDisplay->Open().IsFailure() )
                {
                    m_DisplayFactory.Destroy(pDisplay);
                    return nullptr;
                }

                m_OpenDisplays.push_back(*pDisplay);
            }

            return pDisplay;
        }
    }

    // TODO: enumerate other displays (useful for PC)

    return nullptr;
}

void nn::vi::detail::DisplayManager::Close(Display* pDisplay) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    pDisplay->Close();

    if( !pDisplay->IsOpen() )
    {
        m_OpenDisplays.erase(m_OpenDisplays.iterator_to(*pDisplay));
        m_DisplayFactory.Destroy(pDisplay);
    }
}

nn::os::Mutex& nn::vi::detail::DisplayManager::GetDisplayListLock() const NN_NOEXCEPT
{
    return m_Lock;
}

const nn::vi::detail::DisplayManager::DisplayList& nn::vi::detail::DisplayManager::GetDisplayList() const NN_NOEXCEPT
{
    return m_OpenDisplays;
}
