﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "vi_ObjectManager.h"

#include <algorithm>
#include <nn/util/util_StringUtil.h>
#include "vi_MemoryManagement.h"

namespace nn{ namespace vi{
    namespace {
        DisplayHolderList g_ValidDisplayHolderList;
        LayerHolderList g_ValidLayerHolderList;
    }

    nn::os::Mutex ObjectManager::m_DisplayLock(false);
    nn::os::Mutex ObjectManager::m_LayerLock(false);

//-----------
// Display
//-----------

    DisplayHolderList* ObjectManager::GetValidDisplayHolderList() NN_NOEXCEPT
    {
        return &g_ValidDisplayHolderList;
    }

    DisplayHolderList::iterator ObjectManager::CreateValidateDisplayHolder(DisplayId id, const char* name) NN_NOEXCEPT
    {
        DisplayHolder* pHolder = new(AllocateDisplayHolder()) DisplayHolder(id, name);
        NN_SDK_ASSERT_NOT_NULL(pHolder);
        g_ValidDisplayHolderList.push_back(*pHolder);
        return --g_ValidDisplayHolderList.end();
    }

    DisplayHolderList::iterator ObjectManager::InvalidateDestroyDisplayHolder(DisplayHolderList::iterator it) NN_NOEXCEPT
    {
        auto next = it;
        ++next;

        // invalidate all layers
        {
            auto itLayer = GetValidLayerHolderList()->begin();
            auto itLayerEnd = GetValidLayerHolderList()->end();
            while(itLayer != itLayerEnd)
            {
                if(itLayer->GetDisplayHolder() == &(*it))
                {
                    itLayer = InvalidateDestroyLayerHolder(itLayer);
                }
                else
                {
                    ++itLayer;
                }
            }
        }

        g_ValidDisplayHolderList.erase(it);
        it->~DisplayHolder();
        FreeDisplayHolder(&(*it));

        return next;
    }

    DisplayHolderList::iterator ObjectManager::FindValidDisplayHolder(Display* pDisplay) NN_NOEXCEPT
    {
        auto pHolder = reinterpret_cast<DisplayHolder*>(pDisplay);
        return std::find_if(
            g_ValidDisplayHolderList.begin(),
            g_ValidDisplayHolderList.end(),
            [pHolder](const DisplayHolder& v){ return &v == pHolder; }
        );
    }

    DisplayHolderList::const_iterator ObjectManager::FindValidDisplayHolder(const Display* pDisplay) NN_NOEXCEPT
    {
        auto pHolder = reinterpret_cast<const DisplayHolder*>(pDisplay);
        return std::find_if(
            g_ValidDisplayHolderList.begin(),
            g_ValidDisplayHolderList.end(),
            [pHolder](const DisplayHolder& v){ return &v == pHolder; }
        );
    }

    DisplayHolderList::iterator ObjectManager::FindValidDisplayHolderByName(const char* name) NN_NOEXCEPT
    {
        return std::find_if(
            g_ValidDisplayHolderList.begin(),
            g_ValidDisplayHolderList.end(),
            [name](const DisplayHolder& v){ return (nn::util::Strncmp(v.GetName(), name, sizeof(DisplayName)) == 0); }
        );
    }

    nn::os::Mutex& ObjectManager::GetDisplayLock() NN_NOEXCEPT
    {
        return m_DisplayLock;
    }

//-----------
// Layer
//-----------
    LayerHolderList* ObjectManager::GetValidLayerHolderList() NN_NOEXCEPT
    {
        return &g_ValidLayerHolderList;
    }

    LayerHolderList::iterator ObjectManager::CreateValidateLayerHolder(LayerId id, DisplayHolder* pDisplayHolder) NN_NOEXCEPT
    {
        LayerHolder* pHolder = new(AllocateLayerHolder()) LayerHolder(id, pDisplayHolder);
        NN_SDK_ASSERT_NOT_NULL(pHolder);

        g_ValidLayerHolderList.push_back(*pHolder);
        pDisplayHolder->IncrementLayerCount();

        return --g_ValidLayerHolderList.end();
    }

    LayerHolderList::iterator ObjectManager::InvalidateDestroyLayerHolder(LayerHolderList::iterator it) NN_NOEXCEPT
    {
        auto next = it;
        ++next;

        it->GetDisplayHolder()->DecrementLayerCount();
        g_ValidLayerHolderList.erase(it);

        it->~LayerHolder();
        FreeLayerHolder(&(*it));

        return next;
    }

    LayerHolderList::iterator ObjectManager::FindValidLayerHolder(Layer* pLayer) NN_NOEXCEPT
    {
        auto pHolder = reinterpret_cast<LayerHolder*>(pLayer);
        return std::find_if(
            g_ValidLayerHolderList.begin(),
            g_ValidLayerHolderList.end(),
            [pHolder](const LayerHolder& v){ return &v == pHolder; }
        );
    }

    LayerHolderList::const_iterator ObjectManager::FindValidLayerHolder(const Layer* pLayer) NN_NOEXCEPT
    {
        auto pHolder = reinterpret_cast<const LayerHolder*>(pLayer);
        return std::find_if(
            g_ValidLayerHolderList.begin(),
            g_ValidLayerHolderList.end(),
            [pHolder](const LayerHolder& v){ return &v == pHolder; }
        );
    }

    nn::os::Mutex& ObjectManager::GetLayerLock() NN_NOEXCEPT
    {
        return m_LayerLock;
    }

}}
