﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/manager/vi_ManagerFrameBuffer.h>
#include "vi_ManagerFrameBufferData.h"

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_Result.h>
#include "vi_ManagerServiceData.h"

namespace nn{ namespace vi{ namespace manager{

    namespace {
        const uint32_t BufferUsageBits = GRALLOC_USAGE_HW_COMPOSER | GRALLOC_USAGE_HW_VIDEO_ENCODER;

        uint32_t GetComposerImageFormat(PixelFormatType format) NN_NOEXCEPT
        {
            switch(format)
            {
            case PixelFormat_Rgba8888:
                {
                    return android::PIXEL_FORMAT_RGBA_8888;
                }
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }


    ManagerFrameBuffer::ManagerFrameBuffer() NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(m_Storage) >= sizeof(ManagerFrameBufferData));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_Storage) >= NN_ALIGNOF(ManagerFrameBufferData));
        std::memset(this, 0, sizeof(*this));
    }

    ManagerFrameBuffer::~ManagerFrameBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
    }

    nn::Result ManagerFrameBuffer::Initialize(DisplayManagerService* pService, int64_t width, int64_t height, nn::vi::PixelFormatType format) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pService);
        NN_SDK_REQUIRES_EQUAL(format, nn::vi::PixelFormat_Rgba8888);

        auto pServiceData = pService->GetData();
        auto pData = new(&m_Storage[0]) ManagerFrameBufferData;
        NN_UTIL_SCOPE_EXIT {
            if(!m_IsInitialized)
            {
                pData->~ManagerFrameBufferData();
            }
        };

        buffer_handle_t bufferHandle;
        android::sp<android::GraphicBuffer> pGraphicBuffer;
        int stride = 0;

        NN_RESULT_THROW_UNLESS(
            pServiceData->pGralloc->alloc(
                pServiceData->pGralloc,
                static_cast<uint32_t>(width),
                static_cast<uint32_t>(height),
                GetComposerImageFormat(format),
                BufferUsageBits,
                &bufferHandle,
                &stride
            ) == 0,
            ResultOperationFailed()
        );
        NN_UTIL_SCOPE_EXIT {
            if(!m_IsInitialized)
            {
                NN_ABORT_UNLESS_EQUAL(0, pServiceData->pGralloc->free(pServiceData->pGralloc, bufferHandle));
            }
        };

        pGraphicBuffer = new android::GraphicBuffer(
            static_cast<uint32_t>(width),
            static_cast<uint32_t>(height),
            GetComposerImageFormat(format),
            BufferUsageBits,
            stride,
            const_cast<native_handle_t*>(bufferHandle),
            false
            );
        NN_RESULT_THROW_UNLESS(pGraphicBuffer != nullptr, ResultOperationFailed());

        pData->pService       = pService;
        pData->bufferHandle   = bufferHandle;
        pData->pGraphicBuffer = pGraphicBuffer;
        pData->width  = width;
        pData->height = height;
        pData->format = format;
        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    void ManagerFrameBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);

        auto pData = GetData();
        auto pServiceData = pData->pService->GetData();

        pData->pGraphicBuffer.clear();
        NN_ABORT_UNLESS_EQUAL(0, pServiceData->pGralloc->free(pServiceData->pGralloc, pData->bufferHandle));
        pData->bufferHandle = 0;
        pData->pService = nullptr;
        pData->width  = 0;
        pData->height = 0;
        pData->format = 0;
        pData->~ManagerFrameBufferData();
        m_IsInitialized = false;
    }

    nn::Result ManagerFrameBuffer::CopyFromMemory(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_UNUSED(bufferSize);

        auto pData = GetData();

        size_t frameSize = 4 * pData->width * pData->height;
        NN_RESULT_THROW_UNLESS(
            bufferSize >= frameSize,
            ResultInvalidRange()
        );

        const NvRmSurface* surfaceList = nullptr;
        size_t surfaceCount = 0;
        nvgr_get_surfaces(pData->bufferHandle, &surfaceList, &surfaceCount);

        NN_SDK_ASSERT_EQUAL(surfaceCount, 1);

        NvRmSurfaceWrite(&const_cast<NvRmSurface*>(surfaceList)[0], 0, 0, pData->width, pData->height, pBuffer);
        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerFrameBuffer::CopyFromMemory(int x, int y, int width, int height, const void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_UNUSED(bufferSize);

        auto pData = GetData();

        size_t frameSize = static_cast<size_t>(4 * width * height);
        NN_RESULT_THROW_UNLESS(
            bufferSize >= frameSize,
            ResultInvalidRange()
        );

        const NvRmSurface* surfaceList = nullptr;
        size_t surfaceCount = 0;
        nvgr_get_surfaces(pData->bufferHandle, &surfaceList, &surfaceCount);

        NN_SDK_ASSERT_EQUAL(surfaceCount, 1);

        Bit32 zeroLineBuffer[1280];
        NN_SDK_ASSERT(pData->width <= 1280);
        std::memset(zeroLineBuffer, 0, sizeof(zeroLineBuffer));

        for (uint32_t i = 0; i < pData->height; i++)
        {
            NvRmSurfaceWrite(&const_cast<NvRmSurface*>(surfaceList)[0], 0, i, pData->width, 1, zeroLineBuffer);
        }

        if (pBuffer)
        {
            NvRmSurfaceWrite(&const_cast<NvRmSurface*>(surfaceList)[0], x, y, width, height, pBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ManagerFrameBuffer::Clear(Bit32 clearColor) NN_NOEXCEPT
    {
        auto pData = GetData();

        const NvRmSurface* surfaceList = nullptr;
        size_t surfaceCount = 0;
        nvgr_get_surfaces(pData->bufferHandle, &surfaceList, &surfaceCount);

        NN_SDK_ASSERT_EQUAL(surfaceCount, 1);

        Bit32 clearLineBuffer[1280];
        NN_SDK_ASSERT(pData->width <= 1280);
        for (auto i = 0; i < sizeof(clearLineBuffer) / sizeof(Bit32); ++i)
        {
            clearLineBuffer[i] = clearColor;
        }

        for (uint32_t i = 0; i < pData->height; ++i)
        {
            NvRmSurfaceWrite(&const_cast<NvRmSurface*>(surfaceList)[0], 0, i, pData->width, 1, clearLineBuffer);
        }

        NN_RESULT_SUCCESS;
    }
}}}
