﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedNativeWindow.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include <nn/vi/fbshare/vi_SharedFrameBufferSize.h>

#include <nvrm_channel.h>
#include <nvrm_sync.h>
#include <sync/sync.h>
#include <utils/Errors.h>
#include <ui/PixelFormat.h>

#include "../vi_Log.h"

namespace nn{ namespace vi{ namespace fbshare{

    const int SharedNativeWindow::WindowTextureCountMax;

    SharedNativeWindow::SharedNativeWindow() NN_NOEXCEPT
        : TNativeWindow()
        , m_ExchangeTable()
        , m_PreAcquiredBufferTextureIndex(-1)
        , m_WindowBufferList()
    {
        m_pService = nullptr;
        m_hBuffer = nn::vi::fbshare::SharedBufferHandle::GetInvalidValue();
        m_hLayer = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();

        for(int i = 0; i < WindowTextureCountMax; i++)
        {
            m_WindowBufferList[i].Clear();
        }

        m_Format = android::PIXEL_FORMAT_RGBA_8888;
        m_Usage = 0;
        m_CropRegion = CropRegion();
        m_Transform = 0;
        m_SwapInterval = 1;
    }

    SharedNativeWindow::~SharedNativeWindow() NN_NOEXCEPT
    {
    }

    nn::Result SharedNativeWindow::Initialize(
        nn::sf::SharedPointer<ServiceType> pService,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        nn::vi::fbshare::SharedLayerHandle hLayer
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(pService->OpenSharedLayer(hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(pService->ConnectSharedLayer(hLayer));
        {
            nn::sf::NativeHandle handle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetSharedFrameBufferAcquirableEvent(&handle, hLayer));
            nn::os::AttachReadableHandleToSystemEvent(&m_AcquirableEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
            handle.Detach();
        }

        m_ExchangeTable.Reset();
        m_PreAcquiredBufferTextureIndex = -1;

        for(int i = 0; i < WindowTextureCountMax; i++)
        {
            m_WindowBufferList[i].Clear();
        }

        m_pService = pService;
        m_hBuffer = hBuffer;
        m_hLayer = hLayer;
        m_CropRegion = CropRegion();
        m_Transform = 0;
        m_SwapInterval = 1;
        NN_RESULT_SUCCESS;
    }

    void SharedNativeWindow::Finalize() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&m_AcquirableEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pService->CloseSharedLayer(m_hLayer));

        m_pService = nullptr;
        m_hBuffer = nn::vi::fbshare::SharedBufferHandle::GetInvalidValue();
        m_hLayer = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();
    }

    void SharedNativeWindow::ResetExchangeTable(int* pBufferTextureIndexList, int count) NN_NOEXCEPT
    {
        m_ExchangeTable.Reset(pBufferTextureIndexList, count);

        if(m_PreAcquiredBufferTextureIndex >= 0)
        {
            NN_SDK_ASSERT_GREATER_EQUAL(m_ExchangeTable.ExchangeToWindowTextureIndex(m_PreAcquiredBufferTextureIndex), 0);
        }
    }

    nn::Result SharedNativeWindow::PreAcquireTexture(SharedLayerTextureIndexList* pOutIndexList, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutIndexList);
        NN_ABORT_UNLESS_LESS(m_PreAcquiredBufferTextureIndex, 0);

        int64_t iBuf = -1;
        nn::vi::native::NativeSync sync = {};
        auto indexList = SharedLayerTextureIndexList::GetInvalidValue();

        if(!nn::os::TimedWaitSystemEvent(&m_AcquirableEvent, timeout))
        {
            NN_VI_LOG_DEV("[share][native]aquirable event timeout.\n");
            NN_RESULT_THROW(nn::vi::ResultNotReady());
        }

        auto result = m_pService->AcquireSharedFrameBuffer(&iBuf, &sync, &indexList, m_hLayer);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::vi::ResultPrivateSharedLayerNotAcquirable)
            {
                NN_VI_LOG_DEV("[share][native]not acquirable.\n");
                NN_RESULT_THROW(nn::vi::ResultNotReady());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_VI_LOG_ERR("[share][native]Aquire shared frame buffer failed (%d-%d)\n", result.GetModule(), result.GetDescription());
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        NN_RESULT_END_TRY;

        // ok
        NN_VI_LOG_DEV("acquired\n");
        NN_ABORT_UNLESS_RANGE(iBuf, 0, SharedBufferTextureCountMax);

        m_PreAcquiredBufferTextureIndex = iBuf;
        m_PreAcquiredBufferSync = sync;
        *pOutIndexList = indexList;

        // アタッチされているバッファに変化があったかチェック
        if(m_ExchangeTable.ExchangeToWindowTextureIndex(iBuf) < 0)
        {
            NN_VI_LOG_DEV("[share][native]detected attached buffer change (acqIdx=%d)\n");
            NN_RESULT_THROW(nn::vi::ResultPrivateSharedLayerBufferChanged());
        }

        NN_RESULT_SUCCESS;
    }

    void SharedNativeWindow::CancelTexture(int iWin) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]CancelTexture(iWin=%d)\n", iWin);
        int iBuf = m_ExchangeTable.ExchangeToBufferTextureIndex(iWin);
        NN_ABORT_UNLESS_RANGE(iBuf, 0, SharedBufferTextureCountMax);

        // もし Cancel 前に強制デタッチされていた場合失敗が返ってくるので失敗は無視する。
        auto result = m_pService->CancelSharedFrameBuffer(m_hLayer, iBuf);
        if(result.IsFailure())
        {
            NN_VI_LOG_DEV("[share][native]CancelSharedFrameBuffer error ignored (%d-%d)\n", result.GetModule(), result.GetDescription());
        }
    }

    void SharedNativeWindow::SetSwapInterval(int value) NN_NOEXCEPT
    {
        m_SwapInterval = value;
    }

    void SharedNativeWindow::SetCropRegion(const CropRegion& crop) NN_NOEXCEPT
    {
        m_CropRegion = crop;
    }

    int SharedNativeWindow::SetSwapIntervalImpl(int interval) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]SetSwapInterval %d(0x%X) -> ignored\n", interval, interval);
        return android::OK;
    }

    int SharedNativeWindow::DequeueBufferImpl(ANativeWindowBuffer** pOutBuffer, int* pOutFenceFd) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
        NN_VI_LOG_DEV("[share][native]DequeueBuffer\n");

        NN_SDK_REQUIRES_RANGE(m_PreAcquiredBufferTextureIndex, 0, SharedBufferTextureCountMax);

        int iWin = m_ExchangeTable.ExchangeToWindowTextureIndex(m_PreAcquiredBufferTextureIndex);
        NN_SDK_ASSERT_RANGE(iWin, 0, WindowTextureCountMax);


        // make sync fd
        int fd = -1;
        auto pSync = reinterpret_cast<nn::vi::native::NativeSyncImpl*>(&m_PreAcquiredBufferSync);
        if(pSync->count > 0)
        {
            NN_VI_LOG_DEV("acq idx=%db(->%dw) fence (%d) %d-%d | %d-%d | %d-%d | %d-%d\n",
                m_PreAcquiredBufferTextureIndex,
                iWin,
                pSync->count,
                pSync->list[0].id, pSync->list[0].value,
                pSync->list[1].id, pSync->list[1].value,
                pSync->list[2].id, pSync->list[2].value,
                pSync->list[3].id, pSync->list[3].value);
            fd = sync_create_nvrm_mapping(
                reinterpret_cast<NvRmFence*>(&pSync->list[0]),
                std::min<uint32_t>(pSync->count, nn::vi::native::NativeSyncImpl::CountMax)
            );
        }
        else
        {
            NN_VI_LOG_DEV("acq idx=%db(->%dw) no-fence\n", m_PreAcquiredBufferTextureIndex, iWin);
        }

        if(pOutFenceFd)
        {
            *pOutFenceFd = fd;
            NN_VI_LOG_DEV("fence fd = %d\n", fd);
        }
        else
        {
            NN_VI_LOG("deplicated version of dequeue called\n");
            if(fd >= 0)
            {
                sync_wait(fd, -1);
                sync_close(fd);
            }
        }

        m_PreAcquiredBufferTextureIndex = -1;
        m_PreAcquiredBufferSync         = nn::vi::native::NativeSync();
        *pOutBuffer = m_WindowBufferList[iWin].GetNativeWindowBuffer();
        //NN_VI_LOG_DEV("[share][native]DequeueBuffer -> OK:%d\n", index);
        return android::OK;
    }

    int SharedNativeWindow::CancelBufferImpl(ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]CancelBuffer\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueueBufferImpl(ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueueBuffer\n");

        int iWin = SharedNativeWindowBuffer::GetSelf(pBuffer)->GetWindowBufferIndex();
        int iBuf = m_ExchangeTable.ExchangeToBufferTextureIndex(iWin);
        NN_ABORT_UNLESS_GREATER_EQUAL(iBuf, 0);

        // extract syncpoints
        nn::vi::native::NativeSync sync = {};
        if(fenceFd >= 0)
        {
            auto pSync = reinterpret_cast<nn::vi::native::NativeSyncImpl*>(&sync);
            syncfd_nvrm_mapping mapping = {};
            sync_get_nvrm_mapping(fenceFd, &mapping);

            pSync->count = std::min<uint32_t>(pSync->CountMax, mapping.num_fences);
            std::memcpy(&pSync->list, mapping.fences, sizeof(NvRmFence) * pSync->count);

            sync_close(fenceFd);
        }

        NN_VI_LOG_DEV("[share][native]  iBuf=%d,crop=(%d,%d;%dx%d),trans=%X,swp=%d\n", iBuf, m_CropRegion.x, m_CropRegion.y, m_CropRegion.width, m_CropRegion.height, m_Transform, m_SwapInterval);
        auto result = m_pService->PresentSharedFrameBuffer(m_hLayer, iBuf, sync, m_CropRegion, static_cast<nn::vi::ImageTransformType>(m_Transform), m_SwapInterval);
        if(result.IsFailure())
        {
            // もし Cancel 前に強制デタッチされていた場合失敗が返ってくるので失敗は無視する。
            // それ以外の理由で失敗していた場合、次の PreAcquireBuffer で同じエラーが捕捉されるはずなので無視で構わない。
            NN_VI_LOG_DEV("[share][native]PresentSharedFrameBuffer error ignored (%d-%d)\n", result.GetModule(), result.GetDescription());
        }
        return android::OK;
    }

    int SharedNativeWindow::QueryFormatImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryFormat -> OK: %d(0x%X)\n", m_Format, m_Format);
        *pOutValue = m_Format;
        return android::OK;
    }

    int SharedNativeWindow::QueryQueuesToWindowComposerImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryQueuesToWindowComposer\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryConcreteTypeImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryConcreteType\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryDefaultWidthImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryDefaultWidth -> OK: %d\n", nn::vi::fbshare::SharedFrameBufferWidth);
        *pOutValue = nn::vi::fbshare::SharedFrameBufferWidth;
        return android::OK;
    }

    int SharedNativeWindow::QueryDefaultHeightImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryDefaultHeight -> OK: %d\n", nn::vi::fbshare::SharedFrameBufferHeight);
        return android::OK;
    }

    int SharedNativeWindow::QueryTransformHintImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryTransformHint\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryWidthImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryWidth\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryHeightImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryHeight\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryMinUndequeuedBuffersImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryMinUndequeuedBuffers\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryConsumerRunningBehindImpl(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryConsumerRunningBehind\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryConsumerUsageBits(int* pOutValue) const NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]QueryConsumerUsage\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::QueryDefault(int *pOutValue, int what) const NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(what);
        NN_VI_LOG_ERR("[share][native]unknown query %d(0x%X) -> BAD_VALUE\n", what, what);
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetUsageImpl(uint32_t usage) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetUsage %d(0x%X) -> OK\n", usage, usage);
        m_Usage = usage;
        return android::OK;
    }

    int SharedNativeWindow::PerformSetCropImpl(const android_native_rect_t* pRect) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetCrop -> ignored\n");
        return android::OK;
    }

    int SharedNativeWindow::PerformSetBufferCountImpl(size_t bufferCount) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBufferCount\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetBuffersGeometryImpl(int w, int h, int f) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersGeometry\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetBuffersTransformImpl(int transform) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersTransform %d(0x%X) -> OK\n", transform, transform);
        m_Transform = transform;
        return android::OK;
    }

    int SharedNativeWindow::PerformSetBuffersFlagsImpl(uint32_t flags) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersFlags\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetBuffersTimestampImpl(int64_t timestamp) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersTimestamp\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetBuffersDimensionsImpl(int w, int h) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersDimensions w=%d, h=%d -> ignored\n", w, h);
        return android::OK;
    }

    int SharedNativeWindow::PerformSetBuffersUserDimensionsImpl(int w, int h) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersUserDimensions\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetBuffersFormatImpl(int f) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetBuffersFormat\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformLockImpl(ANativeWindow_Buffer* pOutBuffer, ARect* pInOutDirtyBounds) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformLock\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformUnlockAndPostImpl() NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformUnlockAndPostIm\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetScalingModeImpl(int mode) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetScalingMode\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformConnectImpl(int api) NN_NOEXCEPT
    {
        NN_UNUSED(api);
        // Connect/Disconnect は NVN の外側で行う
        NN_VI_LOG_DEV("[share][native]PerformConnect(%d) -> ignored\n", api);
        return android::OK;
    }

    int SharedNativeWindow::PerformDisconnectImpl(int api) NN_NOEXCEPT
    {
        NN_UNUSED(api);
        // Connect/Disconnect は NVN の外側で行う
        NN_VI_LOG_DEV("[share][native]PerformDisconnect(%d) -> ignored\n", api);
        return android::OK;
    }

    int SharedNativeWindow::PerformSetSidebandStreamImpl(native_handle_t* pHandle) NN_NOEXCEPT
    {
        NN_VI_LOG_DEV("[share][native]PerformSetSidebandStream\n");
        return android::BAD_VALUE;
    }

    int SharedNativeWindow::PerformSetPreallocatedBufferImpl(size_t slot, const native_handle_t* pHandle, int width, int height, int format, int usage, int stride) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slot, 0, WindowTextureCountMax);
        if(pHandle == nullptr)
        {
            NN_VI_LOG_DEV("[share][native]PerformSetPreallocatedBuffer(Clean) slot=%d,H=0x%X,w=%d,h=%d,f=%d,u=%d,s=%d -> OK\n",
                slot, pHandle, width, height, format, usage, stride);
            m_WindowBufferList[slot].Clear();
            return android::OK;
        }

        if(!m_WindowBufferList[slot].IsEmpty())
        {
            NN_VI_LOG_DEV("[share][native]PerformSetPreallocatedBuffer slot=%d,H=0x%X,w=%d,h=%d,f=%d,u=%d,s=%d -> BAD_VALUE\n",
                slot, pHandle, width, height, format, usage, stride);
            return android::BAD_VALUE;
        }

        auto& buf = m_WindowBufferList[slot];
        buf.Set(width, height, stride, format, usage, pHandle, slot);
        NN_VI_LOG_DEV("[share][native]PerformSetPreallocatedBuffer slot=%d,H=0x%X,w=%d,h=%d,f=%d,u=%d,s=%d -> OK\n",
            slot, pHandle, width, height, format, usage, stride);
        return android::OK;
    }

    int SharedNativeWindow::PerformDefault(int operation, va_list args) NN_NOEXCEPT
    {
        NN_UNUSED(operation);
        NN_UNUSED(args);
        NN_VI_LOG_ERR("[share][native]unknown perform %d(%X)\n", operation, operation);
        return android::BAD_VALUE;
    }

}}}
