﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedLayerWindowGfx.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/fbshare/vi_SharedBufferConfig.h>
#include <nn/gfx.h>

namespace nn{ namespace vi{ namespace fbshare{

    NN_STATIC_ASSERT(SharedLayerWindowGfx::InternalTextureCountMax == SharedBufferTextureCountMax);

    bool SharedLayerWindowGfx::IsSupported() NN_NOEXCEPT
    {
        return SharedLayerWindow::IsSupported();
    }

    SharedLayerWindowGfx::SharedLayerWindowGfx() NN_NOEXCEPT
        : m_WindowNvn()
        , m_pDevice(nullptr)
        , m_ImageFormat()
        , m_IsGfxTextureInitializedList({})
    {
        std::memset(m_GfxTextureList, 0, sizeof(m_GfxTextureList));
    }

    SharedLayerWindowGfx::~SharedLayerWindowGfx() NN_NOEXCEPT
    {
    }

    nn::Result SharedLayerWindowGfx::Initialize(nn::gfx::Device* pDevice, int bufferCount, nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        NVNdevice* pNvnDevice = pDevice->ToData()->pNvnDevice;
        NVNformat nvnFormat;
        switch(format)
        {
        case nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm:
            nvnFormat = NVN_FORMAT_RGBA8;
            break;
        case nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb:
            nvnFormat = NVN_FORMAT_RGBA8_SRGB;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_DO(m_WindowNvn.Initialize(pNvnDevice, bufferCount, nvnFormat));

        m_pDevice = pDevice;
        m_ImageFormat = format;
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowGfx::Finalize() NN_NOEXCEPT
    {
        for(int i = 0; i < InternalTextureCountMax; i++)
        {
            if(!m_IsGfxTextureInitializedList[i])
            {
                continue;
            }
            m_GfxColorTargetViewList[i].Finalize(m_pDevice);
        }

        // Interop で初期化した nn::gfx::Texture の初期化済フラグを落とす正規の手段が存在しないので memset でクリアする
        std::memset(m_GfxTextureList, 0, sizeof(m_GfxTextureList));

        m_IsGfxTextureInitializedList = {};
        m_WindowNvn.Finalize();

        m_pDevice = nullptr;
        m_ImageFormat = {};
    }

    int SharedLayerWindowGfx::GetTextureWidth(nn::gfx::Texture* pTexture) const NN_NOEXCEPT
    {
        return m_WindowNvn.GetTextureWidth(pTexture->ToData()->pNvnTexture);
    }

    int SharedLayerWindowGfx::GetTextureHeight(nn::gfx::Texture* pTexture) const NN_NOEXCEPT
    {
        return m_WindowNvn.GetTextureHeight(pTexture->ToData()->pNvnTexture);
    }

    //------------------------------------------------------

    bool SharedLayerWindowGfx::IsTexturePreAcquired() const NN_NOEXCEPT
    {
        return m_WindowNvn.IsTexturePreAcquired();
    }

    bool SharedLayerWindowGfx::IsTextureAcquired() const NN_NOEXCEPT
    {
        return m_WindowNvn.IsTextureAcquired();
    }

    nn::Result SharedLayerWindowGfx::PreAcquireTexture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        return m_WindowNvn.PreAcquireTexture(timeout);
    }

    void SharedLayerWindowGfx::CancelPreAcquiredTexture() NN_NOEXCEPT
    {
        return m_WindowNvn.CancelPreAcquiredTexture();
    }

    nn::Result SharedLayerWindowGfx::AcquireTexture(nn::gfx::Semaphore* pOutTextureAvailableSemaphore, nn::gfx::Fence* pOutTextureAvailableFence, int* pOutTextureIndex, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT
    {
        NVNsync* pNvnSync = pOutTextureAvailableSemaphore->ToData()->pNvnSync;
        NVNtexture* pNvnTexture = nullptr;
        NN_RESULT_DO(m_WindowNvn.AcquireTexture(pNvnSync, pOutTextureIndex, &pNvnTexture));

        if(pNvnTexture != nullptr)
        {
            ConvertToGfxTextureImpl(pOutTexture, pNvnTexture);
        }
        else
        {
            *pOutTexture = nullptr;
        }

        if(pOutTextureAvailableFence)
        {
            *static_cast<NVNsync*>(pOutTextureAvailableFence->ToData()->pNvnSync) = *pNvnSync;
        }

        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowGfx::AcquirePreAcquiredTexture(nn::gfx::Semaphore* pOutTextureAvailableSemaphore, nn::gfx::Fence* pOutTextureAvailableFence, int* pOutTextureIndex, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT
    {
        NVNsync* pNvnSync = pOutTextureAvailableSemaphore->ToData()->pNvnSync;
        NVNtexture* pNvnTexture = nullptr;
        m_WindowNvn.AcquirePreAcquiredTexture(pNvnSync, pOutTextureIndex, &pNvnTexture);

        if(pNvnTexture != nullptr)
        {
            ConvertToGfxTextureImpl(pOutTexture, pNvnTexture);
        }
        else
        {
            *pOutTexture = nullptr;
        }

        if(pOutTextureAvailableFence)
        {
            *static_cast<NVNsync*>(pOutTextureAvailableFence->ToData()->pNvnSync) = *pNvnSync;
        }
    }

    void SharedLayerWindowGfx::PresentTexture(nn::gfx::Queue* pQueue, int textureIndex) NN_NOEXCEPT
    {
        NVNqueue* pNvnQueue = pQueue->ToData()->pNvnQueue;
        m_WindowNvn.PresentTexture(pNvnQueue, textureIndex);
    }

    void SharedLayerWindowGfx::CancelTexture(int textureIndex) NN_NOEXCEPT
    {
        m_WindowNvn.CancelTexture(textureIndex);
    }

    void SharedLayerWindowGfx::SetSwapInterval(int value) NN_NOEXCEPT
    {
        m_WindowNvn.SetSwapInterval(value);
    }

    void SharedLayerWindowGfx::SetCrop(int x, int y, int width, int height) NN_NOEXCEPT
    {
        m_WindowNvn.SetCrop(x, y, width, height);
    }

    nn::gfx::ColorTargetView* SharedLayerWindowGfx::GetTextureColorTargetView(nn::gfx::Texture* pTexture) NN_NOEXCEPT
    {
        if(pTexture == nullptr)
        {
            return nullptr;
        }

        int iBuf = -1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetInternalTextureIndex(&iBuf, pTexture));
        NN_SDK_REQUIRES(m_IsGfxTextureInitializedList[iBuf]);
        return &m_GfxColorTargetViewList[iBuf];
    }

    //------------------------------------------------------

    nn::Result SharedLayerWindowGfx::AcquireLastApplicationTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT
    {
        NVNtexture* pNvnTexture = nullptr;
        NN_RESULT_DO(m_WindowNvn.AcquireLastApplicationTexture(pOutIsScreenShotPermitted, &pNvnTexture));
        ConvertToGfxTextureImpl(pOutTexture, pNvnTexture);
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowGfx::ReleaseLastApplicationTexture() NN_NOEXCEPT
    {
        m_WindowNvn.ReleaseLastApplicationTexture();
    }

    nn::Result SharedLayerWindowGfx::AcquireLastForegroundTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT
    {
        NVNtexture* pNvnTexture = nullptr;
        NN_RESULT_DO(m_WindowNvn.AcquireLastForegroundTexture(pOutIsScreenShotPermitted, &pNvnTexture));
        ConvertToGfxTextureImpl(pOutTexture, pNvnTexture);
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowGfx::ReleaseLastForegroundTexture() NN_NOEXCEPT
    {
        m_WindowNvn.ReleaseLastForegroundTexture();
    }

    nn::Result SharedLayerWindowGfx::AcquireCallerAppletTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT
    {
        NVNtexture* pNvnTexture = nullptr;
        NN_RESULT_DO(m_WindowNvn.AcquireCallerAppletTexture(pOutIsScreenShotPermitted, &pNvnTexture));
        ConvertToGfxTextureImpl(pOutTexture, pNvnTexture);
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindowGfx::ReleaseCallerAppletTexture() NN_NOEXCEPT
    {
        m_WindowNvn.ReleaseCallerAppletTexture();
    }

    //------------------------------------------------------

    nn::Result SharedLayerWindowGfx::GetInternalTextureIndex(int* pOutValue, const nn::gfx::Texture* pTexture) const NN_NOEXCEPT
    {
        NVNtexture* pNvnTexture = pTexture->ToData()->pNvnTexture;
        return m_WindowNvn.GetInternalTextureIndex(pOutValue, pNvnTexture);
    }

    void SharedLayerWindowGfx::ConvertToGfxTextureImpl(nn::gfx::Texture** pOutGfxTexture, NVNtexture* pNvnTexture) NN_NOEXCEPT
    {
        int iBuf = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_WindowNvn.GetInternalTextureIndex(&iBuf, pNvnTexture));

        if(!m_IsGfxTextureInitializedList[iBuf])
        {
            // テクスチャの登録
            nn::gfx::InteroperationNvn::ConvertToGfxTexture(&m_GfxTextureList[iBuf], pNvnTexture);

            // カラーターゲットビューの登録
            {
                nn::gfx::ColorTargetViewInfo info;
                info.SetDefault();
                info.SetImageDimension(nn::gfx::ImageDimension_2d);
                info.SetImageFormat(m_ImageFormat);
                info.SetMipLevel(0);
                info.SetTexturePtr(&m_GfxTextureList[iBuf]);
                m_GfxColorTargetViewList[iBuf].Initialize(m_pDevice, info);
            }

            m_IsGfxTextureInitializedList[iBuf] = true;
        }

        *pOutGfxTexture = &m_GfxTextureList[iBuf];
    }

}}}


