﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>

#include <nn/usb/pd/server/usb_PdHipcServer.h>

#include "../detail/usb_IPdManager.h"
#include "../detail/usb_IPdCradleManager.h"
#include "../detail/usb_PdServiceName.h"
#include "../driver/detail/usb_PdSession.h"

#include "usb_PdManagerImpl.h"
#include "usb_PdCradleManagerImpl.h"

namespace nn { namespace usb { namespace pd { namespace server {

namespace {


// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1>
{};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyCradleServerManagerStorage;
MyServerManager* g_pMyCradleServerManager;

nn::sf::UnmanagedServiceObject<detail::IPdManager, PdManagerImpl> g_PdManager;
nn::sf::UnmanagedServiceObject<detail::IPdCradleManager, PdCradleManagerImpl> g_PdCradleManager;
}

void InitializePdServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);
    NN_SDK_ASSERT(!g_pMyCradleServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;
    g_pMyCradleServerManager = new (&g_MyCradleServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    g_pMyServerManager->RegisterObjectForPort(g_PdManager.GetShared(), driver::detail::MaxSessions, detail::ServiceName);
    g_pMyCradleServerManager->RegisterObjectForPort(g_PdCradleManager.GetShared(), driver::detail::MaxCradleSessions, detail::CradleServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
    g_pMyCradleServerManager->Start();
}

void FinalizePdServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyCradleServerManager);
    NN_SDK_ASSERT(g_pMyServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyCradleServerManager->~MyServerManager();
    g_pMyCradleServerManager = nullptr;
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

void LoopPdServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // サーバの処理ループ実行を実行する。
    // - ポートへの接続要求
    // - セッションへの処理要求
    // RequestStop が呼ばれるまで返らない。
    g_pMyServerManager->LoopAuto();
}

void LoopPdCradleServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyCradleServerManager);

    // サーバの処理ループ実行を実行する。
    // - ポートへの接続要求
    // - セッションへの処理要求
    // RequestStop が呼ばれるまで返らない。
    g_pMyCradleServerManager->LoopAuto();
}

void RequestStopPdServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyCradleServerManager);
    NN_SDK_ASSERT(g_pMyServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyCradleServerManager->RequestStop();
    g_pMyServerManager->RequestStop();
}

}}}}
