﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/usb/pd/driver/usb_Pd.h>

#include "usb_PdResult.h"
#include "usb_PdSession.h"

namespace nn {
namespace usb {
namespace pd {
namespace driver {
namespace detail {

void Session::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount < INT_MAX ); // 値域チェック

    m_InitializeCount++;

    // 初期化済みならば何も行わない
    if ( m_InitializeCount > 1 )
    {
        USBPD_DBG_LOG("%s(%d) Already Initialized. COUNT=%d\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 1); // 念のためチェック

    for ( int i=0; i<MaxSessions; i++ )
    {
        // Driver コンストラクタで初期化できないため placement new を使う
        new (&m_EventHolder[i]) EventHolder();
    }
}

void Session::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount > INT_MIN ); // 値域チェック

    m_InitializeCount--;

    // 終了回数が初期化された回数に達していない場合は何も行わない
    if ( m_InitializeCount > 0 )
    {
        USBPD_DBG_LOG("%s(%d) Not Finalized. COUNT=%d.\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 0);

    std::lock_guard<nn::os::Mutex> lockSession( m_SessionMutex );

    // 登録イベントの破棄
    for ( auto itr = m_NoticeEventList.begin(); itr != m_NoticeEventList.end(); ++itr )
    {
        // Bound 済みのものが残っていた場合
        if ( itr->IsBoundEvent() )
        {
            // 渡されたイベントを破棄
            itr->DestroyEvent();
        }
    }
    m_NoticeEventList.clear();

    for( int i = 0; i < MaxSessions; i++ )
    {
        m_IsSessionOpen[i] = false;
    }
    for ( int i = 0; i < MaxCradleSessions; i++ )
    {
        m_IsCradleSessionOpen[i] = false;
    }
}

int Session::FindFreeSessionId( bool* isOpenTable, int maxSessions ) const NN_NOEXCEPT
{
    for( int i = 0; i < MaxSessions; i++ )
    {
        if ( !isOpenTable[i])
        {
            return i; // 見つけた
        }
    }

    return -1; // 全部オープン済だった
}

void Session::OpenSession( driver::Session* pOutSession ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    int maxSessions = MaxSessions;
    int sessionId = FindFreeSessionId( m_IsSessionOpen, maxSessions );
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < maxSessions, "No available sessions on usb:pd");

    pOutSession->_sessionId = sessionId;
    m_IsSessionOpen[sessionId] = true;

//    USBPD_DBG_LOG("%s(%d) OpenSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}

void Session::CloseSession( const driver::Session& session ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    auto sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxSessions, "Invalid session ID");

    NN_SDK_ASSERT(m_IsSessionOpen[sessionId], "%s(%d) Session %d is already closed", __FUNCTION__, __LINE__, sessionId );

    // クライアントの Unbind 漏れ対応
    UnbindNoticeEvent( session );

    m_IsSessionOpen[sessionId] = false;

//    USBPD_DBG_LOG("%s(%d) CloseSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}

bool Session::IsOpen( int sessionId ) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxSessions, "Invalid session ID");
    return m_IsSessionOpen[sessionId];
}

void Session::OpenCradleSession( driver::CradleSession* pOutSession ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    int maxSessions = MaxCradleSessions;
    int sessionId = FindFreeSessionId( m_IsCradleSessionOpen, maxSessions );
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < maxSessions, "No available sessions on usb:pd:c");

    pOutSession->_sessionId = sessionId;
    m_IsCradleSessionOpen[sessionId] = true;

//    USBPD_DBG_LOG("%s(%d) OpenCradleSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}

void Session::CloseCradleSession( const driver::CradleSession& session ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    auto sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxCradleSessions, "Invalid cradle session ID");

    NN_SDK_ASSERT(m_IsCradleSessionOpen[sessionId], "%s(%d) Session %d is already closed", __FUNCTION__, __LINE__, sessionId );

    m_IsCradleSessionOpen[sessionId] = false;

//    USBPD_DBG_LOG("%s(%d) CloseCradleSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
}

bool Session::IsCradleOpen( int sessionId ) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxCradleSessions, "Invalid cradle session ID");
    return m_IsCradleSessionOpen[sessionId];
}

Result Session::BindNoticeEvent( nn::os::SystemEventType* pEvent, const driver::Session& session ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

//    USBPD_DBG_LOG("Session::BindNoticeEvent\n");

    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    auto sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxSessions, "Invalid session ID");
    EventHolder* pEventHolder = &::nn::util::Get(m_EventHolder[sessionId]);

    // セッションがオープンでなければエラーを返す
    if ( !m_IsSessionOpen[sessionId] )
    {
        USBPD_WARNING("Session %d isn't opened!", sessionId);
        return ResultSessionNotOpen();
    }
    // すでにバインド済みなら登録せずにエラーを返す
    if ( pEventHolder->IsBoundEvent() )
    {
        USBPD_WARNING("Session::NoticeEvent id already bound!\n");
        return ResultAlreadyBound();
    }

    // SystemEvent を初期化する。
    nn::os::CreateSystemEvent( pEvent, nn::os::EventClearMode_ManualClear, true );

    // イベントをセッション内に登録
    pEventHolder->AttachEvent( pEvent );

    // Session で管理している紐付済みセッションに登録する
    m_NoticeEventList.push_back( *pEventHolder );

    return result;
}

Result Session::UnbindNoticeEvent( const driver::Session& session ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

//    USBPD_DBG_LOG("Session::UnbindNoticeEvent\n");

    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    auto sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxSessions, "Invalid session ID");
    EventHolder* pEventHolder = &::nn::util::Get(m_EventHolder[sessionId]);

    // セッションがオープンでなければエラーを返す
    if ( !m_IsSessionOpen[sessionId] )
    {
        USBPD_WARNING("Session %d isn't opened!", sessionId);
        return ResultSessionNotOpen();
    }
    // バインド済みでなければ何もしない
    if ( !pEventHolder->IsBoundEvent() )
    {
        return ResultSuccess();
    }

    // EventHolder を紐付済みリストから外す
    m_NoticeEventList.erase( m_NoticeEventList.iterator_to( *pEventHolder ) );

    // 渡されたイベントを破棄する。
    pEventHolder->DestroyEvent();

    return result;
}

void Session::NoticeEventCallback( int sessionId ) NN_NOEXCEPT
{
    USBPD_DBG_LOG("Session%d::NoticeEvent\n", sessionId);
    NN_ABORT_UNLESS(0 <= sessionId && sessionId < MaxSessions, "Invalid session ID");

    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    EventHolder* pEventHolder = &::nn::util::Get(m_EventHolder[sessionId]);
    pEventHolder->SignalEvent();

}

void Session::NoticeEventCallback() NN_NOEXCEPT
{
    USBPD_DBG_LOG("AllSession::NoticeEvent\n");

    std::lock_guard<nn::os::Mutex> lock( m_SessionMutex );

    for (auto&& e : m_NoticeEventList)
    {
        e.SignalEvent();
    }
}

} // detail
} // driver
} // pd
} // usb
} // nn
