﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

UsbRequestBlock::UsbRequestBlock(UrbPool& urbPool, HostControllerDriverEndpoint *pHcEp)
    : m_State(State_Invalid)
    , m_pHcEp(pHcEp)
    , m_IoVa(0LL)
    , m_BufferSize(0)
    , m_IsSmmuMapDynamic(false)
    , m_Token(UsbBusToken_Invalid)
    , m_pDeferredRequest(nullptr)
    , m_Context(nullptr)
    , m_Callback(nullptr)
    , m_TimeoutInMs(0)
    , m_Pool(urbPool)
{
}

UsbRequestBlock::~UsbRequestBlock()
{
    // nothing special
}

// submit an URB (which could carry several xfers)
Result UsbRequestBlock::Submit()
{
    Result result = ResultSuccess();

    NN_USB_ABORT_UNLESS(m_State == State_Allocated);

    if (m_ProcVa != 0 && m_BufferSize > 0)
    {
        if (m_Token == UsbBusToken_In)
        {
            nn::dd::InvalidateProcessDataCache(m_ProcHandle, m_ProcVa, m_BufferSize);
        }
        else
        {
            nn::dd::FlushProcessDataCache(m_ProcHandle, m_ProcVa, m_BufferSize);
        }
    }

    result = m_pHcEp->SubmitUrb(this);
    if (result.IsSuccess())
    {
        m_Pool.OnUrbSubmission(this);

        if (m_TimeoutInMs != 0)
        {
            m_Pool.StartTimedEvent(&m_TimeoutEvent, m_TimeoutInMs);
        }
    }

    return result;
}

// complete a single xfer carried in the URB
void UsbRequestBlock::Complete()
{
    auto& xfer = m_Xfer[m_CompletionCount++];

    NN_USB_ABORT_UNLESS(m_State == State_Pending);

    // Are short transfers OK for this endpoint?
    HostEndpoint *pHep = m_pHcEp->GetHostEndpoint();
    if (!pHep->isShortTransferError && ResultShortPacket::Includes(xfer.result))
    {
        xfer.result = ResultSuccess();
    }

    if (m_CompletionCount == m_XferCount)
    {
        if (m_TimeoutInMs != 0)
        {
            m_Pool.StopTimedEvent(&m_TimeoutEvent);
        }

        // Need to invalidate again
        if ((m_ProcVa != 0) && (m_Token == UsbBusToken_In))
        {
            nn::dd::InvalidateProcessDataCache(m_ProcHandle, m_ProcVa, m_BufferSize);
        }

        m_pHcEp->CompleteUrb(this);
        m_Pool.OnUrbCompletion(this);
    }
}

UrbPool::UrbPool(Hs *pHs, HostControllerDriverEndpoint *pHcEp)
    : m_pHs(pHs)
    , m_pHcEp(pHcEp)
    , m_pPlatform(pHcEp->GetPlatformController())
{
    m_pHs->m_LocalEventManager.RegisterPool(
        this,
        &m_LocalEventPool,
        TimerCallbackStatic,
        m_LocalEventStorage,
        sizeof(m_LocalEventStorage[0]),
        NN_USB_ARRAY_SIZE(m_LocalEventStorage)
    );
}

UrbPool::~UrbPool()
{
    NN_USB_ABORT_UNLESS(m_UrbFreeList.empty());
    NN_USB_ABORT_UNLESS(m_UrbAllocatedList.empty());
    NN_USB_ABORT_UNLESS(m_UrbPendingList.empty());

    m_pHs->m_LocalEventManager.UnRegisterPool(&m_LocalEventPool);
}

Result UrbPool::Initialize(uint32_t count)
{
    Result result = ResultSuccess();

    for (uint32_t i = 0; i < count; i++)
    {
        UsbRequestBlock *pUrb = new UsbRequestBlock(*this, m_pHcEp);
        if (pUrb == nullptr)
        {
            DeleteAllUrbs();
            result = ResultMemAllocFailure();
            break;
        }
        else
        {
            m_pHs->m_LocalEventManager.InitStaticEvent(&m_LocalEventPool,
                                                       &pUrb->m_TimeoutEvent);

            pUrb->m_State = UsbRequestBlock::State_Free;

            m_UrbFreeList.push_back(*pUrb);
        }
    }

    return result;
}

Result UrbPool::Finalize()
{
    DeleteAllUrbs();

    return ResultSuccess();
}

UsbRequestBlock* UrbPool::AllocateUrb()
{
    UsbRequestBlock *pNewUrb = nullptr;

    if (!m_UrbFreeList.empty())
    {
        pNewUrb = &m_UrbFreeList.front();

        // remove from the free list
        m_UrbFreeList.pop_front();

        pNewUrb->m_IoVa            = 0;
        pNewUrb->m_State           = UsbRequestBlock::State_Allocated;
        pNewUrb->m_CompletionCount = 0;

        // add to the allocated list
        m_UrbAllocatedList.push_back(*pNewUrb);
    }

    return pNewUrb;
}

Result UrbPool::AllocateUrb(UsbRequestBlock                    **ppCreatedUrb,
                            ClientEpSession::UrbDeferredRequest *pRequest)
{
    Result result = ResultSuccess();
    UsbRequestBlock *pUrb  = nullptr;
    auto& data = pRequest->m_Data;

    // allocate and initialize the URB
    pUrb = AllocateUrb();
    if (pUrb == nullptr)
    {
        return ResultUrbAllocFailure();
    }

    pUrb->m_ProcHandle         = data.procHandle;
    pUrb->m_pSmmuSpace         = data.pSmmuSpace;
    pUrb->m_ProcVa             = data.procVa;
    pUrb->m_IoVa               = 0;
    pUrb->m_BufferSize         = 0;
    pUrb->m_IsSmmuMapDynamic   = (pUrb->m_pSmmuSpace == nullptr);
    pUrb->m_XferCount          = data.xferCount;
    pUrb->m_Policy             = data.policy;
    pUrb->m_FrameId            = data.frameId;
    pUrb->m_pDeferredRequest   = pRequest;
    pUrb->m_TimeoutInMs        = data.timeoutInMs;

    for (uint32_t i = 0; i < data.xferCount; i++)
    {
        pUrb->m_Xfer[i].size        = data.xfers[i].size;
        pUrb->m_Xfer[i].xferredSize = 0;
        pUrb->m_Xfer[i].result      = ResultSuccess();

        pUrb->m_BufferSize         += data.xfers[i].size;
    }

    if (data.epDirection == UsbEndpointDirection_ToDevice)
    {
        pUrb->m_Token = UsbBusToken_Out;
    }
    else
    {
        pUrb->m_Token = UsbBusToken_In;
    }

    // sMMU Map
    if (pUrb->m_IsSmmuMapDynamic)
    {
        if (pUrb->m_BufferSize > 0)
        {
            if (pUrb->m_Token == UsbBusToken_Out)
            {
                result = m_pPlatform->DoSmmu().Map(
                    &pUrb->m_IoVa,
                    pUrb->m_ProcHandle, pUrb->m_ProcVa,
                    Util::RoundupDmaBufferSize(pUrb->m_BufferSize),
                    pUrb->BufferAlignment,
                    nn::dd::MemoryPermission_ReadOnly,
                    nn::dd::MemoryPermission_ReadOnly
                );
            }
            else
            {
                result = m_pPlatform->DoSmmu().Map(
                    &pUrb->m_IoVa,
                    pUrb->m_ProcHandle, pUrb->m_ProcVa,
                    Util::RoundupDmaBufferSize(pUrb->m_BufferSize),
                    pUrb->BufferAlignment,
                    nn::dd::MemoryPermission_WriteOnly,
                    nn::dd::MemoryPermission_WriteOnly
                );
            }
        }
    }
    else
    {
        detail::SmmuSpace& smmuSpace = *pUrb->m_pSmmuSpace;
        pUrb->m_IoVa = smmuSpace.ioVa + (pUrb->m_ProcVa - smmuSpace.procVa);
    }

    if (result.IsFailure())
    {
        FreeUrb(pUrb);
    }
    else
    {
        *ppCreatedUrb = pUrb;
    }

    return result;
}

// internal use, hub intr xfer
Result UrbPool::AllocateUrb(UsbRequestBlock         **ppCreatedUrb,
                            nn::dd::ProcessHandle     procHandle,
                            uint64_t                  procVa,
                            uint32_t                  bufferSize,
                            bool                      isSmmuMapDynamic,
                            UsbBusToken               token,
                            UsbRequestBlock::Callback callback,
                            void                     *context,
                            uint32_t                  timeoutInMs)
{
    Result result = ResultSuccess();
    UsbRequestBlock *pNewUrb = nullptr;
    nn::dd::DeviceVirtualAddress  ioVa = 0;

    // allocate the URB
    pNewUrb = AllocateUrb();
    if (pNewUrb == nullptr)
    {
        return ResultUrbAllocFailure();
    }

    // sMMU Map
    if (bufferSize > 0)
    {
        if (isSmmuMapDynamic)
        {
            if (token == UsbBusToken_Out)
            {
                result = m_pPlatform->DoSmmu().Map(
                    &ioVa,
                    procHandle, procVa, Util::RoundupDmaBufferSize(bufferSize),
                    nn::dd::MemoryPermission_ReadOnly,
                    nn::dd::MemoryPermission_ReadOnly
                );
            }
            else
            {
                result = m_pPlatform->DoSmmu().Map(
                    &ioVa,
                    procHandle, procVa, Util::RoundupDmaBufferSize(bufferSize),
                    nn::dd::MemoryPermission_WriteOnly,
                    nn::dd::MemoryPermission_WriteOnly
                );
            }
        }
        else
        {
            ioVa = m_pPlatform->DoSmmu().GetIoVa(procHandle, procVa);
            if(ioVa == 0)
            {
                result = ResultIoVaError();
            }

        }
    }

    if (result.IsFailure())
    {
        FreeUrb(pNewUrb);
        return result;
    }

    pNewUrb->m_ProcHandle           = procHandle;
    pNewUrb->m_pSmmuSpace           = nullptr;
    pNewUrb->m_ProcVa               = procVa;
    pNewUrb->m_IoVa                 = ioVa;
    pNewUrb->m_BufferSize           = bufferSize;
    pNewUrb->m_Xfer[0].size         = bufferSize;
    pNewUrb->m_Xfer[0].xferredSize  = 0;
    pNewUrb->m_Xfer[0].result       = ResultSuccess();
    pNewUrb->m_XferCount            = 1;
    pNewUrb->m_IsSmmuMapDynamic     = isSmmuMapDynamic;
    pNewUrb->m_Token                = token;
    pNewUrb->m_pDeferredRequest     = nullptr;
    pNewUrb->m_Callback             = callback;
    pNewUrb->m_Context              = context;
    pNewUrb->m_TimeoutInMs          = timeoutInMs;

    *ppCreatedUrb = pNewUrb;

    return ResultSuccess();
}

// internal use, ctrl xfer
Result UrbPool::AllocateUrb(UsbRequestBlock         **ppCreatedUrb,
                            nn::dd::ProcessHandle     procHandle,
                            uint64_t                  procVa,
                            UsbCtrlRequest&           ctrlRequest,
                            bool                      isSmmuMapDynamic,
                            UsbRequestBlock::Callback callback,
                            void                     *context,
                            uint32_t                  timeoutInMs)
{
    Result result = ResultSuccess();
    UsbBusToken      token;
    UsbRequestBlock *pNewUrb = nullptr;

    if (ctrlRequest.wLength == 0)
    {
        token = UsbBusToken_Invalid;
    }
    else if (ctrlRequest.bmRequestType & UsbCtrlXferReqTypeMask_DirDeviceToHost)
    {
        token = UsbBusToken_In;
    }
    else
    {
        token = UsbBusToken_Out;
    }

    result = AllocateUrb(
        &pNewUrb,
        procHandle, procVa, ctrlRequest.wLength, isSmmuMapDynamic,
        token, callback, context, timeoutInMs
    );

    if (result.IsSuccess())
    {
        pNewUrb->m_CtrlRequest = ctrlRequest;
        *ppCreatedUrb = pNewUrb;
    }

    return result;
}

// Allocated -> Free
void UrbPool::FreeUrb(UsbRequestBlock *pUrb)
{
    NN_USB_ABORT_UNLESS(pUrb->m_State == UsbRequestBlock::State_Allocated);

    // remove from the allocated list
    m_UrbAllocatedList.erase(m_UrbAllocatedList.iterator_to(*pUrb));

    pUrb->m_State = UsbRequestBlock::State_Free;

    // sMMU unmap
    if (pUrb->m_IsSmmuMapDynamic && pUrb->m_IoVa)
    {
        NN_USB_ABORT_UNLESS_SUCCESS(
            m_pPlatform->DoSmmu().UnmapByIoVa(pUrb->m_IoVa)
        );
        pUrb->m_IoVa = 0;
    }

    // add to the free list
    m_UrbFreeList.push_back(*pUrb);
}

// Allocated -> Pending
void UrbPool::OnUrbSubmission(UsbRequestBlock *pUrb)
{
    NN_USB_ABORT_UNLESS(pUrb->m_State == UsbRequestBlock::State_Allocated);

    // remove from the allocated list
    m_UrbAllocatedList.erase(m_UrbAllocatedList.iterator_to(*pUrb));

    pUrb->m_State = UsbRequestBlock::State_Pending;

    // add to the pending list
    m_UrbPendingList.push_back(*pUrb);
}

// Pending -> Free
void UrbPool::OnUrbCompletion(UsbRequestBlock *pUrb)
{
    NN_USB_ABORT_UNLESS(pUrb->m_State == UsbRequestBlock::State_Pending);

    // remove from the pending list
    m_UrbPendingList.erase(m_UrbPendingList.iterator_to(*pUrb));

    pUrb->m_State = UsbRequestBlock::State_Free;

    // Unmap sMMU when necessary
    if (pUrb->m_IsSmmuMapDynamic && pUrb->m_IoVa)
    {
        NN_USB_ABORT_UNLESS_SUCCESS(
            m_pPlatform->DoSmmu().UnmapByIoVa(pUrb->m_IoVa)
        );
        pUrb->m_IoVa = 0;
    }

    // add to the free list
    m_UrbFreeList.push_back(*pUrb);
}

void UrbPool::CompleteAll()
{
    while (!m_UrbPendingList.empty())
    {
        m_UrbPendingList.front().Complete();
    }
}

void UrbPool::StartTimedEvent(LocalEventType *pEvent, uint32_t timeoutInMs)
{
    m_pHs->m_LocalEventManager.StartTimedEvent(
        pEvent,
        nn::TimeSpan::FromMilliSeconds(timeoutInMs)
    );
}

void UrbPool::StopTimedEvent(LocalEventType *pEvent)
{
    m_pHs->m_LocalEventManager.StopTimedEvent(pEvent);
}

void UrbPool::DeleteAllUrbs()
{
    UrbListType* urbLists[] = {
        &m_UrbFreeList, &m_UrbAllocatedList, &m_UrbPendingList
    };

    for (UrbListType *pList : urbLists)
    {
        while (!pList->empty())
        {
            UsbRequestBlock *pUrb = &pList->front();

            if (pUrb->m_IsSmmuMapDynamic && pUrb->m_IoVa)
            {
                NN_USB_ABORT_UNLESS_SUCCESS(
                    m_pPlatform->DoSmmu().UnmapByIoVa(pUrb->m_IoVa)
                );

                pUrb->m_IoVa = 0;
            }

            pList->pop_front();
            delete pUrb;
        }
    }
}

void UrbPool::TimerCallbackStatic(void *pContext, LocalEventType *pEvent)
{
    UrbPool *self = reinterpret_cast<UrbPool*>(pContext);
    self->m_pHcEp->TimerCallback(&pEvent->data);
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn


