﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief USB Controller Abstraction
 *
 * @details
 *
 *
 */

#pragma once

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {


class UsbComplex;


// capabilities mask for device controller
class PlatformControllerTegra30 : public PlatformController
{
public:
    PlatformControllerTegra30() NN_NOEXCEPT
        : PlatformController(UsbCapability_SuperSpeed)
        , m_pSuperSpeedRootHub(NULL)
        , m_pHighSpeedRootHub(NULL)
        , m_pHostControllerDriver(NULL)
        , m_pComplex(NULL)
        , m_HighSpeedPortCount(4)
        , m_SuperSpeedPortCount(4)
        , m_IsRootPortPowerControlled(false)
        , m_RootPortPowerRefCount(0)
        , m_ControllerRefCount(0)
        , m_pFirmwareImage(nullptr)
        , m_FirmwareSize(0)
    {
        m_SmiIrqHandler = [&] {
            nn::os::ClearInterruptEvent(&m_SmiIrqEvent);
            SmiIrqHandler();
        };
        m_HostIrqHandler = [&] {
            nn::os::ClearInterruptEvent(&m_HostIrqEvent);
            HostIrqHandler();
        };
    }

    ~PlatformControllerTegra30() NN_NOEXCEPT
    {
        // do nothing
    }

    struct FirmwareConfigTable
    {
        uint32_t boot_loadaddr_in_imem;
        uint32_t boot_codedfi_offset;
        uint32_t boot_codetag;
        uint32_t boot_codesize;

        /* Physical memory reserved by Bootloader/BIOS */
        uint32_t phys_memaddr;
        uint16_t reqphys_memsize;
        uint16_t alloc_phys_memsize;

        /* .rodata section */
        uint32_t rodata_img_offset;
        uint32_t rodata_section_start;
        uint32_t rodata_section_end;
        uint32_t main_fnaddr;

        uint32_t fwimg_cksum;
        uint32_t fwimg_created_time;

        /* Fields that get filled by linker during linking phase
         * or initialized in the FW code.
         */
        uint32_t imem_resident_start;
        uint32_t imem_resident_end;
        uint32_t idirect_start;
        uint32_t idirect_end;
        uint32_t l2_imem_start;
        uint32_t l2_imem_end;
        uint32_t version_id;
        uint8_t init_ddirect;
        uint8_t reserved[3];
        uint32_t phys_addr_log_buffer;
        uint32_t total_log_entries;
        uint32_t dequeue_ptr;

        /*	Below two dummy variables are used to replace
         *	L2IMemSymTabOffsetInDFI and L2IMemSymTabSize in order to
         *	retain the size of struct _CFG_TBL used by other AP/Module.
         */
        uint32_t dummy_var1;
        uint32_t dummy_var2;

        /* fwimg_len */
        uint32_t fwimg_len;
        uint8_t magic[8];
        uint32_t SS_low_power_entry_timeout;
        uint8_t num_hsic_port;
        uint8_t ss_portmap;
        uint8_t build_log:4;
        uint8_t build_type:4;
        uint8_t padding[137]; /* padding bytes to makeup 256-bytes cfgtbl */
    }NN_USB_PACKED_STRUCT_ATTRIBUTE;

    /* Root hub termination options used upon controller disable
     */
    static const Device::TerminationOptions RootHubTerminationOptions = Device::TerminationOptions_Disconnected;

    /**
     * @brief XUSB Host PCI Config Registers
     *
     */
    static const size_t FpciBaseOffset = 0x8000;
    NN_USB_DEFINE_REG_OFFSET(XusbCfg0,                   FpciBaseOffset + 0x00);
    static const uint16_t XusbCfg0Tegra21xDevId = 0x0fac;
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfg1, FpciBaseOffset + 0x04);
    NN_USB_DEFINE_FIELD32(XusbCfg1, BusMaster, 2, 1);
    NN_USB_DEFINE_FIELD32(XusbCfg1, MemorySpace, 1, 1);
    NN_USB_DEFINE_FIELD32(XusbCfg1, IoSpace, 1, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfg4,                   FpciBaseOffset + 0x10);
    NN_USB_DEFINE_FIELD32(XusbCfg4, BaseAddress, 16, 16);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfg16,                  FpciBaseOffset + 0x40);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfg24,                  FpciBaseOffset + 0x60);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruMboxCmd,          FpciBaseOffset + 0xE4);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxCmd, FalconIntEn, 27, 1);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxCmd, PmeIntEn,    28, 1);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxCmd, SmiIntEn,    29, 1);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxCmd, XhciIntEn,   30, 1);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxCmd, IntEn,       31, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruMboxDataIn,       FpciBaseOffset + 0xE8);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxDataIn,  Data, 0,  24);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxDataIn,  Type, 24,  8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruMboxDataOut,      FpciBaseOffset + 0xEC);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxDataOut, Data, 0,  24);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxDataOut, Type, 24,  8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruMboxOwner,        FpciBaseOffset + 0xf0);
    NN_USB_DEFINE_FIELD32(XusbCfgAruMboxOwner, Id, 0,  8);
    static const uint8_t XusbCfgAruMboxOwnerIdFirmware = 1;
    static const uint8_t XusbCfgAruMboxOwnerIdSoftware = 2;
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgFpciCfg,             FpciBaseOffset + 0xf8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruC11Csbrange,      FpciBaseOffset + 0x41c);
    NN_USB_DEFINE_FIELD32(XusbCfgAruC11Csbrange, PageSelect, 9, 23);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruSmiIntr,          FpciBaseOffset + 0x428);
    NN_USB_DEFINE_FIELD32(XusbCfgAruSmiIntr, Mbox,     3, 1);
    NN_USB_DEFINE_FIELD32(XusbCfgAruSmiIntr, FwReinit, 1, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruRst,              FpciBaseOffset + 0x42c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruContext,          FpciBaseOffset + 0x43c);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruFwScratch,        FpciBaseOffset + 0x440);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruContextHsPls,     FpciBaseOffset + 0x478);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgAruContextHsfsSpeed, FpciBaseOffset + 0x480);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCfgCsbBaseAddr,         FpciBaseOffset + 0x800);
    NN_USB_DEFINE_FIELD32(XusbCfgCsbBaseAddr, PageOffset, 0, 9);

    /**
     * @brief XUSB Host IPFS Registers
     *
     */
    static const size_t IpfsBaseOffset = 0x9000;
    NN_USB_DEFINE_REG_OFFSET(XusbHostAxiBar0Sz,         IpfsBaseOffset + 0x00);
    NN_USB_DEFINE_REG_OFFSET(XusbHostConfiguration,     IpfsBaseOffset + 0x180);
    NN_USB_DEFINE_FIELD32(XusbHostConfiguration,        EnFpci, 0, 1);
    NN_USB_DEFINE_REG_OFFSET(XusbHostFpciErrorMasks,    IpfsBaseOffset + 0x184);
    NN_USB_DEFINE_REG_OFFSET(XusbHostIntrMask,          IpfsBaseOffset + 0x188);
    NN_USB_DEFINE_FIELD32(XusbHostIntrMask,             IpInt, 16, 1);
    NN_USB_DEFINE_FIELD32(XusbHostIntrMask,             Msi, 8, 1);
    NN_USB_DEFINE_FIELD32(XusbHostIntrMask,             Int, 0, 1);
    NN_USB_DEFINE_REG_OFFSET(XusbHostClkgateHysteresis, IpfsBaseOffset + 0x1bc);

    /**
     * @brief Raptor CSB MP Registers (indirect via XusbCfgAruC11Csbrange)
     *
     */
    NN_USB_DEFINE_REG_OFFSET(XusbFalcCpuCtl,                0x00000100);
    NN_USB_DEFINE_FIELD32(XusbFalcCpuCtl, StartCpu, 1, 1);
    NN_USB_DEFINE_FIELD32(XusbFalcCpuCtl, Sreset,   2, 1);
    NN_USB_DEFINE_FIELD32(XusbFalcCpuCtl, Hreset,   3, 1);
    NN_USB_DEFINE_FIELD32(XusbFalcCpuCtl, Halted,   4, 1);
    NN_USB_DEFINE_FIELD32(XusbFalcCpuCtl, Stopped,  5, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbFalcBootvec,               0x00000104);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcDmactl,                0x0000010C);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbFalcImfillRng1,            0x00000154);
    NN_USB_DEFINE_FIELD32(XusbFalcImfillRng1, TagLo, 0,  16);
    NN_USB_DEFINE_FIELD32(XusbFalcImfillRng1, TagHi, 16, 16);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbFalcImfillCtl,             0x00000158);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcCmembase,              0x00000160);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcDmemapert,             0x00000164);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcImemcStart,            0x00000180);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcImemdStart,            0x00000184);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcImemtStart,            0x00000188);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcIcdCmd,                0x00000200);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcIcdRdata,              0x0000020C);
    NN_USB_DEFINE_REG_OFFSET(XusbCsbAruScratch0,            0x00100100);
    NN_USB_DEFINE_REG_OFFSET(XusbCsbRstSspi,                0x00100408);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpApmap,                0x0010181C);
    NN_USB_DEFINE_FIELD32(XusbCsbMpApmap, Bootpath, 31, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpIloadAttr,            0x00101A00);
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpIloadBaseLo,          0x00101A04);
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpIloadBaseHi,          0x00101A08);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpL2imemopSize,         0x00101A10);
    const size_t ImemBlockSize = 256;
    NN_USB_DEFINE_FIELD32(XusbCsbMpL2imemopSize, SrcOffset, 8,  10);
    NN_USB_DEFINE_FIELD32(XusbCsbMpL2imemopSize, SrcCount,  24, 8);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMpL2imemopTrig,         0x00101A14);
    const uint32_t XusbCsbMpL2imemopTrigLoadLockedResultMask = 0x11 << 24;
    NN_USB_DEFINE_FIELD32(XusbCsbMpL2imemopTrig, InvalidateAll, 30, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbCsbMempoolL2imemopResult,  0x00101A18);
    NN_USB_DEFINE_FIELD32(XusbCsbMempoolL2imemopResult, Valid, 31, 1);
    //
    NN_USB_DEFINE_REG_OFFSET(XusbFalcSsPvtportsc1,          0x00116000);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcSsPvtportsc2,          0x00116004);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcSsPvtportsc3,          0x00116008);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcHsPvtportsc1,          0x00116800);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcHsPvtportsc2,          0x00116804);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcHsPvtportsc3,          0x00116808);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcFsPvtportsc1,          0x00117000);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcFsPvtportsc2,          0x00117004);
    NN_USB_DEFINE_REG_OFFSET(XusbFalcFsPvtportsc3,          0x00117008);


    // Base class UsbController methods
    virtual Result Initialize(detail::UsbComplex *pComplex,
                              detail::UsbController::Config* pConfig) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Finalize() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Enable() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Disable() NN_NOEXCEPT NN_OVERRIDE;
    virtual uintptr_t GetBaseAddress() NN_NOEXCEPT NN_OVERRIDE;
    virtual detail::RegisterBlock* GetBaseRegister() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetTestMode(uint32_t port, TestMode mode, int32_t offset,
                               uint32_t *pOutStrength) NN_NOEXCEPT NN_OVERRIDE;

    // hs::PlatformController methods
    Result Bind(Hs* pHs) NN_NOEXCEPT;
    Result Unbind()  NN_NOEXCEPT;

private:
    class RootHubTerminationCallbackContext
    {
    public:
    RootHubTerminationCallbackContext() : m_Status(ResultSuccess()), m_Semaphore(0,1) {}
        Result            m_Status;
        nn::os::Semaphore m_Semaphore;
    };

    static const os::InterruptName Usb3HostSmiInterruptName = 32 + 40;
    static const os::InterruptName Usb3HostInterruptName = 32 + 39;

    nn::dd::DeviceVirtualAddress m_FalconFirmwareIoVa;

    enum FalconMailboxCommand
    {
        FalconMailboxCommand_Invalid,
        FalconMailboxCommand_Enabled,
        FalconMailboxCommand_IncFalcClock,
        FalconMailboxCommand_DecFalcClock,
        FalconMailboxCommand_IncSspiClock,
        FalconMailboxCommand_DecSspiClock,
        FalconMailboxCommand_SetBw,
        FalconMailboxCommand_SetSsPwrGating,
        FalconMailboxCommand_SetSsPwrUngating,
        FalconMailboxCommand_SaveDfeCtleCtx,
        FalconMailboxCommand_EnableAirplaneMode,
        FalconMailboxCommand_DisableAirplaneMode,
        FalconMailboxCommand_StartHsicIdle,
        FalconMailboxCommand_StopHsicIdle,
        FalconMailboxCommand_DbcWakeStack,
        FalconMailboxCommand_HsicPretendConnect,
        FalconMailboxCommand_ResetSspi,
        FalconMailboxCommand_DisableSsLpfsDetection,
        FalconMailboxCommand_EnableSsLpfsDetection,
        FalconMailboxCommand_Max,
        //
        FalconMailboxCommand_Ack  = 128,
        FalconMailboxCommand_Nack,
    };


    HubDevice*                            m_pSuperSpeedRootHub;
    HubDevice*                            m_pHighSpeedRootHub;
    HostControllerDriver*                 m_pHostControllerDriver;
    detail::UsbComplexTegra21x*           m_pComplex;
    const uint8_t                         m_HighSpeedPortCount;
    const uint8_t                         m_SuperSpeedPortCount;
    bool                                  m_IsRootPortPowerControlled;
    int                                   m_RootPortPowerRefCount;
    int                                   m_ControllerRefCount;
    uint16_t                              m_DeviceId;
    detail::RegisterBlock                 CapRegister;

    // SMI Interrupt
    detail::IrqHandlerType                m_SmiIrqHandler;
    nn::os::MultiWaitHolderType           m_SmiIrqHolder;
    nn::os::InterruptEventType            m_SmiIrqEvent;

    // Host Interrupt
    detail::IrqHandlerType                m_HostIrqHandler;
    nn::os::MultiWaitHolderType           m_HostIrqHolder;
    nn::os::InterruptEventType            m_HostIrqEvent;

    // Falcon
    uint8_t                              *m_pFirmwareImage;
    size_t                                m_FirmwareSize;

    static void RootHubTerminationCallback(Device *pDevice, Result status, void *context);

    Result EnableXusb();
    Result DisableXusb();

    // Falcon control
    uint32_t CsbRead (uint32_t address);
    void     CsbWrite(uint32_t address, uint32_t data);
    Result   FalconLoadFirmware(int32_t ssPadCount, uint8_t ssPortMap,
                                uint8_t hsicPortCount, bool isResetAru);
    Result   FalconSendMessageAsync(FalconMailboxCommand command, uint32_t data);
    void     FalconHandleIrq();

    // IRQ Handler
    void SmiIrqHandler();
    void HostIrqHandler();

    // Root hub power control
    int SetRootHubPortPower(HubPortNumber hubPortNumber, bool isPowerOn) NN_NOEXCEPT NN_OVERRIDE;

    // GPIO helper
    void SetGpioOutputValue(nn::gpio::GpioPadName gpioPadName, bool isHigh) NN_NOEXCEPT;
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

