﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {
class HubDevice : public Device
{
public:
    class Port;
    friend Port;
    static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;



    HubDevice(Hs *pHs, HostControllerDriverDeviceContext* pHcDevContext, Device *pParent, HubPortNumber hubPortNumber,
              UsbDeviceSpeed speed, UsbDeviceDescriptor *pDevDesc, StartupMode startupMode) :
    Device(pHs, pHcDevContext, pParent, hubPortNumber, speed, pDevDesc, startupMode)
    {
        m_PortCount = 0;
        memset(&m_Ports, 0, sizeof(m_Ports));
        memset(&m_HubDescriptor, 0, sizeof(m_HubDescriptor));
        m_IsRootHub = false;
        m_IsRootHubPortEventConfigured = false;
        m_pIntHep = nullptr;
        m_pDmaBuffers = nullptr;
    }
    ~HubDevice()
    {
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

    Result InitializeRootHub(const UsbHubDescriptor* pDescriptor, HostControllerDriver *pHostControllerDriver, HubPortNumber basePortNumber);
    Result InitializeRootHub(const UsbSsHubDescriptor* pDescriptor, HostControllerDriver *pHostControllerDriver, HubPortNumber basePortNumber);

    // stop and finalize
    Result Finalize();

    int GetObjectName(char *pObjName, uint32_t maxSize);
    static HubDevice::Port* GetParentPort(Device *pChildDevice);

private:
    union DescriptorBuffers
    {
        UsbHubDescriptor    hub;
        UsbHubDescriptor    ssHub;
        UsbDeviceDescriptor device;
    };
    struct HubEventBuffer
    {
        uint8_t data[NN_USB_HUB_EVENT_XFER_SIZE(HsLimitMaxHubPortsCount)];
    };
    struct DmaBuffers
    {
        nn::util::TypedStorage<DescriptorBuffers, sizeof(DescriptorBuffers),
                               HwLimitDataCacheLineSize> descriptorBuffers[HsLimitMaxHubPortsCount];
        nn::util::TypedStorage<HubEventBuffer,    sizeof(HubEventBuffer),
                               HwLimitDataCacheLineSize>   hubEventBuffer;
        nn::util::TypedStorage<UsbHubPortStatus,  sizeof(UsbHubPortStatus),
                               HwLimitDataCacheLineSize>   portStatusBuffers[HsLimitMaxHubPortsCount];
    };

    HubPortNumber            m_RootBasePortNumber;
    int32_t                  m_PortCount;
    bool                     m_IsSuperSpeedHub;
    Port                     *m_Ports[HsLimitMaxHubPortsCount];
    bool                     m_IsRootHub;
    bool                     m_IsRootHubPortEventConfigured;
    bool                     m_IsPortIndicationArmed;
    DmaBuffers               *m_pDmaBuffers;
    HostEndpoint             *m_pIntHep;
    char                     m_ObjectName[HsLimitMaxDebugNameSize];
    UsbHubDescriptorUnion    m_HubDescriptor;

    Result StartHub();
    Result ArmPortIndicationInterrupt();
    void HandlePortEvent(void *pBitmap, uint32_t bitmapSize);
    void HandleRootHubPortEvent(uint32_t portChangeMask);
    void HandleExternalHubPortEvent(UsbRequestBlock *pUrb);
    static void StaticRootHubPortEventCallback(void *context, uint32_t portChangeMask)
    {
        reinterpret_cast<HubDevice *>(context)->HandleRootHubPortEvent(portChangeMask);
    }
    static void StaticExternalHubPortEventCallback(Hs *pHs, UsbRequestBlock *pUrb)
    {
        NN_UNUSED(pHs);
        reinterpret_cast<HubDevice *>(pUrb->m_Context)->HandleExternalHubPortEvent(pUrb);
    }
    bool IsCreatingDevice();
    bool IsRootHubCreatingDevice();
    HubDevice::Port* GrantExclusivity();
    void HandlePortTerminationDone(Port *pPort);
    void ReEvaluateEnumerationExclusivity();

    // Implemented device FSM handlers
    Result DeviceSpecificInitStateHandler(LocalEventDataType *pEvent);
    Result DeviceSpecificFinalizeStateHandler(LocalEventDataType *pEvent);
};


class HubDevice::Port : public Fsm
{
public:
    friend HubDevice;
    enum AdministrativeCommand
    {
        AdministrativeCommand_None = 0,
        AdministrativeCommand_Freeze,
        AdministrativeCommand_Reset,
        AdministrativeCommand_Suspend,
        AdministrativeCommand_Resume,
        AdministrativeCommand_Activate
    };

    Port(Hs *pHs, HubDevice *pHub, HubPortNumber portNumber)
        : Fsm(pHs, LogModule_HsFsmHub)
        , m_pHs(pHs)
        , m_pHub(pHub)
        , m_PortNumber(portNumber)
        , m_PortSpeed(UsbDeviceSpeed_Invalid)
        , m_pChildDevice(nullptr)
        , m_isChildDeviceAddressingDone(false)
        , m_pHc(pHub->m_pHc)
        , m_HcCtrlMgr(pHs)
        , m_pHcDevContext(nullptr)
    {
        m_pDmaPortStatus = reinterpret_cast<UsbHubPortStatus *>
            (&pHub->m_pDmaBuffers->portStatusBuffers[portNumber - 1]);
        m_pDmaDescriptorBuffers = reinterpret_cast<DescriptorBuffers *>
            (&pHub->m_pDmaBuffers->descriptorBuffers[portNumber - 1]);
        memset(&m_CtrlHep, 0, sizeof(m_CtrlHep));
        m_HcPortNumber = 0;
    }
    ~Port()
    {
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    Result Initialize();
    void Terminate();
    void Finalize();
    void GetObjectName(char *pObjName, uint32_t maxSize);
    bool IsTerminated();
    bool IsCreatingDevice();
    bool IsWaitingForExclusivity();
    void GrantExclusivity();
    Result IssueAdministrativeCommand(AdministrativeCommand command);

private:
    enum StaticPortConfig
    {
        StaticPortConfig_MaxRetryNum                     = 1,
        StaticPortConfig_HubPortAttachDebounceInUs       = 100000,
        StaticPortConfig_ResetRootPortDelayInUs          = 100000,
        StaticPortConfig_ResetTimeoutInUs                = 4000000,
        StaticPortConfug_ResetRecoveryInUs               = 50000,
        StaticPortConfug_MaxDevInitRetryCount            = 3,
    };
    enum State
    {
        State_Null = 0,
        State_Init,
        State_PendingSelectiveEnum,
        State_PoweringOn,
        State_ClearingAll,
        State_NotConnected,
        State_ConnectionDebounce,
        State_WaitingForExclusivity,
        State_ResettingPort,
        State_ResetRecovery,
        State_InitializeHostControllerDeviceContext,
        State_ReadDeviceDescriptor,
        State_CreatingChildDevice,
        State_ChildDeviceActive,
        State_DestroyingChildDevice,
        State_TerminatingPort,
        State_Maximum
    };
    enum Event
    {
        Event_Try = Fsm::Event_Base,
        Event_ReTry,
        Event_Skip,
        Event_PrepareExit,
        Event_Done,
        Event_EndpointInitializeComplete,
        Event_EndpointTerminateComplete,
        Event_ControlTransferComplete,
        Event_Ready,
        Event_Exclusive,
        Event_PortStatus,
        Event_PortFeatureOpDone,
        Event_PortIndication,
        Event_Error,
        Event_Reset,
        Event_Freeze,
        Event_Suspend,
        Event_Resume,
        Event_Activate,
        Event_TerminateRequest,
        Event_ChildDeviceCreateComplete,
        Event_ChildDeviceCreateFail,
        Event_Timeout,
        Event_ChildDeviceTerminateComplete,
        Event_HostControllerDeviceContextCreate,
        Event_HostControllerDeviceContextDestroy,
        Event_ReadMaxPacketSize0,
        Event_UpdateEp0,
        Event_ReadDeviceDescriptor,
        Event_Maximum
    };
    static const char *EventNames[Port::Event_Maximum];
    static const char *StateNames[Port::State_Maximum];


    Hs                              *m_pHs;
    HubDevice                       *m_pHub;
    HubPortNumber                   m_PortNumber;
    HubPortNumber                   m_HcPortNumber;
    UsbDeviceSpeed                  m_PortSpeed;
    Device                          *m_pChildDevice;
    bool                            m_isChildDeviceAddressingDone;
    HostControllerDriver            *m_pHc;
    ControlTransferManager          m_HcCtrlMgr;
    HostControllerDriverDeviceContext* m_pHcDevContext;
    UsbHubPortStatus                *m_pDmaPortStatus;
    DescriptorBuffers               *m_pDmaDescriptorBuffers;
    HostEndpoint                    m_CtrlHep;
    LocalEventType                  m_TimedReadyEvent;
    LocalEventType                  m_TimeoutEvent;
    LocalEventType                  m_EpTerminationRetryTimer;
    char                            m_ObjectName[HsLimitMaxDebugNameSize];
    int32_t                         m_Retries;
    bool                            m_IsPortStatusQueryPending;
    UsbHubPortStatus                m_PortStatus;
    AdministrativeCommand           m_PendingAdminCommand;
    int32_t                         m_SuccessiveResetAttempts;
    int32_t                         m_SuccessiveDevInitAttempts;
    Device::TerminationOptionMask   m_DeviceTerminationCause;
    bool                            m_TerminationPending;
    int32_t                         m_PendingFeatOptCount;

    void SetFeatureAsync(uint8_t portFeat);
    void ClearFeatureAsync(uint8_t portFeat);
    Result QueryPortStatusAsync();
    UsbDeviceSpeed GetPortSpeed();
    Result CreateChildDevice(UsbDeviceDescriptor *pDevDesc);


    static void ChildDeviceInitializeProgressCallback(InitializeProgress progress, void *context);
    static void ChildDeviceTerminateDoneCallback(Device *pDevice, Result status, void *context);
    static void HostControllerDriverEndpointInitializeDoneCallback(Hs* pHs, Result status,
                                                                   HostControllerDriverEndpoint* pEp, void* context);
    static void HostControllerDriverEndpointTerminateDoneCallback(Hs *pHs, Result status,
                                                                  HostControllerDriverEndpoint *pEp, void *context);
    static void ReadMaxPacketSize0CompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize);
    static void UpdateEp0CompletionCallback(Hs*, Result result, void *context);
    static void ControlTransferCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize);
    static void PortFeatureCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize);
    static void PortStatusCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize);
    static void CreateHostControllerDeviceContextCallback(Hs*, Result result, HostControllerDriverDeviceContext*, void* context);
    static void DestroyHostControllerDeviceContextCallback(Hs*, Result result, void* context);

    Result FsmHandler(int32_t state, LocalEventDataType *pEvent);
    Result InitStateHandler(LocalEventDataType *pEvent);
    Result PendingSelectiveEnumStateHandler(LocalEventDataType *pEvent);
    Result PoweringOnStateHandler(LocalEventDataType *pEvent);
    Result ClearingAllStateHandler(LocalEventDataType *pEvent);
    Result NotConnectedStateHandler(LocalEventDataType *pEvent);
    Result ConnectionDebounceStateHandler(LocalEventDataType *pEvent);
    Result WaitingForExclusivityStateHandler(LocalEventDataType *pEvent);
    Result ResettingPortStateHandler(LocalEventDataType *pEvent);
    Result ResetRecoveryStateHandler(LocalEventDataType *pEvent);
    Result InitializeHostControllerDeviceContextStateHandler(LocalEventDataType *pEvent);
    Result ReadDeviceDescriptorStateHandler(LocalEventDataType *pEvent);
    Result CreatingChildDeviceStateHandler(LocalEventDataType *pEvent);
    Result ChildDeviceActiveStateHandler(LocalEventDataType *pEvent);
    Result DestroyingChildDeviceStateHandler(LocalEventDataType *pEvent);
    Result TerminatingPortStateHandler(LocalEventDataType *pEvent);
};



} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

