﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

#include <nn/usb/usb_HostTypes.h>

namespace nn { namespace usb { namespace hs {

class Device;

struct DeviceInterface
{
    int32_t                          ifIndex;
    UsbInterfaceDescriptor           *altSettingArray[HsLimitMaxAltInterfaceSettingsCount];
    int32_t                          altSettingCount;
    UsbInterfaceDescriptor           *pActAltSetting;
    Device                           *pDevice;
    DeviceManager::ManagedInterface  *pManagedInterface;
};

class Device : public Fsm
{
public:
    static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;
    static const int32_t MaxUserEndpointPairCount = 15;
    typedef uint32_t TerminationOptionMask;
    enum TerminationOptions
    {
        TerminationOptions_Valid          =  0x01,
        TerminationOptions_Disconnected   =  0x02,
        TerminationOptions_Suspend        =  0x04,
        TerminationOptions_PowerOff       =  0x08,
    };
    enum InitializeProgress
    {
        InitializeProgress_Error,
        InitializeProgress_AddressAssigned,
        InitializeProgress_Complete,
    };
    typedef void (*InitializeCallback)(InitializeProgress progress, void *context);
    typedef void (*TerminateCompleteCallback)(Device *pDevice, Result status, void *context);

    Device(Hs *pHs, HostControllerDriverDeviceContext* pHcDevContext, Device *pParent,
           HubPortNumber hubPortNumber, UsbDeviceSpeed speed, UsbDeviceDescriptor *pDevDesc,
           StartupMode startupMode)
        : Fsm(pHs, LogModule_HsFsmDevice)
        , m_pHs(pHs)
        , m_pParent(pParent)
        , m_ParentPortNumber(hubPortNumber)
        , m_pHcDevContext(pHcDevContext)
        , m_pHc(nullptr)
        , m_pRootDevice(nullptr)
        , m_pPlatform(nullptr)
        , m_Speed(speed)
        , m_StartupMode(startupMode)
        , m_Level(0)
        , m_Address(0)
        , m_TerminationOptionMask(0)
        , m_CtrlTransferMgr(pHs)
        , m_ManagedDevice(pHs, this)
    {
        memset(&m_Ep0, 0, sizeof(m_Ep0));
        memset(&m_EpIn, 0, sizeof(m_EpIn));
        memset(&m_EpOut, 0, sizeof(m_EpOut));
        for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_PortHierarchy); i++)
        {
            m_PortHierarchy[i] = -1;
        }
        if (pDevDesc != nullptr)
        {
            m_DeviceDescriptor = *pDevDesc;
        }
        for (int32_t i = 0; i < MaxUserEndpointPairCount; i++)
        {
            MarkHostEndpointAsInvalid(m_EpIn + i);
            MarkHostEndpointAsInvalid(m_EpOut + i);
        }
        m_EpDestroyPendMask = 0;
        memset(m_CfgDescArray, 0, sizeof(m_CfgDescArray));
        m_pActCfg = nullptr;
        memset(m_IfArray, 0, sizeof(m_IfArray));
        for (int32_t i = 0; i < NN_USB_ARRAY_COUNT32(m_IfArray); i++)
        {
            m_IfArray[i].ifIndex = i;
            m_IfArray[i].pDevice = this;
        }
        m_pDevStringLangIds = nullptr;
        m_Substate = 0;
        m_EnumerationTimestamp = 0;
        m_InitializeCallback = nullptr;
        m_InitializeContext = nullptr;
        m_TerminateCompleteCallback = nullptr;
        m_TerminateCompleteContext = nullptr;
        m_pCtrlTransferBuffer = nullptr;
    }
    virtual ~Device()
    {
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    Result InitializeAsync(InitializeCallback callback, void *context);
    virtual Result Finalize();
    Result TerminateAsync(TerminationOptionMask options, TerminateCompleteCallback completeCbf, void *context);
    void GetDeviceDescriptor(UsbDeviceDescriptor *pReturnedDescriptor);
    void GetConfigDescriptor(UsbConfigDescriptor *pReturnedDescriptor);
    void   GetDeviceProfile(DeviceProfile *pReturnedProfile);
    Result GetInterfaceProfile(int32_t ifIndex, InterfaceProfile *pReturnedProfile);
    Result GetAlternateInterfaceProfile(int32_t ifIndex, uint8_t bAlternateSetting, InterfaceProfile *pReturnedProfile);
    Result SetInterface(int32_t ifIndex, uint8_t bAlternateSetting);
    void GetHubPortHierarchy(HubPortNumber *pHierarchy);
    int32_t GetHierarchicalLevel();
    HubPortNumber GetParentPort();
    Device* GetParentDevice();
    Device* GetRootDevice();
    virtual int GetObjectName(char *pDevName, uint32_t maxSize);
    HostControllerDriver* GetHostControllerDriver();
    HostControllerDriverDeviceContext* GetHostControllerDriverDeviceContext();
    bool IsAddressAssigned();
    uint8_t GetDeviceClass();
    StartupMode GetStartupMode();
    DeviceUid GetUid();
    UsbDeviceSpeed GetSpeed();
    uint8_t GetAddress();
    TerminationOptionMask GetTerminationOptions();
    bool FindHighSpeedParent(HubPortNumber *pRetPort, Device** ppRetDevice);
    Result CreateManagedHostEndpoint(int32_t ifIndex, UsbEndpointType epType, EndpointNumber epNumber, UsbEndpointDirection epDir,
                                     uint16_t maxUrbCount, uint32_t maxTransferSize, bool isToggleReset, void *adminContext);
    Result DestroyManagedHostEndpoint(HostEndpoint *pHep, void *adminContext);
    Result GetHostEndpoint(EndpointNumber epNumber, UsbEndpointDirection epDirection, HostEndpoint **ppHep);
    Result GetManagedHostEndpoint(EndpointNumber epNumber, UsbEndpointDirection epDirection, HostEndpoint **ppHep);
    Result GetControlTransferManager(ControlTransferManager **ppCtrlMgr);
    nn::TimeSpan GetEnumerationTimeStamp();
    Result Reset();

protected:
    enum State
    {
        State_Null = 0,
        State_Initialize,
        State_AssignAddress,
        State_IdentifyConfiguration,
        State_RetrieveStrings,
        State_SetConfiguration,
        State_WarmRecovery,
        State_DeviceSpecificInit,
        State_Configured,
        State_Suspended,
        State_DeviceSpecificFinalize,
        State_FinalizingSelf,
        State_Maximum
    };
    enum Event
    {
        Event_Try = Fsm::Event_Base,
        Event_Skip,
        Event_AddressComplete,
        Event_Continue,
        Event_DeviceSpecificComplete,
        Event_ControlTransferComplete,
        Event_SetAltIfComplete,
        Event_SetIfRequest,
        Event_SuspendRequest,
        Event_ResumeRequest,
        Event_EndpointRequest,
        Event_EndpointInitializeComplete,
        Event_EndpointTerminateComplete,
        Event_Error,
        Event_TerminateRequest,
        Event_RecoveryOk,
        Event_PowerManagementRequest,
        Event_Maximum
    };

    struct UsbEndpointDescriptors
    {
        UsbEndpointDescriptor          *pStandard;
        UsbEndpointCompanionDescriptor *pCompanion;
    };

    Hs*                                m_pHs;
    Device*                            m_pParent;
    HubPortNumber                      m_ParentPortNumber;
    HostControllerDriverDeviceContext* m_pHcDevContext;
    HostControllerDriver*              m_pHc;
    Device*                            m_pRootDevice;
    PlatformController*                m_pPlatform;
    UsbDeviceSpeed                     m_Speed;
    StartupMode                        m_StartupMode;
    int32_t                            m_Level;
    uint8_t                            m_Address;
    TerminationOptionMask              m_TerminationOptionMask;
    HubPortNumber                      m_PortHierarchy[HsLimitMaxHubHierarchyCount];
    ControlTransferManager             m_CtrlTransferMgr;
    int32_t                            m_Substate;
    nn::TimeSpan                       m_EnumerationTimestamp;

    // user endpoints
    HostEndpoint                       m_EpIn[MaxUserEndpointPairCount];
    HostEndpoint                       m_EpOut[MaxUserEndpointPairCount];

    // endpoint masks
    UsbEndpointMask                    m_EpDestroyPendMask;

    static void ControlTransferCompletionCallback(Hs *pHs, void *context, Result status,
                                                  uint32_t transferredSize);
    static void EndpointInitializeDoneCallback(Hs* pHs, Result status,
                                               HostControllerDriverEndpoint* pEp, void* context);
    static void EndpointTerminateDoneCallback(Hs *pHs, Result status,
                                              HostControllerDriverEndpoint *pEp, void *context);
    static void AddressDeviceDoneCallback(Hs*, Result, void*);
    void ReportInitializeProgress(InitializeProgress progress);

private:
    static const char           *EventNames[Event_Maximum];
    static const char           *StateNames[State_Maximum];
    static const uint32_t       ControlTransferBufferSize = 1024 * 64;
    static const int32_t        DefaultConfigurationIndex = 0;
    static const int32_t        DefaultAltIfIndex = 0;
    static const int32_t        MaxStringLanguageCount = 32;


    UsbDeviceDescriptor         m_DeviceDescriptor;
    InitializeCallback          m_InitializeCallback;
    void                        *m_InitializeContext;
    TerminateCompleteCallback   m_TerminateCompleteCallback;
    void                        *m_TerminateCompleteContext;
    char                        m_ObjectName[HsLimitMaxDebugNameSize];
    LocalEventType              m_EpTerminationRetryTimer;
    LocalEventType              m_SetAddrDelayEvent;

    // control endpoint
    HostEndpoint                m_Ep0;
    void                        *m_pCtrlTransferBuffer;

    // configurations
    UsbConfigDescriptor         *m_CfgDescArray[HsLimitMaxConfigurationsPerDeviceCount];
    UsbConfigDescriptor         *m_pActCfg;

    // strings
    UsbStringDescriptor         *m_pDevStringLangIds;

    // interfaces
    DeviceInterface              m_IfArray[HsLimitMaxInterfacesPerConfigurationCount];

    // device manager data
    DeviceManager::ManagedDevice m_ManagedDevice;

    void MarkHostEndpointAsInvalid(HostEndpoint *pHep);
    static void SetAltIfCompletionCallback(Hs *pHs, void *context, Result status,
                                           uint32_t transferredSize);
    Result CheckDescriptors(char *buffer, size_t length);
    Result ParseInterfaces(uint32_t cfgIndex);
    Result ParseEndpoints(UsbConfigDescriptor *pCfgDesc,
                          UsbInterfaceDescriptor *pIntDesc,
                          UsbEndpointDescriptors *epInDescriptors,
                          UsbEndpointDescriptors *epOutDescriptors);
    Result ParseAndTryEndpoints(int32_t ifIndex, uint8_t bAlternateSetting);
    Result ParseAndApplyEndpoints(int32_t ifIndex, uint8_t bAlternateSetting);
    DeviceInterface* GetDeviceInterface(int32_t ifIndex);

    // State handlers
    Result FsmHandler(int32_t state, LocalEventDataType *pEvent);
    Result InitializeStateHandler(LocalEventDataType *pEvent);
    Result AssignAddressStateHandler(LocalEventDataType *pEvent);
    Result IdentifyConfigurationStateHandler(LocalEventDataType *pEvent);
    Result RetrieveStringsStateHandler(LocalEventDataType *pEvent);
    Result SetConfigurationStateHandler(LocalEventDataType *pEvent);
    Result WarmRecoveryStateHandler(LocalEventDataType *pEvent);
    virtual Result DeviceSpecificInitStateHandler(LocalEventDataType *pEvent);
    Result ConfiguredStateHandler(LocalEventDataType *pEvent);
    Result SuspendedStateHandler(LocalEventDataType *pEvent);
    virtual Result DeviceSpecificFinalizeStateHandler(LocalEventDataType *pEvent);
    Result FinalizingSelfStateHandler(LocalEventDataType *pEvent);
};



} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

