﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Host Controller base class
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class ControlTransferManager : public Fsm
{
public:
    static const uint32_t ObjectMemAlignmentSize = HwLimitDmaBufferAlignmentSize;
    typedef void (*CompletionCallback)(Hs *pHs, void *context, Result status, uint32_t transferredSize);

    explicit ControlTransferManager(Hs *pHs)
        : Fsm(pHs, LogModule_HsFsmCtrlXfer)
        , m_pHs(pHs)
        , m_pHcEp(nullptr)
        , m_pCurrentTxn(nullptr)
    {
    }
    ~ControlTransferManager()
    {
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    void Initialize(HostControllerDriverEndpoint *pHcEp, const char *objectName);
    void Finalize();
    Result SubmitAsync(uint8_t bRequest, uint8_t bmRequestType, uint16_t wValue,
                       uint16_t wIndex, uint16_t wLength,
                       void *pData, bool isSmmuMapDynamic,
                       CompletionCallback callback,
                       void *context, uint32_t timeoutInMs);
    Result SubmitAsync(ClientIfSession::CtrlXferDeferredRequest *pDefReq);

private:
    class Transaction
    {
    public:
        static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;

        nn::util::IntrusiveListNode m_ListNode;

        UsbCtrlRequest              m_CtrlRequest;

        nn::dd::ProcessHandle       m_ProcHandle;
        uint64_t                    m_ProcVa;
        bool                        m_IsSmmuMapDynamic;
        CompletionCallback          m_Callback;
        void*                       m_Context;
        uint32_t                    m_TimeoutInMs;
        DeferredRequestBase*        m_pDefReq;
        uint32_t                    m_TransferredSize;
        Result                      m_Status;

        Transaction(uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue,
                    uint16_t wIndex, uint16_t wLength,
                    nn::dd::ProcessHandle procHandle, uint64_t procVa, bool isSmmuMapDynamic,
                    CompletionCallback callback, void *context, uint32_t timeoutInMs)
            : m_CtrlRequest{bmRequestType, bRequest, wValue, wIndex, wLength}
            , m_ProcHandle(procHandle)
            , m_ProcVa(procVa)
            , m_IsSmmuMapDynamic(isSmmuMapDynamic)
            , m_Callback(callback)
            , m_Context(context)
            , m_TimeoutInMs(timeoutInMs)
            , m_pDefReq(nullptr)
            , m_TransferredSize(0)
        {
        }
        ~Transaction()
        {
        }
        void * operator new(size_t size) NN_NOEXCEPT;
        void operator delete(void *p, size_t size) NN_NOEXCEPT;
    };


    enum State
    {
        State_Null = 0,
        State_Idle,
        State_Processing,
        State_StallRecovery,
        State_Maximum
    };
    enum Event
    {
        Event_Request = Fsm::Event_Base,
        Event_UrbComplete,
        Event_StallRecoveryComplete,
        Event_Error,
        Event_Maximum
    };

    typedef detail::NnList<Transaction> TransactionListType;

    static const char *EventNames[ControlTransferManager::Event_Maximum];
    static const char *StateNames[ControlTransferManager::State_Maximum];

    Hs*                            m_pHs;
    HostControllerDriverEndpoint*  m_pHcEp;
    TransactionListType            m_TransactionList;
    Transaction*                   m_pCurrentTxn;

    void CompleteCurrentTransaction(Result status);

    static void DeferredRequestCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize);
    static void ResetEndpointCompletionCallback(Hs* pHs,  Result status, HostControllerDriverEndpoint* pHcEp, void *context);

    Result FsmHandler(int32_t state, LocalEventDataType *pEvent);

    Result IdleStateHandler(LocalEventDataType *pEvent);
    Result ProcessingStateHandler(LocalEventDataType *pEvent);
    Result StallRecoveryStateHandler(LocalEventDataType *pEvent);
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

