﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class Hs;
class ClientRootSession;
class DeviceManager;
class ClientIfSession;

class ClientIfSession : public nn::sf::ISharedObject
{
public:
    enum Command
    {
        Command_Invalid = 0,
        Command_ReleaseUsbIf,
        Command_SetInterface,
        Command_GetInterface,
        Command_GetAltUsbIf,
        Command_OpenUsbEp,
        Command_ControlTransfer,
        Command_ResetDevice,
    };
    struct ReleaseUsbIfArgDataType
    {
        InterfaceHandle ifHandle;
    };
    struct SetInterfaceArgDataType
    {
        uint8_t             bAlternateSetting;
        InterfaceProfile*   pData;
    };
    struct GetInterfaceArgDataType
    {
        uint8_t            bAlternateSetting;
        InterfaceProfile*  pData;
    };
    struct OpenUsbEpArgDataType
    {
        UsbEndpointType         epType;
        EndpointNumber          epNumber;
        UsbEndpointDirection    epDirection;
        uint16_t                maxUrbCount;
        uint32_t                maxXferSize;
        UsbEndpointDescriptor*  pReturnedEpDescriptor;
    };
    struct CtrlXferArgDataType
    {
        InterfaceHandle       ifHandle;

        uint8_t               bRequest;
        uint8_t               bmRequestType;
        uint16_t              wValue;
        uint16_t              wIndex;
        uint16_t              wLength;

        uint32_t              timeoutInMs;

        nn::dd::ProcessHandle procHandle;
        uint64_t              procVa;

        Result                result;
        uint32_t              xferredSize;
    };
    struct ResetDeviceArgDataType
    {
        InterfaceHandle       ifHandle;
    };

    typedef DeferredRequest<'I', SetInterfaceArgDataType>    SetInterfaceDeferredRequest;
    typedef DeferredRequest<'I', OpenUsbEpArgDataType>       OpenUsbEpDeferredRequest;
    typedef DeferredRequest<'I', GetInterfaceArgDataType>    GetInterfaceDeferredRequest;
    typedef DeferredRequest<'I', ReleaseUsbIfArgDataType>    ReleaseUsbIfDeferredRequest;
    typedef DeferredRequest<'I', CtrlXferArgDataType>        CtrlXferDeferredRequest;
    typedef DeferredRequest<'I', ResetDeviceArgDataType>     ResetDeviceDeferredRequest;

    nn::util::IntrusiveListNode  m_ListNode;

    ClientIfSession(Hs                      *pHs,
                    ClientRootSession       *pClient,
                    DeviceProfile           *pDeviceProfile,
                    InterfaceProfile        *pInterfaceProfile,
                    nn::dd::ProcessHandle    processHandle) NN_NOEXCEPT;
    ~ClientIfSession() NN_NOEXCEPT;

    Result GetStateChangeEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle ) NN_NOEXCEPT;

    Result SetInterface(uint8_t bAlternateSetting, const nn::sf::OutBuffer &outBuffer)NN_NOEXCEPT;
    Result GetInterface(const nn::sf::OutBuffer &outBuffer)NN_NOEXCEPT;
    Result GetAlternateInterface(uint8_t bAlternateSetting, const nn::sf::OutBuffer &outBuffer)NN_NOEXCEPT;

    Result GetCurrentFrame(nn::sf::Out<FrameNumber> outFrame)NN_NOEXCEPT;

    Result CtrlXferAsync(uint8_t         bmRequestType,
                         uint8_t         bRequest,
                         uint16_t        wValue,
                         uint16_t        wIndex,
                         uint16_t        wLength,
                         uint64_t        procVa) NN_NOEXCEPT;
    Result GetCtrlXferCompletionEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT;
    Result GetCtrlXferReport(nn::sf::OutBuffer buffer) NN_NOEXCEPT;

    Result ResetDevice() NN_NOEXCEPT;

    Result OpenUsbEp(UsbEndpointType epType, EndpointNumber epNumber, UsbEndpointDirection epDirection,
                     uint16_t maxUrb, uint32_t maxXferSize,
                     nn::sf::Out<UsbEndpointDescriptor> outEpDescriptor,
                     nn::sf::Out<nn::sf::SharedPointer<IClientEpSession>> outSession);

    // Methods called by subordinate sessions
    void AddClientEpSession(ClientEpSession* pClientEp);
    void RemoveClientEpSession(ClientEpSession* pClientEp);
    InterfaceProfile& GetInterfaceProfile();
    InterfaceHandle GetHandle();
    void SignalStateChange();

private:
    typedef detail::NnList<ClientEpSession> ClientEpSessionListType;

    Hs*                                      m_pHs;
    nn::sf::SharedPointer<ClientRootSession> m_pClient;
    DeviceProfile                            m_DeviceProfile;
    InterfaceProfile                         m_InterfaceProfile;
    nn::os::Mutex                            m_AccessMutex;
    ClientEpSessionListType                  m_EpSessionList;
    bool                                     m_IsInitialized;
    nn::os::SystemEvent                      m_StateChangeEvent;
    nn::dd::ProcessHandle                    m_ProcessHandle;

    CtrlXferDeferredRequest                  m_CtrlXferRequest;
    nn::os::SystemEvent                      m_CtrlXferCompletionEvent;

private:
    int GetClientEpSessionCount();
};



} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

