﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

ClientEpSession::ClientEpSession(Hs                    *pHs,
                                 ClientId               clientId,
                                 ClientIfSession       *pIf,
                                 UsbEndpointType        epType,
                                 EndpointNumber         epNumber,
                                 UsbEndpointDirection   epDirection,
                                 uint16_t               maxUrbCount,
                                 uint32_t               maxXferSize,
                                 nn::dd::ProcessHandle  processHandle) NN_NOEXCEPT
    : m_pHs(pHs)
    , m_ClientId(clientId)
    , m_pIfSession(pIf, true)
    , m_EpType(epType)
    , m_EndpointNumber(epNumber)
    , m_EpDirection(epDirection)
    , m_MaxUrbCount(maxUrbCount)
    , m_MaxXferSize(maxXferSize)
    , m_CompletionEvent(nn::os::EventClearMode_ManualClear, true)
    , m_ProcessHandle(processHandle)
    , m_RequestMutex(false)
    , m_ReportMutex(false)
{
    NN_USB_LOG_INFO(
        "ClientEpSession clientId=%d, deviceUid=0x%x, ifHandle=0x%x, "
        "epNumber=%d, epDirection=%d\n",
        m_ClientId,
        m_pIfSession->GetInterfaceProfile().deviceUid,
        m_pIfSession->GetHandle(),
        m_EndpointNumber,
        m_EpDirection
    );

    m_pIfSession->AddClientEpSession(this);
}

ClientEpSession::~ClientEpSession() NN_NOEXCEPT
{
    // Force any URBs still pending to be cancelled. Ignore return code as
    // device may have disconnected. Note that if the device is tearing down,
    // this call cann't enforce the synchronization by its own...
    (void)Close();

    // ... Therefore the synchronization must be enforced here.
    m_RequestRing.Finalize();
    m_ReportRing.Finalize();

    m_pIfSession->RemoveClientEpSession(this);

    NN_USB_ABORT_UNLESS_SUCCESS(DestroySmmuSpace());

    NN_USB_LOG_INFO(
        "~ClientEpSession clientId=%d, deviceUid=0x%x, ifHandle=0x%x, "
        "epNumber=%d, epDirection=%d\n",
        m_ClientId,
        m_pIfSession->GetInterfaceProfile().deviceUid,
        m_pIfSession->GetHandle(),
        m_EndpointNumber,
        m_EpDirection
    );
}

/*
 * Allocate and initialize deferred request ring
 */
Result ClientEpSession::PopulateRing() NN_NOEXCEPT
{
    Result result;

    // Request Ring
    result = m_RequestRing.Initialize(
        m_MaxUrbCount,
        [&](uint32_t index, UrbDeferredRequest *pRequest) {
            new (pRequest) UrbDeferredRequest(m_pHs->GetRequestQueue(),
                                              Command_Urb);

            auto& data = pRequest->m_Data;

            data.id           = index;

            data.ifHandle     = m_pIfSession->GetHandle();
            data.epNumber     = m_EndpointNumber;
            data.epDirection  = m_EpDirection;

            data.procHandle   = m_ProcessHandle;
            data.pSmmuSpace   = nullptr;
            data.procVa       = 0;

            data.status       = XferStatus_Invalid;
            data.xferCount    = 0;
            for (auto& xfer : data.xfers)
            {
                xfer.size        = 0;
                xfer.xferredSize = 0;
                xfer.result      = ResultSuccess();
            }

            data.timeoutInMs  = 0;
        }
    );
    if (result.IsFailure())
    {
        return result;
    }

    // Report Ring
    result = m_ReportRing.Initialize(m_MaxUrbCount);
    if (result.IsFailure())
    {
        m_RequestRing.Finalize();
        return result;
    }

    return result;
}

Result ClientEpSession::CreateSmmuSpace(uint64_t procVa, uint32_t size) NN_NOEXCEPT
{
    Result result;
    nn::dd::MemoryPermission permission;

    // Only one static map is allowed
    if (m_SmmuSpace.procVa != 0)
    {
        return ResultOperationDenied();
    }

    // The alignment is checked in shim
    NN_USB_ABORT_UNLESS(NN_USB_IS_DMA_ALIGNED(procVa));
    NN_USB_ABORT_UNLESS(NN_USB_IS_DMA_ALIGNED(size));

    if (m_EpDirection == UsbEndpointDirection_ToDevice)
    {
        permission = nn::dd::MemoryPermission_ReadOnly;
    }
    else
    {
        permission = nn::dd::MemoryPermission_WriteOnly;
    }

    result = m_pHs->m_pController->DoSmmu().Map(
        &m_SmmuSpace.ioVa,
        m_ProcessHandle,
        procVa,
        size,
        permission,
        permission
    );

    if (result.IsSuccess())
    {
        m_SmmuSpace.procVa = procVa;
        m_SmmuSpace.size   = size;
    }

    return result;
}

Result ClientEpSession::DestroySmmuSpace() NN_NOEXCEPT
{
    Result result;

    if (m_SmmuSpace.procVa == 0)
    {
        return ResultSuccess();
    }

    result = m_pHs->m_pController->DoSmmu().Unmap(
        m_ProcessHandle,
        m_SmmuSpace.procVa,
        m_SmmuSpace.size
    );

    if (result.IsSuccess())
    {
        m_SmmuSpace.procVa = 0;
        m_SmmuSpace.size   = 0;
        m_SmmuSpace.ioVa   = 0;
    }

    return result;
}

Result ClientEpSession::ShareReportRing(nn::sf::NativeHandle ringHandle, uint32_t size) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_ReportRing.Finalize();

    result = m_ReportRing.Initialize(m_MaxUrbCount, ringHandle.GetOsHandle(), size);
    if (result.IsSuccess())
    {
        ringHandle.Detach();
    }

    return result;
}

Result ClientEpSession::GetCompletionEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandle) NN_NOEXCEPT
{
    eventHandle.Set(nn::sf::NativeHandle(m_CompletionEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

Result ClientEpSession::BatchBufferAsync(nn::sf::Out<uint32_t>      pOutXferId,
                                         uint64_t                   procVa,
                                         nn::sf::InBuffer           sizeBuffer,
                                         uint32_t                   xferCount,
                                         uint64_t                   context,
                                         SchedulePolicy             policy,
                                         uint32_t                   frameId) NN_NOEXCEPT
{
    const uint32_t *sizeArray = reinterpret_cast<const uint32_t*>(sizeBuffer.GetPointerUnsafe());

    if (sizeBuffer.GetSize() < xferCount * sizeof(uint32_t))
    {
        return ResultBufferSizeError();
    }

    return BatchBufferAsync(
        pOutXferId.GetPointer(), procVa, sizeArray, xferCount, context, policy, frameId
    );
}


Result ClientEpSession::BatchBufferAsync(uint32_t                  *pOutXferId,
                                         uint64_t                   procVa,
                                         const uint32_t            *sizeArray,
                                         uint32_t                   xferCount,
                                         uint64_t                   context,
                                         SchedulePolicy             policy,
                                         uint32_t                   frameId) NN_NOEXCEPT
{
    /*
     * Since SF calls can be issued in parallel, we have to protect the Request
     * Ring head because we are going to modify it in this call.
     */
    std::lock_guard<nn::os::Mutex> lock(m_RequestMutex);

    uint32_t            xferSize = 0;
    UrbDeferredRequest *pRequest;
    detail::SmmuSpace  *pSmmuSpace = nullptr;

    if (xferCount > HsLimitMaxXferPerUrbCount)
    {
        return ResultImplementationLimit();
    }

    for (uint32_t i = 0; i < xferCount; i++)
    {
        xferSize += sizeArray[i];
    }

    if (m_SmmuSpace.Includes(procVa))
    {
        if (m_SmmuSpace.Includes(procVa + xferSize - 1))
        {
            pSmmuSpace = &m_SmmuSpace;
        }
        else
        {
            return ResultInvalidParameter();
        }
    }

    pRequest = m_RequestRing.Expand();
    if (pRequest == nullptr)
    {
        return ResultUrbAllocFailure();
    }

    auto& data = pRequest->m_Data;

    NN_STATIC_ASSERT(HsLimitMaxUrbPerEndpointCount <= 0xff);
    data.id        += 0x100;
    data.pSmmuSpace = pSmmuSpace;
    data.procVa     = procVa;
    data.status     = XferStatus_Pending;
    data.xferCount  = xferCount;
    for (uint32_t i = 0; i < xferCount; i++)
    {
        data.xfers[i].size        = sizeArray[i];
        data.xfers[i].xferredSize = 0;
    }
    data.policy     = policy;
    data.frameId    = frameId;
    data.context    = context;

    *pOutXferId = data.id;

    pRequest->AsynchronousRequest(
        [=]()-> Result {
            Result result = m_pHs->m_DeviceManager.SubmitUrb(pRequest);

            if (result.IsSuccess())  // submission succeed
            {
                return ResultDeferred();
            }

            auto& data = pRequest->m_Data;

            // Mark the submission error
            data.status = XferStatus_Failed;
            for (uint32_t i = 0; i < data.xferCount; i++)
            {
                data.xfers[i].xferredSize = 0;
                data.xfers[i].result      = result;
            }

            /*
             * The Request Ring tail is changed in bottom-half labmda below,
             * but the bottom-half lambda is implicitly serialized with this
             * top-half labmda so it's fine.
             */
            if (m_RequestRing.Tail() != pRequest) // not the first submission
            {
                // Defer the completion
                result = ResultDeferred();
            }

            return result;
        },
        [=] {
            /*
             * This bottom-half labmda modifies Request Ring tail and Report
             * Ring head, but there is no need to protect them since
             * top/bottom-half lambdas are implicitly serialized - they both
             * run in MainThread.
             */
            auto *pUrbRequest = m_RequestRing.Shrink();

            NN_USB_ABORT_UNLESS(pRequest == pUrbRequest);

            auto& data = pUrbRequest->m_Data;

            // Collect the transfer report
            for (uint32_t i = 0; i < data.xferCount; i++)
            {
                auto *pReport = m_ReportRing.Expand();

                if (pReport != nullptr)
                {
                    pReport->id              = data.id;
                    pReport->result          = data.xfers[i].result;
                    pReport->requestedSize   = data.xfers[i].size;
                    pReport->transferredSize = data.xfers[i].xferredSize;
                    pReport->context         = data.context;
                }
                else
                {
                    NN_USB_LOG_WARN(
                        "Report Ring Overrun (clientId=%d, deviceUid=0x%x, "
                        "ifHandle=0x%x, epNumber=%d, epDirection=%d)\n",
                        m_ClientId,
                        m_pIfSession->GetInterfaceProfile().deviceUid,
                        m_pIfSession->GetHandle(),
                        m_EndpointNumber,
                        m_EpDirection
                    );
                }
            }

            // The report is ready and this transfer is officially completed
            m_CompletionEvent.Signal();

            /*
             * Check and invoke the deferred completion.
             *
             * Request Ring could Expand() in SF thread, but SubmitUrb() is
             * called only in MainThread, so the test below should be fine.
             */
            pUrbRequest = m_RequestRing.Next(pUrbRequest);
            if (pUrbRequest != m_RequestRing.Head() &&
                pUrbRequest->m_Data.status == XferStatus_Failed)
            {
                pUrbRequest->CompleteRequest();
            }
        }
    );

    return ResultSuccess();
} // NOLINT(impl/function_size)

Result ClientEpSession::PostBufferAsync(nn::sf::Out<uint32_t> pOutXferId,
                                        uint64_t              procVa,
                                        uint32_t              bytes,
                                        uint64_t              context) NN_NOEXCEPT
{
    return BatchBufferAsync(
        pOutXferId.GetPointer(), procVa,
        reinterpret_cast<const uint32_t*>(&bytes), 1,
        context, SchedulePolicy_Absolute, 0
    );
}

Result ClientEpSession::GetXferReport(nn::sf::Out<uint32_t> pOutCount,
                                      nn::sf::OutBuffer     buffer,
                                      uint32_t              count) NN_NOEXCEPT
{
    /*
     * Since SF calls can be issued in parallel, we have to protect the Report
     * Ring tail because we are going to modify it in this call.
     */
    std::lock_guard<nn::os::Mutex> lock(m_ReportMutex);

    XferReport *pReport;
    uint32_t    collectedCount = 0;

    // Make sure the buffer is big enough
    if (buffer.GetSize() < sizeof(XferReport) * count)
    {
        return ResultBufferSizeError();
    }

    pReport = reinterpret_cast<XferReport*>(buffer.GetPointerUnsafe());

    while (collectedCount < count)
    {
        auto *ptr = m_ReportRing.Shrink();

        if (ptr == nullptr)
        {
            break;
        }

        pReport[collectedCount++] = *ptr;
    }

    *pOutCount = collectedCount;

    return ResultSuccess();
}

Result ClientEpSession::ReOpen() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    ClientIfSession::OpenUsbEpDeferredRequest request(m_pHs->GetRequestQueue(),
                                                      ClientIfSession::Command_OpenUsbEp);

    request.m_Data.epType      = m_EpType;
    request.m_Data.epNumber    = m_EndpointNumber;
    request.m_Data.epDirection = m_EpDirection;
    request.m_Data.maxUrbCount = m_MaxUrbCount;
    request.m_Data.maxXferSize = m_MaxXferSize;
    request.m_Data.pReturnedEpDescriptor = nullptr;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                m_pIfSession->GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientEpSession::Close() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    CloseDeferredRequest request(m_pHs->GetRequestQueue(),
                                 Command_Close);

    request.m_Data.epNumber    = m_EndpointNumber;
    request.m_Data.epDirection = m_EpDirection;

    request.SynchronousRequest(
        [&] {
            return m_pHs->m_DeviceManager.SubmitDeferredIfAdminRequest(
                m_pIfSession->GetHandle(),
                &request
            );
        },
        [&] {
            result = request.GetActionResult();
        }
    );

    return result;
}

Result ClientEpSession::RequestRing::Initialize(uint32_t itemCount, Initializer initializer) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_pStorage = reinterpret_cast<Storage*>(
        detail::UsbMemoryAllocAligned(
            sizeof(Storage) + sizeof(UrbDeferredRequest) * itemCount,
            alignof(Storage),
            "RequestRing"
        )
    );

    if (m_pStorage == nullptr)
    {
        return ResultMemAllocFailure();
    }

    m_pStorage->head = 0;
    m_pStorage->tail = 0;

    m_ItemCount = itemCount + 1; // +1 for sentinel

    for (uint32_t i = 0; i < m_ItemCount; i++)
    {
        initializer(i, Get(i));
    }

    return result;
}

void ClientEpSession::RequestRing::Finalize() NN_NOEXCEPT
{
    if (m_pStorage)
    {
        for (uint32_t i = 0; i < m_ItemCount; i++)
        {
            Get(i)->Flush();
            Get(i)->~UrbDeferredRequest();
        }

        detail::UsbMemoryFree(m_pStorage, "RequestRing");
    }

    m_pStorage  = nullptr;
    m_ItemCount = 0;
}

ClientEpSession::ReportRing::ReportRing() NN_NOEXCEPT
    : detail::RingBase<XferReport>()
    , m_IsShared(false)
{
}

Result ClientEpSession::ReportRing::Initialize(uint32_t maxUrbCount) NN_NOEXCEPT
{
    Result result = Attach(HsLimitMaxXferPerUrbCount * maxUrbCount);

    if (result.IsSuccess())
    {
        Populate();
    }

    return result;
}

Result ClientEpSession::ReportRing::Initialize(uint32_t             maxUrbCount,
                                               nn::os::NativeHandle ringHandle,
                                               uint32_t             size) NN_NOEXCEPT
{
    Result result = Attach(HsLimitMaxXferPerUrbCount * maxUrbCount, ringHandle, size);

    if (result.IsSuccess())
    {
        Populate();
    }

    return result;
}

void ClientEpSession::ReportRing::Finalize() NN_NOEXCEPT
{
    Detach();
}

Result ClientEpSession::ReportRing::Attach(uint32_t itemCount) NN_NOEXCEPT
{
    uint32_t size = NN_USB_ROUNDUP_DMA_SIZE(sizeof(Storage) + sizeof(ItemType) * itemCount);

    m_pStorage = reinterpret_cast<Storage*>(
        detail::UsbMemoryAllocAligned(size, nn::os::MemoryPageSize, "ReportRing")
    );

    if (m_pStorage == nullptr)
    {
        return ResultMemAllocFailure();
    }

    m_ItemCount = itemCount + 1; // +1 for sentinel
    m_IsShared  = false;

    return ResultSuccess();
}

Result ClientEpSession::ReportRing::Attach(uint32_t             itemCount,
                                           nn::os::NativeHandle ringHandle,
                                           uint32_t             size) NN_NOEXCEPT
{
    Result result;
    void *ptr;

    if (!NN_USB_IS_DMA_ALIGNED(size))
    {
        return ResultAlignmentError();
    }

    if (size < sizeof(Storage) + sizeof(ItemType) * itemCount)
    {
        return ResultInvalidParameter();
    }

    nn::os::AttachTransferMemory(&m_XferMem, size, ringHandle, true);

    result = nn::os::MapTransferMemory(
        &ptr, &m_XferMem, nn::os::MemoryPermission_ReadWrite
    );
    if (result.IsFailure())
    {
        DestroyTransferMemory(&m_XferMem);
        return result;
    }

    m_pStorage  = reinterpret_cast<Storage*>(ptr);
    m_ItemCount = itemCount + 1; // +1 for sentinel
    m_IsShared  = true;

    return ResultSuccess();
}

void ClientEpSession::ReportRing::Detach() NN_NOEXCEPT
{
    if (m_IsShared)
    {
        nn::os::UnmapTransferMemory(&m_XferMem);
        nn::os::DestroyTransferMemory(&m_XferMem);
    }
    else if (m_pStorage)
    {
        detail::UsbMemoryFree(m_pStorage, "ReportRing");
    }

    m_pStorage  = nullptr;
    m_ItemCount = 0;
    m_IsShared  = false;
}

void ClientEpSession::ReportRing::Populate() NN_NOEXCEPT
{
    m_pStorage->head = 0;
    m_pStorage->tail = 0;

    for (uint32_t i = 0; i < m_ItemCount; i++)
    {
        memset(Get(i), 0, sizeof(XferReport));
    }
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
