﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file    usb_HsCh11.h
 * @brief   USB constants and structures related to the USB Hub
 *          Specification which is defined in chapter 11 of the
 *          USB 2.0 specification.
 */
#pragma once
#include <nn/usb/usb_Limits.h>

namespace nn { namespace usb { namespace hs {

/*
 * Hub descriptor
 * See USB 2.0 spec Table 11-13
 */

struct UsbHubDescriptor
{
    uint8_t  bDescLength;
    uint8_t  bDescriptorType;
    uint8_t  bNbrPorts;
    uint16_t wHubCharacteristics;
    uint8_t  bPwrOn2PwrGood;
    uint8_t  bHubContrCurrent;

    /* add 1 bit for hub status change; round to bytes */
    uint8_t  deviceRemovable[(HsLimitMaxHubPortsCount + 1 + 7) / 8];

    /* This field exists for reasons of compatibility with software
       written for 1.0 compliant devices. All bits in this field
       should be set to 1B. */
    uint8_t  portPwrCtrlMask;

}NN_USB_PACKED_STRUCT_ATTRIBUTE;

// bDescriptorType
static const uint8_t UsbHubDescriptorType = 0x29;

#define NN_USB_HUB_DESCRIPTOR_SIZE(maxPorts) (7 + ((maxPorts)+1+7)/8)

/*
 * Ss Hub descriptor
 * See USB 3.0 spec Table 10-5
 */
struct UsbSsHubDescriptor
{
    uint8_t  bDescLength;
    uint8_t  bDescriptorType;
    uint8_t  bNbrPorts;
    uint16_t wHubCharacteristics;
    uint8_t  bPwrOn2PwrGood;
    uint8_t  bHubContrCurrent;
    uint8_t  bHubHdrDecLat;
    uint8_t  bHubDelay;

    /* add 1 bit for hub status change; round to bytes */
    uint8_t  deviceRemovable[(HsLimitMaxHubPortsCount + 1 + 7) / 8];

}NN_USB_PACKED_STRUCT_ATTRIBUTE;

// bDescriptorType
static const uint8_t UsbSsHubDescriptorType = 0x2A;

#define NN_USB_SS_HUB_DESCRIPTOR_SIZE(maxPorts) (10 + ((maxPorts)+1+7)/8)

/*
 * wHubCharacteristics
 */
static const int32_t  HubCharacteristicsTtThinkTimeShift = 5;
static const uint16_t HubCharacteristicsTtThinkTimeMask = 3 << HubCharacteristicsTtThinkTimeShift;

/*
 * Common hub descriptor
 */
union UsbHubDescriptorUnion
{
    UsbHubDescriptor   standard;
    UsbSsHubDescriptor super;
};

/*
 * Port feature numbers
 * See USB 2.0 spec Table 11-17.
 * See USB 3.1 spec Table 10-8
 */
enum UsbHubPortFeature
{
    UsbHubPortFeature_Connection         = 0,
    UsbHubPortFeature_Enable             = 1,  // 2.0 only
    UsbHubPortFeature_Suspend            = 2,  // 2.0 only
    UsbHubPortFeature_OverCurrent        = 3,
    UsbHubPortFeature_Reset              = 4,
    UsbHubPortFeature_L1                 = 5,
    UsbHubPortFeature_Power              = 8,
    UsbHubPortFeature_CConnetion         = 16,
    UsbHubPortFeature_CEnable            = 17, // 2.0 only
    UsbHubPortFeature_CSuspend           = 18, // 2.0 only
    UsbHubPortFeature_COverCurrent       = 19,
    UsbHubPortFeature_CReset             = 20,
    UsbHubPortFeature_Test               = 21,
    UsbHubPortFeature_Indicator          = 22, // 2.0 only
    UsbHubPortFeature_U1Timeout          = 23,
    UsbHubPortFeature_U2Timeout          = 24,
    UsbHubPortFeature_CPortLinkState     = 25,
    UsbHubPortFeature_CPortConfigError   = 26,
    UsbHubPortFeature_PortRemoteWakeMask = 27,
    UsbHubPortFeature_BhPortReset        = 28,
    UsbHubPortFeature_CBhPortReset       = 29,
    UsbHubPortFeature_ForceLinkPmAccept  = 30,
};

/*
 * Hub Status and Hub Change results
 * See USB 2.0 spec Table 11-19 and Table 11-20
 */
struct UsbHubPortStatus
{
    uint16_t wPortStatus;
    uint16_t wPortChange;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/*
 * wPortStatus bit field
 * See USB 2.0 spec Table 11-21
 * See USB 3.1 spec Table 10-13
 */
enum UsbPortStatus
{
    UsbPortStatus_Zero                 = 0x0000,
    UsbPortStatus_Connection           = 0x0001,
    UsbPortStatus_Enable               = 0x0002,
    UsbPortStatus_Suspend              = 0x0004,
    UsbPortStatus_Overcurrent          = 0x0008,
    UsbPortStatus_Reset                = 0x0010,
    UsbPortStatus_Power                = 0x0100,
    UsbPortStatus_LowSpeed             = 0x0200,
    UsbPortStatus_HighSpeed            = 0x0400,
    UsbPortStatus_Test                 = 0x0800,
    UsbPortStatus_Indicator            = 0x1000,

    // 3.0 specific
    UsbPortStatus_LinkStateU0          = 0x0000,
    UsbPortStatus_LinkStateU1          = 0x0020,
    UsbPortStatus_LinkStateU2          = 0x0040,
    UsbPortStatus_LinkStateU3          = 0x0060,
    UsbPortStatus_LinkStateEssDisabled = 0x0080,
    UsbPortStatus_LinkStateRxDetect    = 0x00A0,
    UsbPortStatus_LinkStateEssInactive = 0x00C0,
    UsbPortStatus_LinkStatePolling     = 0x00E0,
    UsbPortStatus_LinkStateRecovery    = 0x0100,
    UsbPortStatus_LinkStateHotReset    = 0x0120,
    UsbPortStatus_LinkStateCompliance  = 0x0140,
    UsbPortStatus_LinkStateLoopback    = 0x0160,
    UsbPortStatus_LinkStateMask        = 0x01E0,
    UsbPortStatus_PowerSs              = 0x0200,
    UsbPortStatus_PortSpeedMask        = 0x1C00,
    UsbPortStatus_PortSpeedShift       = 10,
};

/*
 * wPortChange bit field
 * See USB 2.0 spec Table 11-22 and USB 2.0 LPM ECN Table-4.10
 * See USB 3.1 spec Table 10-13
 */
enum UsbPortStatusChange
{
    UsbPortStatusChange_Zero        = 0x0000,
    UsbPortStatusChange_Connection  = 0x0001,
    UsbPortStatusChange_Enable      = 0x0002,
    UsbPortStatusChange_Suspend     = 0x0004,
    UsbPortStatusChange_Overcurrent = 0x0008,
    UsbPortStatusChange_Reset       = 0x0010,

    // 3.0 specific
    UsbPortStatusChange_BhReset     = 0x0020,
    UsbPortStatusChange_LinkState   = 0x0040,
    UsbPortStatusChange_ConfigError = 0x0080,
};


// size of interrupt transfer is a function of number of ports, per 11.12.4 of USB spec
#define NN_USB_HUB_EVENT_XFER_SIZE(ports) ( ((ports) + 1 + 7) / 8)

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
