﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE

#include "usb_DsController.h"
#include "usb_DsUrbRing.h"

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsUrbRing"

using namespace nn::dd;

namespace nn {
namespace usb {
namespace ds {

//----------------------------------------------------------------------------
//  public functions
//----------------------------------------------------------------------------

void* DsUrbRing::operator new(size_t size) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return detail::UsbMemoryAlloc(size, "DsUrbRing");
}
void DsUrbRing::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_USB_TRACE;
    detail::UsbMemoryFree(p, "DsUrbRing");
}

void DsUrbRing::SetController(DsController * pController) NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_pController = pController;
}

Result DsUrbRing::Reset() NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_Head = 0;
    m_Tail = 0;

    // so `i' can be packed into 3 bits, kind of hack here
    NN_STATIC_ASSERT(NN_USB_ARRAY_SIZE(m_Urb) <= 8);

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_Urb); i++)
    {
        m_Urb[i].id     = m_Address | (i << 4);
        m_Urb[i].status = UrbStatus_Invalid;
    }

    m_CompletionEvent.Clear();

    return ResultSuccess();
}

Result DsUrbRing::Enqueue(uint32_t *pOutUrbId,
                          nn::dd::ProcessHandle processHandle,
                          uint64_t address, uint32_t bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    const uint64_t MemoryAlignmentForXUsb = 64 * 1024;
    uint64_t smmuAddress = 0;
    size_t roundedUpSize = NN_USB_ROUNDUP_SIZE(bytes, DeviceAddressSpaceMemoryRegionAlignment);
    Result result = ResultSuccess();
    // DO MMU setup outside of lock.

    // sMMU map is nunecessary for zero-length-transfer
    if (bytes != 0)
    {
        result = m_pController->DoSmmu().Map(&smmuAddress,
                                                processHandle,
                                                address,
                                                roundedUpSize,
                                                MemoryAlignmentForXUsb,
                                                MemoryPermission_None,
                                                MemoryPermission_ReadWrite);
        if (result.IsFailure())
        {
            return result;
        }
    }

    // Now lock for actual Urb work.
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        Urb&   urb    = m_Urb[m_Head];
        if (IsFull())
        {
            NN_USB_LOG_INFO("Ep 0x%x is full - returning resource busy  \n", m_Address);
            return ResultResourceBusy();
        }

        urb.processHandle = processHandle;
        urb.ioVa   = smmuAddress;
        urb.id    += 0x0100;
        urb.status = UrbStatus_Pending;
        urb.procVa = address;
        urb.bytes  = bytes;
        *pOutUrbId = urb.id;

        m_Head = Advance(m_Head);

        if (Advance(m_Tail) == m_Head)
        {
            result = Prime();
        }
    }

    return result;
}

Result DsUrbRing::Dequeue(Result result, uint32_t bytesRemaining, bool lockMutex) NN_NOEXCEPT
{
    NN_USB_TRACE;

    if (lockMutex)
    {
        m_Mutex.Lock();
    }

    Urb& urb = m_Urb[m_Tail];

    if (bytesRemaining)
    {
        NN_USB_LOG_INFO("Short Packet - Bytes transferred %d original size %d\n", urb.bytes - bytesRemaining, urb.bytes);
    }

    if (!IsEmpty())
    {
        if (result.IsSuccess())
        {
            urb.transferredSize = urb.bytes - bytesRemaining;
            urb.status = UrbStatus_Finished;
        }
        else
        {
            urb.transferredSize = 0;
            urb.status = UrbStatus_Failed;
        }

        if (urb.bytes != 0)
        {
            m_pController->DoSmmu().UnmapByIoVa(urb.ioVa);
        }

        if (((m_Address & UsbEndpointAddressMask_DirDeviceToHost) == 0) && urb.procVa && urb.bytes)
        {
            nn::dd::InvalidateProcessDataCache(urb.processHandle, urb.procVa, urb.bytes);
        }

        m_Tail = Advance(m_Tail);
        //NN_USB_LOG_INFO("%s::%s(@%d) m_Head %d m_Tail=%d m_Address=%d\n", kUsbLogClassName, __FUNCTION__,__LINE__, m_Head, m_Tail, m_Address);

        m_CompletionEvent.Signal();

        // keep the water running
        Prime();
    }

    if (lockMutex)
    {
        m_Mutex.Unlock();
    }

    return ResultSuccess();
}

Result DsUrbRing::Flush() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (IsEmpty())
    {
        return ResultSuccess();
    }

    // cancel all the urbs in the ring
    while (!IsEmpty())
    {
        Urb& urb = m_Urb[m_Tail];

        urb.status = UrbStatus_Cancelled;

        if (urb.bytes != 0)
        {
            m_pController->DoSmmu().UnmapByIoVa(urb.ioVa);
        }

        m_Tail = Advance(m_Tail);
    }

    m_CompletionEvent.Signal();

    return ResultSuccess();
}

Result DsUrbRing::GetReport(UrbReport *pReport) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    uint32_t index = m_Head;
    uint32_t count = 0;

    for (uint32_t n = 0; n < NN_USB_ARRAY_SIZE(m_Urb); n++)
    {
        Urb& urb = m_Urb[index];

        index = Advance(index);

        if (urb.status == UrbStatus_Invalid)
        {
            continue;
        }

        pReport->report[count].id              = urb.id;
        pReport->report[count].requestedSize   = urb.bytes;
        pReport->report[count].transferredSize = urb.transferredSize;
        pReport->report[count].status          = urb.status;

        switch (urb.status)
        {
        case UrbStatus_Finished:
        case UrbStatus_Cancelled:
        case UrbStatus_Failed:
            urb.status = UrbStatus_Invalid;
            break;

        default:
            break;
        }

        count++;
    }

    pReport->count = count;

    m_CompletionEvent.Clear();

    return ResultSuccess();
}

nn::os::SystemEventType* DsUrbRing::GetCompletionEvent() NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_CompletionEvent.GetBase();
}

void DsUrbRing::BlockForCompletion() NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_CompletionEvent.Wait();
    m_CompletionEvent.Clear();
}

//----------------------------------------------------------------------------
//  private functions
//----------------------------------------------------------------------------

/*
 * prime the next urb, if any
 */
Result DsUrbRing::Prime() NN_NOEXCEPT
{
    // caller maintains the lock

    // This is vestigate code from when the EP was inside the controller
    // When we change the interface, this goes away.
    nn::dd::ProcessHandle processHandle(0);
    Urb& urb = m_Urb[m_Tail];

    if (IsEmpty())
    {
        return ResultSuccess();
    }

    urb.status = UrbStatus_Running;

    return m_pController->PostBufferAsync(m_Address, processHandle, urb.ioVa, urb.bytes);
}


} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
