﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE 1

#include "usb_DsInterfaceImpl.h"
#include "usb_DsServiceImpl.h"

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsInterfaceImpl"

using namespace nn::sf;
using namespace nn::usb;

namespace nn {
namespace usb {
namespace ds {

DsInterfaceImpl::DsInterfaceImpl(DsProtocol                   *pProtocol,
                                 DsServiceImpl                *pService,
                                 uint8_t                       bInterfaceNumber,
                                 nn::dd::ProcessHandle         processHandle) NN_NOEXCEPT
    : m_Mutex(false)
    , m_IsEnabled(false)
    , m_pProtocol(pProtocol)
    , m_pService(pService, true)
    , m_SetupEvent(nn::os::EventClearMode_ManualClear, true)
    , m_ProcessHandle(processHandle)
{
    NN_USB_TRACE;

    m_IfNum = bInterfaceNumber;

    m_pService->OnAddInterface(m_IfNum, this);

    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pEndpoint); i++)
    {
        m_pEndpoint[i] = nullptr;
    }

//    NN_USB_LOG_INFO("[0x%08p] DsInterfaceImpl(%02x) %p\n", nn::os::GetCurrentThread(), m_IfNum, this);
}

DsInterfaceImpl::~DsInterfaceImpl() NN_NOEXCEPT
{
    NN_USB_TRACE;
//    NN_USB_LOG_INFO("[0x%08p] ~DsInterfaceImpl(%02x) %p\n", nn::os::GetCurrentThread(), m_IfNum, this);

    Disable();
    m_pProtocol->UnRegisterInterface(m_IfNum);

    m_pService->OnDelInterface(m_IfNum);
}

//////////////////////////////////////////////////////////////////////////////
// Service Framework Interface Methods
//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceImpl::AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, nn::sf::InBuffer pData) NN_NOEXCEPT
{
    NN_USB_TRACE;
    uint8_t data[128];
    size_t dataSize = pData.GetSize();

    if (dataSize > 128)
    {
        return ResultBufferSizeError();
    }

    memcpy(data, pData.GetPointerUnsafe(), dataSize);

    return m_pProtocol->AppendConfigurationData(bInterfaceNumber, usbDeviceSpeed, data, dataSize);
}




Result DsInterfaceImpl::RegisterEndpoint(uint8_t bEndpointAddress, Out<SharedPointer<IDsEndpoint>> pEndpoint) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    result = m_pProtocol->RegisterEndpoint(bEndpointAddress, m_IfNum);

    if (result.IsSuccess())
    {
        *pEndpoint = detail::Factory::CreateSharedEmplaced<IDsEndpoint, DsEndpointImpl>(
            detail::UsbMemoryGetAllocator(),
            m_pProtocol,
            this,
            bEndpointAddress,
            m_ProcessHandle
        );
    }

    return result;
}

Result DsInterfaceImpl::GetSetupEvent(Out<NativeHandle> eventHandle) NN_NOEXCEPT
{
    NN_USB_TRACE;
    eventHandle.Set(NativeHandle(m_SetupEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

Result DsInterfaceImpl::GetSetupPacket(OutBuffer setup) NN_NOEXCEPT
{
    NN_USB_TRACE;
    if (!m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    m_pProtocol->GetSetupPacket(0x00, setup.GetPointerUnsafe(), setup.GetSize(), &m_SetupEvent);

    return ResultSuccess();
}

Result DsInterfaceImpl::Enable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (m_IsEnabled)
    {
        return ResultSuccess();
    }

    m_pProtocol->BindSetupEvent(0x00, m_IfNum, &m_SetupEvent);

    result = m_pProtocol->EnableInterface(m_IfNum);
    if (result.IsSuccess())
    {
        m_IsEnabled = true;
    }

    return result;
}

Result DsInterfaceImpl::Disable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (!m_IsEnabled)
    {
        return ResultSuccess();
    }

    m_pProtocol->UnbindSetupEvent(0x00, m_IfNum);

    result = m_pProtocol->DisableInterface(m_IfNum);
    if (result.IsSuccess())
    {
        m_IsEnabled = false;
    }

    return result;
}

Result DsInterfaceImpl::CtrlInAsync(Out<uint32_t>   pOutUrbId,
                                    uint64_t        address,
                                    uint32_t        bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;
    uint32_t urbId = 0;

    if (!m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    if (!NN_USB_IS_DMA_ALIGNED(address))
    {
        return ResultAlignmentError();
    }

    result = m_pProtocol->PostBufferAsync(
        0x80,
        m_ProcessHandle,
        address,
        bytes,
        &urbId
    );

    if (result.IsSuccess())
    {
        *pOutUrbId = urbId;
    }

    return result;
}

Result DsInterfaceImpl::CtrlOutAsync(Out<uint32_t>   pOutUrbId,
                                     uint64_t        address,
                                     uint32_t        bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;
    uint32_t urbId = 0;

    if (!m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    if (!NN_USB_IS_DMA_ALIGNED(address))
    {
        return ResultAlignmentError();
    }

    result = m_pProtocol->PostBufferAsync(
        0x00,
        m_ProcessHandle,
        address,
        bytes,
        &urbId
    );

    if (result.IsSuccess())
    {
        *pOutUrbId = urbId;
    }

    return result;
}

Result DsInterfaceImpl::GetCtrlInCompletionEvent(Out<NativeHandle> eventHandle) NN_NOEXCEPT
{
    NN_USB_TRACE;
    nn::os::SystemEventType *event = m_pProtocol->GetCompletionEvent(0x80);

    if(event)
        eventHandle.Set(
            NativeHandle(nn::os::GetReadableHandleOfSystemEvent(event), false)
        );

    return ResultSuccess();
}

Result DsInterfaceImpl::GetCtrlInUrbReport(Out<UrbReport> report) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_pProtocol->GetUrbReport(0x80, report.GetPointer());
}

Result DsInterfaceImpl::GetCtrlOutCompletionEvent(Out<NativeHandle> eventHandle) NN_NOEXCEPT
{
    NN_USB_TRACE;
    nn::os::SystemEventType *event = m_pProtocol->GetCompletionEvent(0x00);

    if(event)
        eventHandle.Set(
            NativeHandle(nn::os::GetReadableHandleOfSystemEvent(event), false)
        );

    return ResultSuccess();
}

Result DsInterfaceImpl::GetCtrlOutUrbReport(Out<UrbReport> report) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_pProtocol->GetUrbReport(0x00, report.GetPointer());
}

Result DsInterfaceImpl::CtrlStall() NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (!m_IsEnabled)
    {
        return ResultOperationDenied();
    }

    NN_USB_RETURN_UPON_ERROR(m_pProtocol->Stall(0x80));
    NN_USB_RETURN_UPON_ERROR(m_pProtocol->Stall(0x00));

    return result;
}

//----------------------------------------------------------------------------
// Internal Use Methods
//----------------------------------------------------------------------------

void DsInterfaceImpl::OnAddEndpoint(uint8_t epAddress, DsEndpointImpl *pEndpoint) NN_NOEXCEPT
{
    NN_USB_TRACE;
    int epIndex = detail::GetEndpointIndex(epAddress);

    m_pEndpoint[epIndex] = pEndpoint;
}

void DsInterfaceImpl::OnDelEndpoint(uint8_t epAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;
    int epIndex = detail::GetEndpointIndex(epAddress);

    m_pEndpoint[epIndex] = nullptr;
}

bool DsInterfaceImpl::IsEnabled() NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_IsEnabled;
}

bool DsInterfaceImpl::IsParentOf(uint8_t epAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;
    int epIndex = detail::GetEndpointIndex(epAddress);

    return (m_pEndpoint[epIndex] != nullptr);
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
