﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_DsEndpointImpl.h"
#include "usb_DsInterfaceImpl.h"

using namespace nn::sf;

namespace nn {
namespace usb {
namespace ds {

DsEndpointImpl::DsEndpointImpl(DsProtocol                  *pProtocol,
                               DsInterfaceImpl             *pInterface,
                               uint8_t                      endpointAddress,
                               nn::dd::ProcessHandle        processHandle) NN_NOEXCEPT
    : m_pProtocol(pProtocol)
    , m_pInterface(pInterface, true)
    , m_ProcessHandle(processHandle)
{
    m_Address = endpointAddress;

    m_pInterface->OnAddEndpoint(m_Address, this);

//    NN_USB_LOG_INFO("[0x%08p] DsEndpointImpl(%02x) %p\n", nn::os::GetCurrentThread(), m_Address, this);
}

DsEndpointImpl::~DsEndpointImpl() NN_NOEXCEPT
{
    m_pProtocol->Cancel(m_Address);
    m_pProtocol->UnRegisterEndpoint(m_Address);
    m_pInterface->OnDelEndpoint(m_Address);

//    NN_USB_LOG_INFO("[0x%08p] ~DsEndpointImpl(%02x) %p\n", nn::os::GetCurrentThread(), m_Address, this);
}

Result DsEndpointImpl::PostBufferAsync(Out<uint32_t>   pOutUrbId,
                                       uint64_t        address,
                                       uint32_t        bytes) NN_NOEXCEPT
{
    Result result;
    uint32_t urbId = 0;

    if (!NN_USB_IS_DMA_ALIGNED(address))
    {
        return ResultAlignmentError();
    }

    result = m_pProtocol->PostBufferAsync(
        m_Address,
        m_ProcessHandle,
        address,
        bytes,
        &urbId
    );

    if (result.IsSuccess())
    {
        *pOutUrbId = urbId;
    }

    return result;
}

Result DsEndpointImpl::Cancel() NN_NOEXCEPT
{
    return m_pProtocol->Cancel(m_Address);
}

Result DsEndpointImpl::GetCompletionEvent(Out<NativeHandle> eventHandle) NN_NOEXCEPT
{
    nn::os::SystemEventType *event = m_pProtocol->GetCompletionEvent(m_Address);

    eventHandle.Set(
        NativeHandle(nn::os::GetReadableHandleOfSystemEvent(event), false)
    );

    return ResultSuccess();
}

Result DsEndpointImpl::GetUrbReport(Out<UrbReport> report) NN_NOEXCEPT
{
    return m_pProtocol->GetUrbReport(m_Address, report.GetPointer());
}

Result DsEndpointImpl::Stall() NN_NOEXCEPT
{
    return m_pProtocol->Stall(m_Address);
}

Result DsEndpointImpl::SetZlt(bool zlt) NN_NOEXCEPT
{
    return m_pProtocol->SetZlt(m_Address, zlt);
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
