﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*
 * @file    usb_RegisterDefinition-soc.tegra.h
 * @brief   TX1 USB related register definition
 */

namespace nn {
namespace usb {
namespace detail {

//----------------------------------------------------------------------------
//  Helper Macros
//----------------------------------------------------------------------------

// Address Map definition
#define NN_USB_DEFINE_ADDRESS_MAP(blockName, baseAddressStart, baseAddressEnd) \
    static const uintptr_t blockName##BaseAddress = baseAddressStart;          \
    static const size_t    blockName##RegionSize  = baseAddressEnd - baseAddressStart + 1;

// Register offset definition
#define NN_USB_DEFINE_REG_OFFSET(registerName, offset)                  \
    static const size_t registerName##Offset = static_cast<size_t>(offset)

// Bitfield definition
#define NN_USB_DEFINE_FIELD64(registerName, fieldName, bitStart, bitLength)               \
    static const uint32_t registerName##fieldName##Shift = (bitStart);                    \
    static const uint64_t registerName##fieldName##Mask =                                 \
    ((bitLength) < 64) ? (((1ULL << (bitLength)) - 1) << (bitStart)) : (0xFFFFFFFFFFFFFFFF << (bitStart));


#define NN_USB_DEFINE_FIELD32(registerName, fieldName, bitStart, bitLength)               \
    static const uint32_t registerName##fieldName##Shift = (bitStart);                    \
    static const uint32_t registerName##fieldName##Mask =                                 \
    ((bitLength) < 32) ? (((1ULL << (bitLength)) - 1) << (bitStart)) : (0xFFFFFFFF << (bitStart));

// Bitfield manipulation
#define NN_USB_MAKE_MASK64(fieldName,fieldValue)                                         \
    ((uint64_t)((((uint64_t)(fieldValue)) << fieldName##Shift) & fieldName##Mask))

#define NN_USB_MAKE_MASK32(fieldName,fieldValue)                                         \
    ((uint32_t)((((uint32_t)(fieldValue)) << fieldName##Shift) & fieldName##Mask))

#define NN_USB_MAKE_MASK8(_field,fieldValue)                                             \
    ((uint8_t)((((uint8_t)(fieldValue)) << fieldValue##Shift) & fieldName##Mask))

#define NN_USB_GET_FIELD64(fieldName,fieldValue)                                         \
    ((((uint64_t)(fieldValue))& fieldName##Mask) >> fieldName##Shift)

#define NN_USB_GET_FIELD32(fieldName,fieldValue)                                         \
    ((((uint32_t)(fieldValue))& fieldName##Mask) >> fieldName##Shift)

#define NN_USB_GET_FIELD8(fieldName,fieldValue)                                          \
    ((((uint8_t)(fieldValue))& fieldName##Mask) >> fieldName##Shift)

// TODO: find better names
#define NN_USB_MAKE_VALUE32_1(register, field, value)                                    \
    ((uint32_t)(                                                                         \
        (((uint32_t)(value)) << register##field##Shift) &                                \
        register##field##Mask                                                            \
    ))

#define NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2)                  \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_1(register, field1, value1) |                                \
        NN_USB_MAKE_VALUE32_1(register, field2, value2)                                  \
    ))

#define NN_USB_MAKE_VALUE32_3(register, field1, value1, field2, value2, field3, value3)  \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_1(register, field1, value1) |                                \
        NN_USB_MAKE_VALUE32_1(register, field2, value2) |                                \
        NN_USB_MAKE_VALUE32_1(register, field3, value3)                                  \
    ))

#define NN_USB_MAKE_VALUE32_4(register,                                                  \
                              field1, value1,                                            \
                              field2, value2,                                            \
                              field3, value3,                                            \
                              field4, value4)                                            \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2) |                \
        NN_USB_MAKE_VALUE32_2(register, field3, value3, field4, value4)                  \
    ))

#define NN_USB_MAKE_VALUE32_5(register,                                                  \
                              field1, value1,                                            \
                              field2, value2,                                            \
                              field3, value3,                                            \
                              field4, value4,                                            \
                              field5, value5)                                            \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2) |                \
        NN_USB_MAKE_VALUE32_2(register, field3, value3, field4, value4) |                \
        NN_USB_MAKE_VALUE32_1(register, field5, value5)                                  \
    ))

#define NN_USB_MAKE_VALUE32_6(register,                                                  \
                              field1, value1,                                            \
                              field2, value2,                                            \
                              field3, value3,                                            \
                              field4, value4,                                            \
                              field5, value5,                                            \
                              field6, value6)                                            \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2) |                \
        NN_USB_MAKE_VALUE32_2(register, field3, value3, field4, value4) |                \
        NN_USB_MAKE_VALUE32_2(register, field5, value5, field6, value6)                  \
    ))

#define NN_USB_MAKE_VALUE32_7(register,                                                  \
                              field1, value1,                                            \
                              field2, value2,                                            \
                              field3, value3,                                            \
                              field4, value4,                                            \
                              field5, value5,                                            \
                              field6, value6,                                            \
                              field7, value7)                                            \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2) |                \
        NN_USB_MAKE_VALUE32_2(register, field3, value3, field4, value4) |                \
        NN_USB_MAKE_VALUE32_2(register, field5, value5, field6, value6) |                \
        NN_USB_MAKE_VALUE32_1(register, field7, value7)                                  \
    ))

#define NN_USB_MAKE_VALUE32_8(register,                                                  \
                              field1, value1,                                            \
                              field2, value2,                                            \
                              field3, value3,                                            \
                              field4, value4,                                            \
                              field5, value5,                                            \
                              field6, value6,                                            \
                              field7, value7,                                            \
                              field8, value8)                                            \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2) |                \
        NN_USB_MAKE_VALUE32_2(register, field3, value3, field4, value4) |                \
        NN_USB_MAKE_VALUE32_2(register, field5, value5, field6, value6) |                \
        NN_USB_MAKE_VALUE32_2(register, field7, value7, field8, value8)                  \
    ))

#define NN_USB_MAKE_MASK32_1(register, field)                                            \
    ((uint32_t)register##field##Mask)

#define NN_USB_MAKE_MASK32_2(register, field1, field2)                                   \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_MASK32_1(register, field1) |                                         \
        NN_USB_MAKE_MASK32_1(register, field2)                                           \
    ))

#define NN_USB_MAKE_MASK32_3(register, field1, field2, field3)                           \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_MASK32_1(register, field1) |                                         \
        NN_USB_MAKE_MASK32_1(register, field2) |                                         \
        NN_USB_MAKE_MASK32_1(register, field3)                                           \
    ))

#define NN_USB_MAKE_MASK32_4(register, field1, field2, field3, field4)                   \
    ((uint32_t)(                                                                         \
        NN_USB_MAKE_MASK32_2(register, field1, field2) |                                 \
        NN_USB_MAKE_MASK32_2(register, field3, field4)                                   \
    ))

#define NN_USB_CHECK_FIELD32_1(register, value,                                          \
                               field1,   value1)                                         \
    ((value & NN_USB_MAKE_MASK32_1(register, field1)) ==                                 \
     NN_USB_MAKE_VALUE32_1(register, field1, value1))

#define NN_USB_CHECK_FIELD32_2(register, value,                                          \
                               field1,   value1,                                         \
                               field2,   value2)                                         \
    ((value & NN_USB_MAKE_MASK32_2(register, field1, field2)) ==                         \
     NN_USB_MAKE_VALUE32_2(register, field1, value1, field2, value2))

#define NN_USB_SET_FIELD32_1(register, var,                                              \
                             field1,   value1)                                           \
    do {                                                                                 \
        uint32_t _mask  = NN_USB_MAKE_MASK32_1(register, field1);                        \
        uint32_t _value = NN_USB_MAKE_VALUE32_1(register,                                \
                                                field1, value1);                         \
        (var) = ((var) & ~_mask) | _value;                                               \
    } while(false)

#define NN_USB_SET_FIELD32_2(register, var,                                              \
                             field1,   value1,                                           \
                             field2,   value2)                                           \
    do {                                                                                 \
        uint32_t _mask  = NN_USB_MAKE_MASK32_2(register, field1, field2);                \
        uint32_t _value = NN_USB_MAKE_VALUE32_2(register,                                \
                                                field1, value1,                          \
                                                field2, value2);                         \
        (var) = ((var) & ~_mask) | _value;                                               \
    } while(false)

#define NN_USB_SET_FIELD32_3(register, var,                                              \
                             field1,   value1,                                           \
                             field2,   value2,                                           \
                             field3,   value3)                                           \
    do {                                                                                 \
        uint32_t _mask  = NN_USB_MAKE_MASK32_3(register, field1, field2, field3);        \
        uint32_t _value = NN_USB_MAKE_VALUE32_3(register,                                \
                                                field1, value1,                          \
                                                field2, value2,                          \
                                                field3, value3);                         \
        (var) = ((var) & ~_mask) | _value;                                               \
    } while(false)

//----------------------------------------------------------------------------
// System Address Map
//----------------------------------------------------------------------------

NN_USB_DEFINE_ADDRESS_MAP( Fuse,       0x7000f800, 0x7000fbff );
NN_USB_DEFINE_ADDRESS_MAP( XusbPadctl, 0x7009f000, 0x7009ffff );
NN_USB_DEFINE_ADDRESS_MAP( XusbHost,   0x70090000, 0x70099fff );
NN_USB_DEFINE_ADDRESS_MAP( XusbDev,    0x700d0000, 0x700d9fff );

//----------------------------------------------------------------------------
// USB related fuses (NSBG-8267)
//----------------------------------------------------------------------------

NN_USB_DEFINE_REG_OFFSET ( FuseUsbCalib0, 0x1f0 );

NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, HsCurrLevel3,  23, 6 ); // HS_CURR_LEVEL for USB Port 3
NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, HsCurrLevel2,  17, 6 ); // HS_CURR_LEVEL for USB Port 2
NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, HsCurrLevel1,  11, 6 ); // HS_CURR_LEVEL for USB Port 1
NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, TermRangeAdj0,  7, 4 ); // TERM_RANGE_ADJ for Port 0 (Mariko)
NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, TermRangeAdj,   7, 4 ); // TERM_RANGE_ADJ for ALL Ports (TX1)
NN_USB_DEFINE_FIELD32    ( FuseUsbCalib0, HsCurrLevel0,   0, 6 ); // HS_CURR_LEVEL for USB Port 0

NN_USB_DEFINE_REG_OFFSET ( FuseUsbCalibExt0,  0x350 );

NN_USB_DEFINE_FIELD32    ( FuseUsbCalibExt0, TermRangeAdj3, 13, 4 ); // TERM_RANGE_ADJ for Port 3 (Mariko)
NN_USB_DEFINE_FIELD32    ( FuseUsbCalibExt0, TermRangeAdj2,  9, 4 ); // TERM_RANGE_ADJ for Port 2 (Mariko)
NN_USB_DEFINE_FIELD32    ( FuseUsbCalibExt0, TermRangeAdj1,  5, 4 ); // TERM_RANGE_ADJ for Port 1 (Mariko)
NN_USB_DEFINE_FIELD32    ( FuseUsbCalibExt0, RpdCtrl,        0, 5 ); // RPD_CTRL for all USB ports

//----------------------------------------------------------------------------
// USB1 Controller Registers (Base: USB)
//----------------------------------------------------------------------------

/**
 * @brief 23.16.1.18 USB2_CONTROLLER_USB2D_USBCMD_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dUsbcmd, 0x130);

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbcmd, Itc,   16, 8 ); // Interrupt Threshold Control
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbcmd, Atdtw, 14, 1 ); // Add DTD Tripwire
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbcmd, Sutw,  13, 1 ); // Setup Tripwire
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbcmd, Rst,    1, 1 ); // Controller Reset
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbcmd, Rs,     0, 1 ); // Run / Stop

/**
 * @brief 23.16.1.19 USB2_CONTROLLER_USB2D_USBSTS_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dUsbsts, 0x134);

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Sli, 8, 1 ); // DCSuspend
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Sri, 7, 1 ); // SOF Received
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Uri, 6, 1 ); // USB Reset Received
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Pci, 2, 1 ); // Port Change Detect
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Uei, 1, 1 ); // USB Error Interrupt
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbsts, Ui,  0, 1 ); // USB Interrupt

/**
 * @brief 23.16.1.20 USB2_CONTROLLER_USB2D_USBINTR_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dUsbintr, 0x138);

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Sle, 8, 1 ); // Sleep Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Sre, 7, 1 ); // SOF Received Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Ure, 6, 1 ); // USB Reset Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Pce, 2, 1 ); // Port Change Detect Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Uee, 1, 1 ); // USB Error Interrupt Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbintr, Ue,  0, 1 ); // USB Interrupt Enable

/**
 * @brief 23.16.1.22 USB2_CONTROLLER_USB2D_PERIODICLISTBASE_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dPeriodiclistbase, 0x144 );

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dPeriodiclistbase, Usbadr, 25, 7 ); // Device Address

/**
 * @brief 23.16.1.23 USB2_CONTROLLER_USB2D_ASYNCLISTADDR_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dAsynclistaddr, 0x148 );

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dAsynclistaddr, Epbase, 11, 21 ); // dQH List Base Address

/**
 * @brief 23.16.1.29 USB2_CONTROLLER_USB2D_HOSTPC1_DEVLC_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dHostpc1Devlc, 0x1b4 );

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dHostpc1Devlc, Pts, 29, 3 ); // Parallel Transceiver Select
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dHostpc1Devlc, Sts, 28, 1 ); // Serial Transceiver Not Selected

/**
 * @brief 23.16.1.31 USB2_CONTROLLER_USB2D_USBMODE_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dUsbmode, 0x1f8 );

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbmode, Slom, 3, 1 ); // Setup Lockout Mode
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dUsbmode, Cm,   0, 2 ); // Controller Mode

/**
 * @brief 23.16.1.34 USB2_CONTROLLER_USB2D_ENDPTSETUPSTAT_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dEndptsetupstat, 0x208 );

/**
 * @brief 23.16.1.35 USB2_CONTROLLER_USB2D_ENDPTPRIME_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dEndptprime, 0x20c );

/**
 * @brief 23.16.1.36 USB2_CONTROLLER_USB2D_ENDPTFLUSH_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dEndptflush, 0x210 );

/**
 * @brief 23.16.1.37 USB2_CONTROLLER_USB2D_ENDPTSTATUS_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dEndptstatus, 0x214 );

/**
 * @brief 23.16.1.38 USB2_CONTROLLER_USB2D_ENDPTCOMPLETE_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb2ControllerUsb2dEndptcomplete, 0x218 );

/**
 * @brief 23.16.1.39 USB2_CONTROLLER_USB2D_ENDPTCTRL0_0
 *        23.16.1.40 USB2_CONTROLLER_USB2D_ENDPTCTRLn_0
 */
static const size_t Usb2ControllerUsb2dEndptctrlOffset[] = {
    0x220 + ( 0 - 1) * 0x04,  // Ep0
    0x220 + ( 1 - 1) * 0x04,  // Ep1
    0x220 + ( 2 - 1) * 0x04,  // Ep2
    0x220 + ( 3 - 1) * 0x04,  // Ep3
    0x220 + ( 4 - 1) * 0x04,  // Ep4
    0x220 + ( 5 - 1) * 0x04,  // Ep5
    0x220 + ( 6 - 1) * 0x04,  // Ep6
    0x220 + ( 7 - 1) * 0x04,  // Ep7
    0x220 + ( 8 - 1) * 0x04,  // Ep8
    0x220 + ( 9 - 1) * 0x04,  // Ep9
    0x220 + (10 - 1) * 0x04,  // Ep10
    0x220 + (11 - 1) * 0x04,  // Ep11
    0x220 + (12 - 1) * 0x04,  // Ep12
    0x220 + (13 - 1) * 0x04,  // Ep13
    0x220 + (14 - 1) * 0x04,  // Ep14
    0x220 + (15 - 1) * 0x04   // Ep15
};

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Txe, 23, 1 ); // TX Endpoint Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Txr, 22, 1 ); // TX Data Toggle Reset
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Txi, 21, 1 ); // TX Data Toggle Inhibit
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Txt, 18, 2 ); // TX Endpoint Type
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Txs, 16, 1 ); // TX Endpoint Stall

NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Rxe,  7, 1 ); // RX Endpoint Enable
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Rxr,  6, 1 ); // RX Data Toggle Reset
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Rxi,  5, 1 ); // RX Data Toggle Inhibit
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Rxt,  2, 2 ); // RX Endpoint Type
NN_USB_DEFINE_FIELD32    ( Usb2ControllerUsb2dEndptctrl, Rxs,  0, 1 ); // RX Endpoint Stall

//----------------------------------------------------------------------------
// USB1 Controller Interface Registers (Base: USB)
//----------------------------------------------------------------------------

/**
 * @brief 23.16.2.1 USB1_IF_USB_SUSP_CTRL_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1IfUsbSuspCtrl, 0x400 );

NN_USB_DEFINE_FIELD32    ( Usb1IfUsbSuspCtrl, UtmipPhyEnb,    12, 1 ); // Enable UTMIP PHY mode
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbSuspCtrl, UtmipReset,     11, 1 ); // Reset UTMIP PHY
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbSuspCtrl, UsbPhyClkValid,  7, 1 ); // USB PHY clock valid status

/**
 * @brief 23.16.2.2 USB1_IF_USB_PHY_VBUS_SENSORS_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1IfUsbPhyVbusSensors, 0x404 );

NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusSensors, BSessVldSwValue, 12, 1 ); // B_SESS_VLD software value
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusSensors, BSessVldSwEn,    11, 1 ); // B_SESS_VLD software enable

/**
 * @brief 23.16.2.3 USB1_IF_USB_PHY_VBUS_WAKEUP_ID_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1IfUsbPhyVbusWakeupId, 0x408 );

NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusWakeupId, VdatDetChgDet, 17, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusWakeupId, VdatDetSts,    18, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusWakeupId, VdcdDetChgDet, 25, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1IfUsbPhyVbusWakeupId, VdcdDetSts,    26, 1 );

//----------------------------------------------------------------------------
// USB1 UTMIP Configuration Registers (Base: USB)
//----------------------------------------------------------------------------

/**
 * @brief 23.16.3.1 USB1_UTMIP_XCVR_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipXcvrCfg0, 0x808 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipXcvrHsslewMsb,      25, 7 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipXcvrSetupMsb,       22, 3 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipXcvrLsbiasSel,      21, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipForcePdziPowerdown, 18, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipForcePd2Powerdown,  16, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipForcePdPowerdown,   14, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg0, UtmipXcvrSetup,           0, 4 );

/**
 * @brief 23.16.3.2 USB1_UTMIP_BIAS_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipBiasCfg0, 0x80c );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipBiasCfg0, UtmipOtgpd,  11, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBiasCfg0, UtmipBiaspd, 10, 1 );

/**
 * @brief 23.16.3.3 USB1_UTMIP_HSRX_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipHsrxCfg0, 0x810 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipHsrxCfg0, UtmipIdleWait,     15, 5);
NN_USB_DEFINE_FIELD32    ( Usb1UtmipHsrxCfg0, UtmipElasticLimit, 10, 5);

/**
 * @brief 23.16.3.4 USB1_UTMIP_HSRX_CFG1_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipHsrxCfg1, 0x814 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipHsrxCfg1, UtmipHsSyncStartDly, 1, 5);

/**
 * @brief 23.16.3.5 USB1_UTMIP_FSLSRX_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipFslsrxCfg0, 0x818 );

/**
 * @brief 23.16.3.6 USB1_UTMIP_FSLSRX_CFG1_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipFslsrxCfg1, 0x81c );

/**
 * @brief 23.16.3.7 USB1_UTMIP_TX_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipTxCfg0, 0x820 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipTxCfg0, UtmipFsPreambleJ, 19, 1 );

/**
 * @brief 23.16.3.8 USB1_UTMIP_MISC_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipMiscCfg0, 0x824 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipMiscCfg0, UtmipSuspendExitOnEdge, 22, 1 );

/**
 * @brief 23.16.3.9 USB1_UTMIP_MISC_CFG1_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipMiscCfg1, 0x828 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipMiscCfg1, UtmipPhyXtalClocken, 30, 1 );

/**
 * @brief 23.16.3.10 USB1_UTMIP_DEBOUNCE_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipDebounceCfg0, 0x82c );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipDebounceCfg0, UtmipBiasDebounceB, 16, 16 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipDebounceCfg0, UtmipBiasDebounceA,  0, 16 );

/**
 * @brief 23.16.3.11 USB1_UTMIP_BAT_CHRG_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipBatChrgCfg0, 0x830 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipPdChrg,   0, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipOpSinkEn, 1, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipOnSinkEn, 2, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipOpSrcEn,  3, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipOnSrcEn,  4, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipBatChrgCfg0, UtmipOpISrcEn, 5, 1 );

/**
 * @brief 23.16.3.12 USB1_UTMIP_SPARE_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipSpareCfg0, 0x834 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipSpareCfg0, FuseSetupSel, 3, 1 );

/**
 * @brief 23.16.3.13 USB1_UTMIP_XCVR_CFG1_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipXcvrCfg1, 0x838 );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg1, UtmipXcvrTermRangeAdj,    18, 4 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg1, UtmipForcePddrPowerdown,   4, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg1, UtmipForcePdchrpPowerdown, 2, 1 );
NN_USB_DEFINE_FIELD32    ( Usb1UtmipXcvrCfg1, UtmipForcePddiscPowerdown, 0, 1 );

/**
 * @brief 23.16.3.14 USB1_UTMIP_BIAS_CFG1_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipBiasCfg1, 0x83c );

NN_USB_DEFINE_FIELD32    ( Usb1UtmipBiasCfg1, UtmipBiasPdtrkCount, 3, 5);

/**
 * @brief 23.16.3.15 USB1_UTMIP_BIAS_STS0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipBiasSts0, 0x840 );

/**
 * @brief 23.16.3.16 USB1_UTMIP_CHRG_DEB_CFG0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipChrgDebCfg0, 0x844 );

/**
 * @brief 23.16.3.17 USB1_UTMIP_MISC_STS0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipMiscSts0, 0x848 );

/**
 * @brief 23.16.3.18 USB1_UTMIP_PMC_WAKEUP0_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipPmcWakeup0, 0x84c );

/**
 * @brief 23.16.3.19 USB1_UTMIP_BIAS_CFG2_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipBiasCfg2, 0x850 );

/**
 * @brief 23.16.3.20 USB1_UTMIP_XCVR_CFG2_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipXcvrCfg2, 0x854 );

/**
 * @brief 23.16.3.21 USB1_UTMIP_XCVR_CFG3_0
 */
NN_USB_DEFINE_REG_OFFSET ( Usb1UtmipXcvrCfg3, 0x858 );


//----------------------------------------------------------------------------
// XUSB PADCTL Registers (Base: XusbPadctl)
//----------------------------------------------------------------------------

/**
 * @brief 23.16.7.2 XUSB_PADCTL_USB2_PAD_MUX_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2PadMux, 0x004 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2P0,           0, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2P1,           2, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2P2,           4, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2P3,           6, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2HsicP0,      14, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2HsicP1,      15, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, HsicPadTrk,      16, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, Usb2BiasPad,     18, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, HsicPort0Config, 20, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PadMux, HsicPort1Config, 21, 1 );

/**
 * @brief 23.16.7.3 XUSB_PADCTL_USB2_PORT_CAP_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2PortCap, 0x008 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PortCap, Port0Cap,  0, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PortCap, Port1Cap,  4, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PortCap, Port2Cap,  8, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2PortCap, Port3Cap, 12, 2 );

const int32_t XusbPadctlUsb2PortCapDisabled   = 0;
const int32_t XusbPadctlUsb2PortCapHostOnly   = 1;
const int32_t XusbPadctlUsb2PortCapDeviceOnly = 2;
const int32_t XusbPadctlUsb2PortCapOtgCap     = 3;

/**
 * @brief 23.16.7.5 XUSB_PADCTL_USB2_OC_MAP_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OcMap, 0x010 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OcMap, Port0OcPin,  0, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OcMap, Port1OcPin,  4, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OcMap, Port2OcPin,  8, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OcMap, Port3OcPin, 12, 4 );

const uint32_t XusbPadctlUsb2OcMapDetectionDisabled = 15;

/**
 * @brief 23.16.7.6 XUSB_PADCTL_SS_PORT_MAP_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlSsPortMap, 0x014 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 0,          0, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 0Internal,  4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 1,          5, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 1Internal,  9, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 2,         10, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 2Internal, 14, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 3,         15, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlSsPortMap, 3Internal, 19, 1 );

const uint32_t XusbPadctlSsPortMapDisabled = 7;

/**
 * @brief 23.16.7.7 XUSB_PADCTL_VBUS_OC_MAP_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlVbusOcMap, 0x018 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlVbusOcMap, Enable0, 0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlVbusOcMap, 0,       1, 4 );

const uint32_t XusbPadctlVbusOcMapDetectionDisabled = 15;

/**
 * @brief 23.16.7.8 XUSB_PADCTL_OC_DET_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlOcDet, 0x01c );

/**
 * @brief 23.16.7.9 XUSB_PADCTL_ELPG_PROGRAM_0_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlElpgProgram0, 0x020 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port0WakeIntEnable,      0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port1WakeIntEnable,      1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port2WakeIntEnable,      2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port3WakeIntEnable,      3, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port0WakeupEvent,        7, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port1WakeupEvent,        8, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port2WakeupEvent,        9, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2Port3WakeupEvent,       10, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort0WakeIntEnable,       14, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort1WakeIntEnable,       15, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort2WakeIntEnable,       16, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort3WakeIntEnable,       17, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort0WakeupEvent,         21, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort1WakeupEvent,         22, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort2WakeupEvent,         23, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, SsPort3WakeupEvent,         24, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2HsicPort0WakeIntEnable, 28, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2HsicPort1WakeIntEnable, 29, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2HsicPort0WakeupEvent,   30, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram0, Usb2HsicPort1WakeupEvent,   31, 1 );

/**
 * @brief 23.16.7.10 XUSB_PADCTL_ELPG_PROGRAM_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlElpgProgram1, 0x024 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp0ElpgClampEn,        0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp0ElpgClampEnEarly,   1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp0ElpgVcoreDown,      2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp1ElpgClampEn,        3, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp1ElpgClampEnEarly,   4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp1ElpgVcoreDown,      5, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp2ElpgClampEn,        6, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp2ElpgClampEnEarly,   7, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp2ElpgVcoreDown,      8, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp3ElpgClampEn,        9, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp3ElpgClampEnEarly,  10, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, Ssp3ElpgVcoreDown,     11, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, AuxMuxLp0ClampEn,      29, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, AuxMuxLp0ClampEnEarly, 30, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlElpgProgram1, AuxMuxLp0VcoreDown,    31, 1 );

/**
 * @brief 23.16.7.11 XUSB_PADCTL_USB3_PAD_MUX_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb3PadMux, 0x028 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask0,  1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask1,  2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask2,  3, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask3,  4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask4,  5, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask5,  6, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadIddqDisableMask6,  7, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, SataPadIddqDisableMask0,  8, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane0,            12, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane1,            14, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane2,            16, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane3,            18, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane4,            20, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane5,            22, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, PciePadLane6,            24, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb3PadMux, SataPadLane0,            30, 2 );

/**
 * @brief 23.16.7.15 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD0_CTL0_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, 0x80 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, PdChg,           0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdcdDet,         1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdcdDetStChng,   2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdcdDetFilterEn, 4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdatDet,         5, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdatDetStChng,   6, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, VdatDetFilterEn, 8, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, OpSinkEn,        9, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, OpSrcEn,        10, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, OnSinkEn,       11, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, OnSrcEn,        12, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, OpISrcEn,       13, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, Zip,            18, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, ZipFilterEn,    21, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, Zin,            22, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, ZinFilterEn,    25, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, DcdDetected,    26, 1 );

/**
 * @brief 23.16.7.16 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD0_CTL1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, 0x84 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VonDiv2p0Det,     0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VonDiv2p7Det,     1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VopDiv2p0Det,     2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VopDiv2p7Det,     3, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, DivDetEn,         4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VregFix18,        6, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VregLev,          7, 2 );
const uint8_t XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev500mA  = 0;
const uint8_t XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev900mA  = 1;
const uint8_t XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev1500mA = 2;
const uint8_t XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev2000mA = 3;
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, VregDynDly,       9, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbopRpdOvrd,    16, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbopRpdOvrdVal, 17, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbopRpuOvrd,    18, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbopRpuOvrdVal, 19, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbonRpdOvrd,    20, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbonRpdOvrdVal, 21, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbonRpuOvrd,    22, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, UsbonRpuOvrdVal, 23, 1 );

/**
 * @brief 23.16.7.17 XUSB_PADCTL_USB2_OTG_PAD0_CTL_0_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad0Ctl0, 0x88 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, HsCurrLevel,  0, 6 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, HsSlew,       6, 3 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, FsRslew,      9, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, FsFslew,     13, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, LsRslew,     17, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, LsFslew,     21, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, TermSel,     25, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, Pd,          26, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, Pd2,         27, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, Pd2OvrdEn,   28, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl0, PdZi,        29, 1 );

/**
 * @brief 23.16.7.18 XUSB_PADCTL_USB2_OTG_PAD0_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad0Ctl1, 0x8c );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PdChrpOvrd,         0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PdDiscOvrd,         1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PdDr,               2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, TermRangeAdj,       3, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, HsCoupEn,          11, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, RpuRangeAdj,       13, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PdChrpOvrdVal,     15, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PdDiscOvrdVal,     16, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, PtermRangeAdj,     17, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, HsLoopbackOvrdEn,  21, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, HsLoopbackOvrdVal, 22, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, RpuSwitchOvrd,     23, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, RpuSwitchLow,      24, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, RpuStatusHigh,     25, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2OtgPad0Ctl1, RpdCtrl,           26, 5 );

/**
 * @brief 23.16.7.21 XUSB_PADCTL_USB2_OTG_PAD1_CTL_0_0
 *        23.16.7.22 XUSB_PADCTL_USB2_OTG_PAD1_CTL_1_0
 *        23.16.7.25 XUSB_PADCTL_USB2_OTG_PAD2_CTL_0_0
 *        23.16.7.26 XUSB_PADCTL_USB2_OTG_PAD2_CTL_1_0
 *        23.16.7.29 XUSB_PADCTL_USB2_OTG_PAD3_CTL_0_0
 *        23.16.7.30 XUSB_PADCTL_USB2_OTG_PAD3_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad1Ctl0, 0x0c8 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad1Ctl1, 0x0cc );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad2Ctl0, 0x108 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad2Ctl1, 0x10c );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad3Ctl0, 0x148 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2OtgPad3Ctl1, 0x14c );

static const size_t XusbPadctlUsb2OtgPadCtl0Offsets[] = {
    XusbPadctlUsb2OtgPad0Ctl0Offset,
    XusbPadctlUsb2OtgPad1Ctl0Offset,
    XusbPadctlUsb2OtgPad2Ctl0Offset,
    XusbPadctlUsb2OtgPad3Ctl0Offset
};
static const size_t XusbPadctlUsb2OtgPadCtl1Offsets[] = {
    XusbPadctlUsb2OtgPad0Ctl1Offset,
    XusbPadctlUsb2OtgPad1Ctl1Offset,
    XusbPadctlUsb2OtgPad2Ctl1Offset,
    XusbPadctlUsb2OtgPad3Ctl1Offset
};

/**
 * @brief 23.16.7.19 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD1_CTL0_0
 *        23.16.7.15 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD1_CTL1_0
 *        23.16.7.23 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD2_CTL0_0
 *        23.16.7.24 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD2_CTL1_0
 *        23.16.7.27 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD3_CTL0_0
 *        23.16.7.28 XUSB_PADCTL_USB2_BATTERY_CHRG_OTGPAD3_CTL1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad1Ctl0, 0x0c0 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad1Ctl1, 0x0c4 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad2Ctl0, 0x100 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad2Ctl1, 0x104 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad3Ctl0, 0x140 );
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgOtgpad3Ctl1, 0x144 );

static const size_t XusbPadctlUsb2BatteryChrgOtgpadCtl0Offsets[] = {
    XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset,
    XusbPadctlUsb2BatteryChrgOtgpad1Ctl0Offset,
    XusbPadctlUsb2BatteryChrgOtgpad2Ctl0Offset,
    XusbPadctlUsb2BatteryChrgOtgpad3Ctl0Offset
};
static const size_t XusbPadctlUsb2BatteryChrgOtgpadCtl1Offsets[] = {
    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset,
    XusbPadctlUsb2BatteryChrgOtgpad1Ctl1Offset,
    XusbPadctlUsb2BatteryChrgOtgpad2Ctl1Offset,
    XusbPadctlUsb2BatteryChrgOtgpad3Ctl1Offset
};

/**
 * @brief 23.16.7.31 XUSB_PADCTL_USB2_BATTERY_CHRG_TDCD_DBNC_TIMER_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BatteryChrgTdcdDbncTimer, 0x280);

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgTdcdDbncTimer, TdcdDbnc,   0, 11 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BatteryChrgTdcdDbncTimer, IddigDbnc, 11,  6 );

/**
 * @brief 23.16.7.32 XUSB_PADCTL_USB2_BIAS_PAD_CTL_0_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BiasPadCtl0, 0x284 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl0, HsSquelchLevel, 0, 3 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl0, HsDisconLevel,  3, 3 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl0, TermOffset,     8, 3 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl0, Pd,            11, 1 );

/**
 * @brief 23.16.7.33 XUSB_PADCTL_USB2_BIAS_PAD_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2BiasPadCtl1, 0x288 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, Tctrl,              0, 6 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, Pctrl,              6, 6 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, TrkStartTimer,     12, 7 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, TrkDoneResetTimer, 19, 7 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, PdTrk,             26, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, TrkStart,          27, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, TrkDone,           28, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, TrkSwOvrd,         29, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2BiasPadCtl1, ForceTrkClkEn,     30, 1 );

/**
 * @brief 23.16.7.53 XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP0Ctl1, 0x460 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, Tctrl,              0, 6 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, Pctrl,              6, 6 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, TrkStartTimer,     12, 7 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, TrkDoneResetTimer, 19, 7 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, PdTrk,             26, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, TrkStart,          27, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, TrkDone,           28, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, TrkSwOvrd,         29, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl1, ForceTrkClkEn,     30, 1 );

/**
 * @brief 23.16.7.54 XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP0Ctl2, 0x464 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, TxIddq,     0, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, RxIddq,     8, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, TxIddqOvrd, 1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, RxIddqOvrd, 9, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, TxSleep,    4, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, RxSleep,   12, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, TxPwrOvrd, 24, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, RxPwrOvrd, 25, 1 );

/**
 * @brief 23.16.7.62 XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP1Ctl1, 0x4A0 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, AuxRxIdleTh, 24, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyMiscPadP0Ctl2, AuxRxIdleEn, 22, 1 );

/**
 * @brief 23.16.7.63 XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP1Ctl2, 0x4A4 );

/**
 * @brief 23.16.7.71 XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP2Ctl1, 0x4E0 );

/**
 * @brief 23.16.7.72 XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP2Ctl2, 0x4E4 );

/**
 * @brief 23.16.7.80 XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP3Ctl1, 0x520 );

/**
 * @brief 23.16.7.81 XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP3Ctl2, 0x524 );

/**
 * @brief 23.16.7.89 XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP4Ctl1, 0x560 );

/**
 * @brief 23.16.7.90 XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP4Ctl2, 0x564 );

/**
 * @brief 23.16.7.99 XUSB_PADCTL_UPHY_MISC_PAD_P5_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP5Ctl2, 0x5A4 );

/**
 * @brief 23.16.7.108 XUSB_PADCTL_UPHY_MISC_PAD_P6_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadP6Ctl2, 0x5E4 );

/**
 * @brief 23.16.7.128 XUSB_PADCTL_UPHY_MISC_PAD_S0_CTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyMiscPadS0Ctl2, 0x964 );

/**
 * @brief 23.16.7.136 XUSB_PADCTL_UPHY_USB3_PAD0_ECTL_1_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyUsb3Ectl1, 0xa60 );


/**
 * @brief 23.16.7.137 XUSB_PADCTL_UPHY_USB3_PAD0_ECTL_2_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyUsb3Ectl2, 0xa64 );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyUsb3Ectl2, RxCtle, 0, 16 );

/**
 * @brief 23.16.7.138 XUSB_PADCTL_UPHY_USB3_PAD0_ECTL_3_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyUsb3Ectl3, 0xa68 );

/**
 * @brief 23.16.7.139 XUSB_PADCTL_UPHY_USB3_PAD0_ECTL_4_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyUsb3Ectl4, 0xa6c );

NN_USB_DEFINE_FIELD32    ( XusbPadctlUphyUsb3Ectl4, RxCdrCtrl, 16, 16 );

/**
 * @brief 23.16.7.141 XUSB_PADCTL_UPHY_USB3_PAD0_ECTL_6_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUphyUsb3Ectl6, 0xa74 );

/**
 * @brief 23.16.7.164 XUSB_PADCTL_USB2_VBUS_ID_0
 */
NN_USB_DEFINE_REG_OFFSET ( XusbPadctlUsb2VbusId0, 0xc60 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, OtgVbusSessVld,           0, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, OtgVbusSessVldStChng,     1, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, OtgVbusSessVldChngIntrEn, 2, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusValid,                3, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusValidStChng,          4, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusValidChngIntrEn,      5, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, Iddig,                    6, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IddigA,                   7, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IddigB,                   8, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IddigC,                   9, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IddigStChng,             10, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IddigChngIntrEn,         11, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusSourceSelect,        12, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusOverride,            14, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusWakeupOverride,      15, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IdSourceSelect,          16, 2 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, IdOverride,              18, 4 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusWakeup,              22, 1 ); // RO
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusWakeupStChng,        23, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, VbusWakeupChngIntrEn,    24, 1 );
NN_USB_DEFINE_FIELD32    ( XusbPadctlUsb2VbusId0, FullRegister,  0, 31 );

static const size_t XusbPadctlUphyUsb3EctlBlockSize = 0x40;


enum SsPhyPad
{
    SsPhyPad_PexP0 = 0,
    SsPhyPad_PexP1,
    SsPhyPad_PexP2,
    SsPhyPad_PexP3,
    SsPhyPad_PexP4,
    SsPhyPad_PexP5,
    SsPhyPad_PexP6,
    SsPhyPad_SataS0
};

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
