﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_RegisterBlock.h
 * @brief   Block of registers
 */

#include <nn/os.h>
#include <nn/dd.h>

#include "usb_Util.h"
#include "usb_RegisterDefinition-soc.tegra.h"

namespace nn {
namespace usb {
namespace detail {

class RegisterBlock
{
public:
    RegisterBlock() NN_NOEXCEPT
        : m_BasePhysicalAddress(0)
        , m_BaseVirtualAddress(0)
        , m_Size(0)
    {
    }

    ~RegisterBlock() NN_NOEXCEPT
    {
    }

    void Initialize(uintptr_t basePhysicalAddress, size_t size) NN_NOEXCEPT
    {
        NN_USB_ABORT_UNLESS(size > 0);

        size_t offset         = static_cast<size_t>(basePhysicalAddress & 0xFFF);
        m_BasePhysicalAddress = basePhysicalAddress;
        m_Size                = size;
        m_BaseVirtualAddress  = nn::dd::QueryIoMappingAddress(
            static_cast<uintptr_t>(m_BasePhysicalAddress - offset),
            size + offset
        ) + offset;

        NN_USB_ABORT_UNLESS(m_BaseVirtualAddress != 0);
    }

    void Initialize(RegisterBlock* pParent, size_t offset, size_t size) NN_NOEXCEPT
    {
        NN_USB_ABORT_UNLESS(pParent->IsInitialized());
        NN_USB_ABORT_UNLESS(size > 0);

        // Don't abort for the condition below: some times we abuse the API
        // NN_USB_ABORT_UNLESS(offset + size <= pParent->GetSize());

        m_BasePhysicalAddress = pParent->GetBasePa() + offset;
        m_BaseVirtualAddress  = pParent->GetBaseVa() + offset;
        m_Size                = size;
    }

    void Finalize() NN_NOEXCEPT
    {
        m_Size                = 0;
        m_BasePhysicalAddress = 0;
        m_BaseVirtualAddress  = 0;
    }

    bool IsInitialized() NN_NOEXCEPT
    {
        return (m_Size > 0);
    }

    /*
     * 32-bit accessors
     */
    uint32_t Read32(size_t offset) NN_NOEXCEPT
    {
        return *(reinterpret_cast<volatile uint32_t*>(m_BaseVirtualAddress + offset));
    }

    uint64_t Read64(size_t offset) NN_NOEXCEPT
    {
        return *(reinterpret_cast<volatile uint64_t*>(m_BaseVirtualAddress + offset));
    }

    void Write32(size_t offset, uint32_t value) NN_NOEXCEPT
    {
        *(reinterpret_cast<volatile uint32_t*>(m_BaseVirtualAddress + offset)) = value;
    }

    void Write64(size_t offset, uint64_t value) NN_NOEXCEPT
    {
        *(reinterpret_cast<volatile uint64_t*>(m_BaseVirtualAddress + offset)) = value;
    }

    void SetField32(size_t offset, uint32_t setMask) NN_NOEXCEPT
    {
        Write32(offset, Read32(offset) | setMask);
    }

    void ClearField32(size_t offset, uint32_t clearMask) NN_NOEXCEPT
    {
        Write32(offset, Read32(offset) & ~clearMask);
    }

    void SetField32(size_t offset, uint32_t clearMask, uint32_t setMask) NN_NOEXCEPT
    {
        Write32(offset, (Read32(offset) & ~clearMask) | setMask);
    }

    bool TestField32(size_t offset, uint32_t testedMask) NN_NOEXCEPT
    {
        return static_cast<bool>(Read32(offset) & testedMask);
    }

    bool TestField32(size_t offset, uint32_t testedMask, uint32_t expectedValue) NN_NOEXCEPT
    {
        return static_cast<bool>((Read32(offset) & testedMask) == (expectedValue & testedMask));
    }

    bool PollField32(size_t offset, uint32_t testedMask, uint32_t expectedValue,
                     nn::TimeSpan maxWaitTime, int32_t maxRetryCount)
    {
        bool isExpectedValue = false;
        int32_t attempts = 0;
        os::Tick timeoutTick = nn::os::ConvertToTick(maxWaitTime) + nn::os::GetSystemTick();

        do
        {
            if((isExpectedValue = TestField32(offset, testedMask, expectedValue)))
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(maxWaitTime.GetNanoSeconds() / maxRetryCount));

            if((isExpectedValue = TestField32(offset, testedMask, expectedValue)))
            {
                break;
            }
        } while((attempts++ < maxRetryCount) && ((nn::os::GetSystemTick() <= timeoutTick)));

        return isExpectedValue;
    }

    void SetBit32(size_t offset, int32_t zeroBasedBitNumber)
    {
        Write32(offset, (Read32(offset) | (1 << zeroBasedBitNumber)));
    }

    void ClearBit32(size_t offset, int32_t zeroBasedBitNumber)
    {
        Write32(offset, (Read32(offset) & ~static_cast<uint32_t>(1 << zeroBasedBitNumber)));
    }

    bool TestBit32(size_t offset, int32_t zeroBasedBitNumber) NN_NOEXCEPT
    {
        return static_cast<bool>(Read32(offset) & (1 << zeroBasedBitNumber));
    }

    /*
     * Register block attributes
     */
    size_t GetSize() NN_NOEXCEPT
    {
        return m_Size;
    }

    uintptr_t GetBaseVa() NN_NOEXCEPT
    {
        return m_BaseVirtualAddress;
    }

    uintptr_t GetBasePa() NN_NOEXCEPT
    {
        return m_BasePhysicalAddress;
    }

private:
    uintptr_t      m_BasePhysicalAddress;
    uintptr_t      m_BaseVirtualAddress;
    size_t         m_Size;
};

#define SET_FIELD32_1(register,                                                            \
                      field, value)                                                        \
    SetField32(                                                                            \
        ::nn::usb::detail::register##Offset,                                               \
        NN_USB_MAKE_MASK32_1(::nn::usb::detail::register, field),                          \
        NN_USB_MAKE_VALUE32_1(::nn::usb::detail::register,                                 \
                              field, value)                                                \
    )

#define SET_FIELD32_2(register,                                                            \
                      field1, value1,                                                      \
                      field2, value2)                                                      \
    SetField32(                                                                            \
        nn::usb::detail::register##Offset,                                                 \
        NN_USB_MAKE_MASK32_2(::nn::usb::detail::register, field1, field2),                 \
        NN_USB_MAKE_VALUE32_2(::nn::usb::detail::register,                                 \
                              field1, value1,                                              \
                              field2, value2)                                              \
    )

#define SET_FIELD32_3(register,                                                            \
                      field1, value1,                                                      \
                      field2, value2,                                                      \
                      field3, value3)                                                      \
    SetField32(                                                                            \
        ::nn::usb::detail::register##Offset,                                               \
        NN_USB_MAKE_MASK32_3(::nn::usb::detail::register, field1, field2, field3),         \
        NN_USB_MAKE_VALUE32_3(::nn::usb::detail::register,                                 \
                              field1, value1,                                              \
                              field2, value2,                                              \
                              field3, value3)                                              \
    )

#define SET_FIELD32_4(register,                                                            \
                      field1, value1,                                                      \
                      field2, value2,                                                      \
                      field3, value3,                                                      \
                      field4, value4)                                                      \
    SetField32(                                                                            \
        ::nn::usb::detail::register##Offset,                                               \
        NN_USB_MAKE_MASK32_4(::nn::usb::detail::register, field1, field2, field3, field4), \
        NN_USB_MAKE_VALUE32_4(::nn::usb::detail::register,                                 \
                              field1, value1,                                              \
                              field2, value2,                                              \
                              field3, value3,                                              \
                              field4, value4)                                              \
    )

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
