﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_ComplexTegra21x.h
 * @brief   USB Complex Abstraction for Tegra21x
 */

#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/dd.h>
#include <nn/gpio/gpio.h>
#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_PmTypes.h>

#include "usb_Complex.h"
#include "usb_RegisterBlock.h"

#include "../ds/usb_DsController30-soc.tegra.h"
#include "../hs/usb_HsPlatformController30-soc.tegra.h"

namespace nn {
namespace usb {
namespace detail {

class UsbComplexTegra21x : public UsbComplex
{
public:
    UsbComplexTegra21x() NN_NOEXCEPT
        : UsbComplex(nn::usb::ComplexId_Tegra21x)
        , m_LastDetectedChargerType(UsbChargerType_Unknown)
    {
    }

    virtual ~UsbComplexTegra21x() NN_NOEXCEPT
    {
        // do nothing
    }

    enum PadMode
    {
        PadMode_Disabled = 0,
        PadMode_Host     = 1,
        PadMode_Device   = 2,
        PadMode_Otg      = 3
    };

    const int32_t OtgPad = 0; // attribute of hardware - first pad is OTG and charge detection capable

    virtual Result Initialize(UsbPlatform *pPlatform) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Finalize() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Enable() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Disable() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result CreatePort(uint32_t hsLane, uint32_t ssLane, uint32_t capability) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DestroyPort(uint32_t hsLane, uint32_t ssLane, uint32_t capability) NN_NOEXCEPT NN_OVERRIDE;

    virtual UsbChargerType ChargerDetection() NN_NOEXCEPT NN_OVERRIDE;

    void   XusbPadctlToggleVbusId0(bool isSet) NN_NOEXCEPT;
    size_t XusbPadctlReadBatteryChrgOtgPadCtl0Hs() NN_NOEXCEPT;

    void   OverrideDriveStrength(uint32_t port, int32_t offset, uint32_t *pOutStrength) NN_NOEXCEPT;

    RegisterBlock    XusbPadctlRegister;

private:
    void XusbPadctlSetUtmiPhyEnable(bool isEnabled)NN_NOEXCEPT;
    void XusbPadctlEnableSsPhy(SsPhyPad lane) NN_NOEXCEPT;
    void XusbSsPadctlSetWakeDetect(uint32_t ssPortMask, bool isWakeDetectionEnabled) NN_NOEXCEPT;
    void XusbPadctlUphyMisc(SsPhyPad lane, bool isOverride) NN_NOEXCEPT;
    void XusbPadctlHsInit(int32_t pad, PadMode padMode) NN_NOEXCEPT;
    void XusbPadctlSsInit(int32_t pad, PadMode padMode) NN_NOEXCEPT;
    void XusbPadctlSetVbusPadProtection(UsbChargerType chargerType) NN_NOEXCEPT;

    uint32_t GetHsCurrLevel(uint32_t pad) const NN_NOEXCEPT;
    uint32_t GetTermRangeAdj(uint32_t pad) const  NN_NOEXCEPT;
    uint32_t GetRpdCtrl(uint32_t pad) const NN_NOEXCEPT;
    uint32_t GetTermOffset() const NN_NOEXCEPT;

    void           StartChargerDetection() NN_NOEXCEPT;
    void           StopChargerDetection(UsbChargerType chargerType) NN_NOEXCEPT;
    UsbChargerType DetectCompliantCharger() NN_NOEXCEPT;

    void WaitForPcie() NN_NOEXCEPT;

private:
    static const int32_t                     SuperSpeedPortCount = 4;
    static const int32_t                     HighSpeedPortCount = 4;
    static const os::InterruptName           PadctlInterruptName = 32 + 49;

    ds::DsControllerTegra30                  m_DeviceControllerXusb;
    hs::PlatformControllerTegra30            m_HostControllerXusb;

    // sMMU
    SmmuMapManager                           m_SmmuXusbHost;
    SmmuMapManager                           m_SmmuXusbDevice;

    // settings
    UsbChargerType                           m_LastDetectedChargerType;

    uint32_t                                 m_FuseUsbCalib0;
    uint32_t                                 m_FuseUsbCalibExt0;
};


} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
