﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>

#include <nn/updater/updater.h>
#include <nn/updater/updater_DebugApi.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include "updater_BisSave.h"
#include "updater_Common.h"
#include "updater_PartitionAccessor.h"
#include "updater_UpdateTypeResolver.h"

namespace nn { namespace updater {

namespace
{

// true の場所を invalidate する
struct BreakPattern
{
    UpdatePhase phase;
    bool bctMain;
    bool bctSub;
    bool package1Main;
    bool package1Sub;
    bool package2Main;
    bool package2Sub;
};
const BreakPattern BreakPatterns[] = {
    {UpdatePhase::Phase1, false, false, false,  true, false, false},
    {UpdatePhase::Phase2, false, false, false,  true, false,  true},
    {UpdatePhase::Phase3, false,  true, false,  true, false,  true},
    {UpdatePhase::Phase4,  true, false,  true, false,  true, false},
    {UpdatePhase::Phase5, false, false,  true, false,  true, false},
    {UpdatePhase::Phase6, false, false,  true, false, false, false},
};

template <typename P1> struct AccessParameters
{
    BootPartition1Type bctMain;
    BootPartition1Type bctSub;
    P1 package1Main;
    P1 package1Sub;
    Package2Index package2Main;
    Package2Index package2Sub;
};
AccessParameters<BootPartition1Type> NormalParameters = {
    BootPartition1Type::BctNormalMain_BCT0,
    BootPartition1Type::BctNormalSub_BCT2,
    BootPartition1Type::Package1NormalMain,
    BootPartition1Type::Package1NormalSub,
    Package2Index::NormalMain,
    Package2Index::NormalSub,
};
AccessParameters<BootPartition2Type> SafeParameters = {
    BootPartition1Type::BctSafeMain_BCT1,
    BootPartition1Type::BctSafeSub_BCT3,
    BootPartition2Type::Package1SafeMain,
    BootPartition2Type::Package1SafeSub,
    Package2Index::SafeMain,
    Package2Index::SafeSub,
};

template <typename T, typename P1> Result SetBootImagesIntermediatePhaseDetail(const AccessParameters<P1>& parameters, UpdatePhase phase, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    const BreakPattern* pattern = nullptr;
    for (const auto& p : BreakPatterns)
    {
        if (p.phase == phase)
        {
            pattern = &p;
        }
    }
    NN_ABORT_UNLESS(pattern != nullptr, "Invalid phase");

    if (pattern->bctMain)
    {
        BootPartition1Accessor accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, parameters.bctMain));
    }
    if (pattern->bctSub)
    {
        BootPartition1Accessor accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, parameters.bctSub));
    }
    if (pattern->package1Main)
    {
        T accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, parameters.package1Main));
    }
    if (pattern->package1Sub)
    {
        T accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, parameters.package1Sub));
    }
    if (pattern->package2Main)
    {
        Package2Accessor accessor(parameters.package2Main);
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, Package2Type::Package2));
    }
    if (pattern->package2Sub)
    {
        Package2Accessor accessor(parameters.package2Sub);
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, Package2Type::Package2));
    }

    NN_RESULT_SUCCESS;
}


// マウントされている前提
template <typename T, typename P1> Result GetBootImageHashDetail(BootImageHashes* pOutHashes, const AccessParameters<P1>& parameters, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    {
        // bct
        BootPartition1Accessor accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        {
            // このスコープ内でのみ、ワークバッファを bct 用バッファとして扱う
            BctBuffer* ptr = reinterpret_cast<BctBuffer*>(workBuffer);

            NN_RESULT_DO(ReadFile(&(pOutHashes->bctSize), ptr->bct, BctSize, ResolveBctPath(bootImageUpdateType)));
            if (NeedsUpdateSecureInfo(bootImageUpdateType))
            {
                NN_RESULT_DO(accessor.UpdateSecureInfoAuto(ptr->bct, BctSize, ptr->eks, sizeof(ptr->eks)));
            }

            // ハッシュ計算は BctSize で行うので、こちらで
            pOutHashes->bctSize = BctSize;
            crypto::GenerateSha256Hash(pOutHashes->bctFile, sizeof(pOutHashes->bctFile), ptr->bct, BctSize);
        }
        NN_RESULT_DO(accessor.CalculateHash(pOutHashes->bctMain, sizeof(pOutHashes->bctMain), BctSize, workBuffer, workSize, parameters.bctMain));
        NN_RESULT_DO(accessor.CalculateHash(pOutHashes->bctSub, sizeof(pOutHashes->bctSub), BctSize, workBuffer, workSize, parameters.bctSub));
    }
    {
        // package1
        NN_RESULT_DO(ReadFileAndCalculateHash(&(pOutHashes->package1Size), pOutHashes->package1File, sizeof(pOutHashes->package1Main), ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize));

        T accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
        NN_RESULT_DO(accessor.CalculateHash(pOutHashes->package1Main, sizeof(pOutHashes->package1Main), pOutHashes->package1Size, workBuffer, workSize, parameters.package1Main));
        NN_RESULT_DO(accessor.CalculateHash(pOutHashes->package1Sub, sizeof(pOutHashes->package1Sub), pOutHashes->package1Size, workBuffer, workSize, parameters.package1Sub));
    }
    {
        // package2
        NN_RESULT_DO(ReadFileAndCalculateHash(&(pOutHashes->package2Size), pOutHashes->package2File, sizeof(pOutHashes->package1Main), ResolvePackage2Path(bootImageUpdateType), workBuffer, workSize));

        {
            Package2Accessor accessor(parameters.package2Main);
            NN_RESULT_DO(accessor.Initialize());
            NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
            NN_RESULT_DO(accessor.CalculateHash(pOutHashes->package2Main, sizeof(pOutHashes->package2Main), pOutHashes->package2Size, workBuffer, workSize, Package2Type::Package2));
        }
        {
            Package2Accessor accessor(parameters.package2Sub);
            NN_RESULT_DO(accessor.Initialize());
            NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
            NN_RESULT_DO(accessor.CalculateHash(pOutHashes->package2Sub, sizeof(pOutHashes->package2Sub), pOutHashes->package2Size, workBuffer, workSize, Package2Type::Package2));
        }
    }
    NN_RESULT_SUCCESS;
}
}
Result SetBootImagesIntermediatePhase(TargetBootMode mode, UpdatePhase phase, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    switch(mode)
    {
    case TargetBootMode::Normal:
        {
            // テンプレート関数を RESULT_DO に直接投げると誤った解釈をされる
            auto result = SetBootImagesIntermediatePhaseDetail<BootPartition1Accessor, BootPartition1Type>(NormalParameters, phase, workBuffer, workSize);
            NN_RESULT_DO(result);
        }
        break;
    case TargetBootMode::Safe:
        {
            auto result = SetBootImagesIntermediatePhaseDetail<BootPartition2Accessor, BootPartition2Type>(SafeParameters, phase, workBuffer, workSize);
            NN_RESULT_DO(result);
        }
        break;
    default:
        NN_ABORT("Unknown type: %d\n", mode);
    }

    NN_RESULT_SUCCESS;
}

Result GetBootImageHash(BootImageHashes* pOutHashes, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    ncm::SystemDataId id;
    NN_RESULT_DO(GetBootImagePackageId(&id, mode, workBuffer, workSize));

    NN_RESULT_TRY(fs::MountSystemData(GetMountName(), id))
        NN_RESULT_CATCH(fs::ResultTargetNotFound) {NN_RESULT_THROW(ResultBootImagePackageNotFound());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(GetMountName()); };

    switch(mode)
    {
    case TargetBootMode::Normal:
        {
            // テンプレート関数を RESULT_DO に直接投げると誤った解釈をされる
            auto result = GetBootImageHashDetail<BootPartition1Accessor, BootPartition1Type>(pOutHashes, NormalParameters, workBuffer, workSize, bootImageUpdateType);
            NN_RESULT_DO(result);
        }
        break;
    case TargetBootMode::Safe:
        {
            auto result = GetBootImageHashDetail<BootPartition2Accessor, BootPartition2Type>(pOutHashes, SafeParameters, workBuffer, workSize, bootImageUpdateType);
            NN_RESULT_DO(result);
        }
        break;
    default:
        NN_ABORT("Unknown type: %d\n", mode);
    }

    NN_RESULT_SUCCESS;
}

}}
