﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/uart/uart.h>

#include <nn/uart/uart_IManager.sfdl.h>
#include <nn/uart/uart_IPortSession.sfdl.h>

#include "uart_GetManagerByHipc.h"

// HIPC か DFC を切り替えるマクロ定義
#define NN_UART_ACCESS_BY_HIPC

namespace nn { namespace uart {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::uart::IManager> g_Manager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

inline IPortSession* GetInterface(PortSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession->_handle);
    return static_cast<nn::uart::IPortSession*>(pSession->_handle);
}


// 開いているポート数のカウント
std::atomic<int> g_OpenPortNumber(0);

// OpenPort / OpenPortForDev の共通処理
bool OpenPortImpl(PortSession* pOutSession, int32_t number, bool isForDev,const PortConfigType& portConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    // バッファ指定の事前条件
    NN_SDK_REQUIRES(portConfig._sendBufferLength > 0 ||
                    portConfig._receiveBufferLength > 0);

    // Session を生成する
    nn::sf::SharedPointer<nn::uart::IPortSession> session;
    auto createResult = g_Manager->CreatePortSession(&session);
    NN_ABORT_UNLESS_RESULT_SUCCESS(createResult);

    // 以下、各種情報の設定処理
    nn::sf::NativeHandle        sendHandle;
    nn::sf::NativeHandle        receiveHandle;
    nn::Result                  openResult;
    bool                        isSuccess;

    if(portConfig._sendBufferLength > 0)
    {
        nn::os::TransferMemory  sendBuf(portConfig._sendBuffer,
                                        portConfig._sendBufferLength,
                                        nn::os::MemoryPermission_ReadWrite);

        sendHandle = nn::sf::NativeHandle(sendBuf.Detach(), true);
    }

    if(portConfig._receiveBufferLength > 0)
    {
        nn::os::TransferMemory  receiveBuf(portConfig._receiveBuffer,
                                           portConfig._receiveBufferLength,
                                           nn::os::MemoryPermission_ReadWrite);

        receiveHandle = nn::sf::NativeHandle(receiveBuf.Detach(), true);
    }

    if(isForDev)
    {
        openResult = session->OpenPortForDev(&isSuccess, number,
                                     static_cast<nn::uart::BaudRate>(portConfig._baudRate),
                                     static_cast<nn::uart::FlowControlMode>(portConfig._flowControlMode),
                                     std::move(sendHandle),    static_cast<uint64_t>(portConfig._sendBufferLength),
                                     std::move(receiveHandle), static_cast<uint64_t>(portConfig._receiveBufferLength),
                                     portConfig._isInvertTx, portConfig._isInvertRx, portConfig._isInvertRts, portConfig._isInvertCts);
    }
    else
    {
        openResult = session->OpenPort(&isSuccess, number,
                                     static_cast<nn::uart::BaudRate>(portConfig._baudRate),
                                     static_cast<nn::uart::FlowControlMode>(portConfig._flowControlMode),
                                     std::move(sendHandle),    static_cast<uint64_t>(portConfig._sendBufferLength),
                                     std::move(receiveHandle), static_cast<uint64_t>(portConfig._receiveBufferLength),
                                     portConfig._isInvertTx, portConfig._isInvertRx, portConfig._isInvertRts, portConfig._isInvertCts);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(openResult);
    if(isSuccess)
    {
        // SharedPointer から所有権を切り離して、pOutHandle として返す
        pOutSession->_handle = session.Detach();
        pOutSession->_sendBufferReadyEvent    = nullptr;
        pOutSession->_sendBufferEmptyEvent    = nullptr;
        pOutSession->_receiveBufferReadyEvent = nullptr;
        pOutSession->_receiveEndEvent         = nullptr;
        g_OpenPortNumber++;
    }

    return isSuccess;
}


}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_Manager);

        g_Manager = nn::uart::GetManagerByHipc();
    }
    g_InitializeCount++;
}

void InitializeWith(nn::sf::SharedPointer<nn::uart::IManager> manager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_Manager = manager;

    g_InitializeCount++;
}

bool IsInitialized() NN_NOEXCEPT
{
    if(g_InitializeCount > 0)
    {
        return true;
    }
    else
    {
        return false;
    }

}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    // すべてのポートが閉じられているかチェック
    NN_SDK_ASSERT(g_OpenPortNumber == 0);

    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_Manager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_Manager = nullptr;
    }
}

bool HasPort(PortName name) NN_NOEXCEPT
{
    bool hasPort;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->HasPort(&hasPort, static_cast<std::int32_t>(name));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return hasPort;
}

bool HasPortForDev(int portIndex) NN_NOEXCEPT
{
    bool hasPort;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->HasPortForDev(&hasPort, static_cast<std::int32_t>(portIndex));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return hasPort;
}

bool IsSupportedBaudRate(PortName name, BaudRate baudRate) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedBaudRate(&isSupport, static_cast<std::int32_t>(name), static_cast<std::int32_t>(baudRate));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}

bool IsSupportedBaudRateForDev(int portIndex, BaudRate baudRate) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedBaudRateForDev(&isSupport, static_cast<std::int32_t>(portIndex), static_cast<std::int32_t>(baudRate));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}

bool IsSupportedFlowControlMode(PortName name, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedFlowControlMode(&isSupport, static_cast<std::int32_t>(name), static_cast<std::int32_t>(flowControlMode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}

bool IsSupportedFlowControlModeForDev(int portIndex, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedFlowControlModeForDev(&isSupport, static_cast<std::int32_t>(portIndex), static_cast<std::int32_t>(flowControlMode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}

bool IsSupportedPortEvent(PortName name, PortEventType portEvent) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedPortEvent(&isSupport, static_cast<std::int32_t>(name), static_cast<std::int32_t>(portEvent));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}

bool IsSupportedPortEventForDev(int portIndex, PortEventType portEvent) NN_NOEXCEPT
{
    bool isSupport;

    NN_SDK_ASSERT(g_Manager);
    auto result = g_Manager->IsSupportedPortEventForDev(&isSupport, static_cast<std::int32_t>(portIndex), static_cast<std::int32_t>(portEvent));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isSupport;
}


// この関数は HIPC せず Client 側だけで設定している。Driver 実装が変わった場合は要修正
void InitializePortConfig(PortConfigType* pOutPortConfig, BaudRate baudRate,
        char* sendBuffer, size_t sendBufferLength,
        char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPortConfig);
    NN_SDK_REQUIRES(sendBuffer || receiveBuffer, "Either sendBuffer or receiveBuffer must not be null.\n");

    pOutPortConfig->_baudRate            = baudRate;
    pOutPortConfig->_flowControlMode     = FlowControlMode_None; // Default
    pOutPortConfig->_sendBuffer          = sendBuffer;
    pOutPortConfig->_sendBufferLength    = sendBuffer ? sendBufferLength : 0;
    pOutPortConfig->_receiveBuffer       = receiveBuffer;
    pOutPortConfig->_receiveBufferLength = receiveBuffer ? receiveBufferLength : 0;
    pOutPortConfig->_isInvertTx          = false; // Default
    pOutPortConfig->_isInvertRx          = false; // Default
    pOutPortConfig->_isInvertRts         = false; // Default
    pOutPortConfig->_isInvertCts         = false; // Default
}

// この関数は HIPC せず Client 側だけで設定している。Driver 実装が変わった場合は要修正
void SetPortConfigFlowControlMode(PortConfigType* pPortConfig, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPortConfig);
    pPortConfig->_flowControlMode = flowControlMode;
}

// この関数は HIPC せず Client 側だけで設定している。Driver 実装が変わった場合は要修正
void SetPortConfigInvertPins(PortConfigType* pPortConfig, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPortConfig);
    pPortConfig->_isInvertTx = isInvertTx;
    pPortConfig->_isInvertRx = isInvertRx;
    pPortConfig->_isInvertRts = isInvertRts;
    pPortConfig->_isInvertCts = isInvertCts;
}

bool OpenPort(PortSession* pOutSession, PortName name, const PortConfigType& portConfig) NN_NOEXCEPT
{
    return OpenPortImpl(pOutSession, static_cast<int32_t>(name), false, portConfig);
}

bool OpenPortForDev(PortSession* pOutSession, int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT
{
    return OpenPortImpl(pOutSession, static_cast<int32_t>(portIndex), true, portConfig);
}

size_t GetWritableLength(const PortSession* pSession) NN_NOEXCEPT
{
    int64_t size;
    auto result = static_cast<nn::uart::IPortSession*>(pSession->_handle)->GetWritableLength(&size);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<size_t>(size);
}

void Send(size_t* pOutDoneBytes, PortSession* pSession, const void* data, size_t dataBytes) NN_NOEXCEPT
{
    int64_t size;
    nn::sf::InBuffer inBuffer(static_cast<const char*>(data), dataBytes);

    // TORIAEZU : size が 0 のバッファ(配列)を InBuffer で送るとサーバー側で nullptr となるため、
    // ここで size が 0  という時点で弾く
    NN_SDK_REQUIRES(dataBytes > 0);

    auto result = GetInterface(pSession)->Send(&size, inBuffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    if (pOutDoneBytes != nullptr)
    {
        *pOutDoneBytes = static_cast<size_t>(size);
    }
}

size_t GetReadableLength(const PortSession* pSession) NN_NOEXCEPT
{
    int64_t size;
    auto result = static_cast<nn::uart::IPortSession*>(pSession->_handle)->GetReadableLength(&size);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<size_t>(size);
}

nn::Result Receive(size_t* pOutDoneBytes, void* outData, size_t dataBytes, PortSession* pSession) NN_NOEXCEPT
{
    int64_t size;
    nn::sf::OutBuffer outBuffer(static_cast<char*>(outData), dataBytes);

    // TORIAEZU : size が 0 のバッファ(配列)を OutBuffer で送るとサーバー側で nullptr となるため、
    // ここで size が 0  という時点で弾く
    NN_SDK_REQUIRES(dataBytes > 0);

    auto result = GetInterface(pSession)->Receive(&size, outBuffer);

    if (pOutDoneBytes != nullptr)
    {
        *pOutDoneBytes = static_cast<size_t>(size);
    }

    outData = outBuffer.GetPointerUnsafe();

    return result;
}

bool BindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession, PortEventType eventType, size_t threshold) NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;
    bool isSuccess;

    NN_RESULT_DO(GetInterface(pSession)->BindPortEvent(&isSuccess, &sfHandle, static_cast<int32_t>(eventType), static_cast<int64_t>(threshold)));

    nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                              sfHandle.GetOsHandle(),
                                              sfHandle.IsManaged(),
                                              nn::os::EventClearMode_ManualClear);
    sfHandle.Detach();

    // UnbindPortEvent で SystemEvent を識別できるように Session のメンバ変数にポインタを保存
    switch(static_cast<nn::uart::PortEventType>(eventType))
    {
    case nn::uart::PortEventType_SendBufferEmpty:
        pSession->_sendBufferEmptyEvent = pEvent;
        break;

    case nn::uart::PortEventType_SendBufferReady:
        pSession->_sendBufferReadyEvent = pEvent;
        break;

    case nn::uart::PortEventType_ReceiveBufferReady:
        pSession->_receiveBufferReadyEvent = pEvent;
        break;

    case nn::uart::PortEventType_ReceiveEnd:
        pSession->_receiveEndEvent = pEvent;
        break;

    default:NN_UNEXPECTED_DEFAULT;

    }

    return isSuccess;
}

bool UnbindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession) NN_NOEXCEPT
{
    int32_t eventType;
    bool isSuccess;

    // どの PortEventType に割り当てられているイベントか検索する
    if(pEvent == pSession->_sendBufferEmptyEvent)
    {
        eventType = static_cast<int32_t>(nn::uart::PortEventType_SendBufferEmpty);
    }
    else if(pEvent == pSession->_sendBufferReadyEvent)
    {
        eventType = static_cast<int32_t>(nn::uart::PortEventType_SendBufferReady);
    }
    else if(pEvent == pSession->_receiveBufferReadyEvent)
    {
        eventType = static_cast<int32_t>(nn::uart::PortEventType_ReceiveBufferReady);
    }
    else if(pEvent == pSession->_receiveEndEvent)
    {
        eventType = static_cast<int32_t>(nn::uart::PortEventType_ReceiveEnd);
    }
    else
    {
        return false;
    }

    auto result = GetInterface(pSession)->UnbindPortEvent(&isSuccess, eventType);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // TODO: Bind されていたイベントを破棄する。
    //nn::os::DestroySystemEvent(pEvent);

    if(pEvent == pSession->_sendBufferEmptyEvent)
    {
        pSession->_sendBufferEmptyEvent = nullptr;
    }

    if(pEvent == pSession->_sendBufferReadyEvent)
    {
        pSession->_sendBufferReadyEvent = nullptr;
    }

    if(pEvent == pSession->_receiveBufferReadyEvent)
    {
        pSession->_receiveBufferReadyEvent = nullptr;
    }

    if(pEvent == pSession->_receiveEndEvent)
    {
        pSession->_receiveEndEvent = nullptr;
    }

    return isSuccess;
}

void ClosePort(PortSession* pSession) NN_NOEXCEPT
{
    // Unbind されていないなら Unbind する
    if(pSession->_sendBufferEmptyEvent != nullptr)
    {
        nn::uart::UnbindPortEvent(pSession->_sendBufferEmptyEvent, pSession);
    }

    if(pSession->_sendBufferReadyEvent != nullptr)
    {
        nn::uart::UnbindPortEvent(pSession->_sendBufferReadyEvent, pSession);
    }

    if(pSession->_receiveBufferReadyEvent != nullptr)
    {
        nn::uart::UnbindPortEvent(pSession->_receiveBufferReadyEvent, pSession);
    }

    if(pSession->_receiveEndEvent != nullptr)
    {
        nn::uart::UnbindPortEvent(pSession->_receiveEndEvent, pSession);
    }

    nn::sf::ReleaseSharedObject(GetInterface(pSession));

    g_OpenPortNumber--;

}


}}
