﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/os/os_SystemEvent.h>

#include <nnc/nn_Result.h>
#include <nnc/result/result_CConverter.h>

#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>
#include <nnc/uart/uart_Lib.h>
#include <nnc/uart/uart_PortApiDev.h>
#include <nnc/uart/uart_Port.h>

//-----------------------------------------------------------------------------
// C++ 用の型と C 用の型の定義の一致チェック
//-----------------------------------------------------------------------------

NN_STATIC_ASSERT(sizeof(nnuartPortName) == sizeof(nn::uart::PortName));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartPortName) == NN_ALIGNOF(nn::uart::PortName));

NN_STATIC_ASSERT(sizeof(nnuartBaudRate) == sizeof(nn::uart::BaudRate));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartBaudRate) == NN_ALIGNOF(nn::uart::BaudRate));
NN_STATIC_ASSERT(static_cast<int>(nnuartBaudRate_57600) == static_cast<int>(nn::uart::BaudRate_57600));
NN_STATIC_ASSERT(static_cast<int>(nnuartBaudRate_115200) == static_cast<int>(nn::uart::BaudRate_115200));

NN_STATIC_ASSERT(sizeof(nnuartFlowControlMode) == sizeof(nn::uart::FlowControlMode));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartFlowControlMode) == NN_ALIGNOF(nn::uart::FlowControlMode));
NN_STATIC_ASSERT(static_cast<int>(nnuartFlowControlMode_None) == static_cast<int>(nn::uart::FlowControlMode_None));
NN_STATIC_ASSERT(static_cast<int>(nnuartFlowControlMode_Hardware) == static_cast<int>(nn::uart::FlowControlMode_Hardware));

NN_STATIC_ASSERT(sizeof(nnuartPortEventType) == sizeof(nn::uart::PortEventType));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartPortEventType) == NN_ALIGNOF(nn::uart::PortEventType));
NN_STATIC_ASSERT(static_cast<int>(nnuartPortEventType_ReceiveBufferReady) == static_cast<int>(nn::uart::PortEventType_ReceiveBufferReady));

NN_STATIC_ASSERT(sizeof(nnuartPortConfigType) == sizeof(nn::uart::PortConfigType));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartPortConfigType) == NN_ALIGNOF(nn::uart::PortConfigType));

NN_STATIC_ASSERT(sizeof(nnuartPortSession) == sizeof(nn::uart::PortSession));
NN_STATIC_ASSERT(NN_ALIGNOF(nnuartPortSession) == NN_ALIGNOF(nn::uart::PortSession));

//-----------------------------------------------------------------------------

void nnuartInitialize()
{
    nn::uart::Initialize();
}

bool nnuartIsInitialized()
{
    return nn::uart::IsInitialized();
}

void nnuartFinalize()
{
    nn::uart::Finalize();
}

bool nnuartHasPort(nnuartPortName name)
{
    return nn::uart::HasPort(
        static_cast<nn::uart::PortName>(name)
        );
}

bool nnuartHasPortForDev(int portIndex)
{
    return nn::uart::HasPortForDev(portIndex);
}

bool nnuartIsSupportedBaudRate(nnuartPortName name, nnuartBaudRate baudRate)
{
    return nn::uart::IsSupportedBaudRate(
        static_cast<nn::uart::PortName>(name),
        static_cast<nn::uart::BaudRate>(baudRate)
        );
}

bool nnuartIsSupportedBaudRateForDev(int portIndex, nnuartBaudRate baudRate)
{
    return nn::uart::IsSupportedBaudRateForDev(
        portIndex,
        static_cast<nn::uart::BaudRate>(baudRate)
        );
}

bool nnuartIsSupportedFlowControlMode(nnuartPortName name, nnuartFlowControlMode flowControlMode)
{
    return nn::uart::IsSupportedFlowControlMode(
        static_cast<nn::uart::PortName>(name),
        static_cast<nn::uart::FlowControlMode>(flowControlMode)
        );
}

bool nnuartIsSupportedFlowControlModeForDev(int portIndex, nnuartFlowControlMode flowControlMode)
{
    return nn::uart::IsSupportedFlowControlModeForDev(
        portIndex,
        static_cast<nn::uart::FlowControlMode>(flowControlMode)
        );
}

bool nnuartIsSupportedPortEvent(nnuartPortName name, nnuartPortEventType portEvent)
{
    return nn::uart::IsSupportedPortEvent(
        static_cast<nn::uart::PortName>(name),
        static_cast<nn::uart::PortEventType>(portEvent)
        );
}

bool nnuartIsSupportedPortEventForDev(int portIndex, nnuartPortEventType portEvent)
{
    return nn::uart::IsSupportedPortEventForDev(
        portIndex,
        static_cast<nn::uart::PortEventType>(portEvent)
        );
}

void nnuartInitializePortConfig(nnuartPortConfigType* pOutPortConfig, nnuartBaudRate baudRate, char* sendBuffer, size_t sendBufferLength, char* receiveBuffer, size_t receiveBufferLength)
{
    nn::uart::InitializePortConfig(
        reinterpret_cast<nn::uart::PortConfigType*>(pOutPortConfig),
        static_cast<nn::uart::BaudRate>(baudRate),
        sendBuffer,
        sendBufferLength,
        receiveBuffer,
        receiveBufferLength
        );
}

void nnuartSetPortConfigFlowControlMode(nnuartPortConfigType* pPortConfig, nnuartFlowControlMode flowControlMode)
{
    nn::uart::SetPortConfigFlowControlMode(
        reinterpret_cast<nn::uart::PortConfigType*>(pPortConfig),
        static_cast<nn::uart::FlowControlMode>(flowControlMode)
        );
}

void nnuartSetPortConfigInvertPins(nnuartPortConfigType* pPortConfig, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts)
{
    nn::uart::SetPortConfigInvertPins(reinterpret_cast<nn::uart::PortConfigType*>(pPortConfig), isInvertTx, isInvertRx, isInvertRts, isInvertCts);
}

bool nnuartOpenPort(nnuartPortSession* pOutSession, nnuartPortName name, const nnuartPortConfigType* pPortConfig)
{
    return nn::uart::OpenPort(
        reinterpret_cast<nn::uart::PortSession*>(pOutSession),
        static_cast<nn::uart::PortName>(name),
        *reinterpret_cast<const nn::uart::PortConfigType*>(pPortConfig)
        );
}

bool nnuartOpenPortForDev(nnuartPortSession* pOutSession, int portIndex, const nnuartPortConfigType* pPortConfig)
{
    return nn::uart::OpenPortForDev(
        reinterpret_cast<nn::uart::PortSession*>(pOutSession),
        portIndex,
        *reinterpret_cast<const nn::uart::PortConfigType*>(pPortConfig)
        );
}

size_t nnuartGetWritableLength(const nnuartPortSession* pSession)
{
    return nn::uart::GetWritableLength(
        reinterpret_cast<const nn::uart::PortSession*>(pSession)
        );
}

void nnuartSend(size_t* pOutDoneBytes, nnuartPortSession* pSession, const void* data, size_t dataBytes)
{
    nn::uart::Send(
        pOutDoneBytes,
        reinterpret_cast<nn::uart::PortSession*>(pSession),
        data,
        dataBytes
        );
}

size_t nnuartGetReadableLength(const nnuartPortSession* pSession)
{
    return nn::uart::GetReadableLength(
        reinterpret_cast<const nn::uart::PortSession*>(pSession)
        );
}

nnResult nnuartReceive(size_t* pOutDoneBytes, void* outData, size_t dataBytes, nnuartPortSession* pSession)
{
    return nn::result::ConvertToC(
        nn::uart::Receive(
            pOutDoneBytes,
            outData,
            dataBytes,
            reinterpret_cast<nn::uart::PortSession*>(pSession)
            )
        );
}

bool nnuartBindPortEvent(nnosSystemEventType* pEvent, nnuartPortSession* pSession, nnuartPortEventType eventType, size_t threshold)
{
    return nn::uart::BindPortEvent(
        reinterpret_cast<nn::os::SystemEventType*>(pEvent),
        reinterpret_cast<nn::uart::PortSession*>(pSession),
        static_cast<nn::uart::PortEventType>(eventType),
        threshold
        );
}

bool nnuartUnbindPortEvent(nnosSystemEventType* pEvent, nnuartPortSession* pSession)
{
    return nn::uart::UnbindPortEvent(
        reinterpret_cast<nn::os::SystemEventType*>(pEvent),
        reinterpret_cast<nn::uart::PortSession*>(pSession)
        );
}

void nnuartClosePort(nnuartPortSession* pSession)
{
    return nn::uart::ClosePort(
        reinterpret_cast<nn::uart::PortSession*>(pSession)
        );
}


