﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkConditionVariable.h>

#include <nn/uart/uart_PortTypes.h>
#include "uart_TargetSpec.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

/**
 * @brief   割り込みイベント通知用コールバック関数の型宣言
 */
typedef void (*InterruptCallbackFuncType)(uintptr_t arg);

/**
* @brief   割り込みイベント通知用コールバック関数の型宣言
*/
const int InvalidIndex = -1;


/**
 * @brief   割り込みイベント通知クラス。
 * @details 内部に持っているスタック・優先度でスレッドを立てる。@n
 *          スレッド内で 1 つ以上の割り込みイベントを多重待ちする。@n
 *          割り込みが起こると、初期化時に登録したコールバックに割り込み番号を通知する。@n
 *          シングルトンパターンのため実体は常に唯一である。
 */
class InterruptNotifier
{
    NN_DISALLOW_COPY(InterruptNotifier);
    NN_DISALLOW_MOVE(InterruptNotifier);

private: // Singleton
    InterruptNotifier() NN_NOEXCEPT :
        m_IsInitialized(false),
        m_IsThreadRunning(false),
        m_ConfigFinishedIndex(InvalidIndex),
        m_PortIndexForConfig(InvalidIndex),
#ifdef NN_DETAIL_UART_ENABLE_DMA
        m_IsSetDmaHandler(false),
        m_IsUnsetDmaHandler(false),
#endif
        m_IsFinalizeRequested(false)
    {
        for (int i = 0; i < PortCountMax; i++)
        {
            m_Callback[i] = nullptr;
            m_CallbackArg[i] = 0;
            m_UnexpectedCallback[i] = nullptr;
            m_UnexpectedCallbackArg[i] = 0;
        }
    }

public:
    /**
     * @brief   本クラスの唯一のインスタンスを取得する。
     */
    static InterruptNotifier& GetInstance() NN_NOEXCEPT
    {
        static InterruptNotifier inst;
        return inst;
    }

    /**
     * @brief   初期化。割り込みイベントを設定する。
     *          また、イベント通知スレッドを作成する。
     * @pre     初期化前の状態であること。
     * @post    初期化完了。スレッドが作成され、停止した状態である。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief   クラスの終了。割り込みイベントとの紐付を破棄し、初期化時に作成したスレッドも破棄する。
     * @pre     クラスが初期化された状態であること。スレッドが停止した状態であること。
     * @post    初期化前の状態に戻る。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   イベント通知スレッドを作成から呼ばれるコールバックを登録する。
     *          nullptr を渡すと未登録状態に戻る。
     * @pre     クラスが初期化された状態であること。
     *          callback がヌルでないとき、コールバック未登録状態であること
     */
    void SetHandler(int portIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT;

    void SetDmaHandler(int portIndex, int channelIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT;

    void LinkHandler(int portIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT;

    void LinkDmaHandler(int portIndex, int channelIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT;

    /**
     * @brief   イベント通知スレッドの開始。
     * @pre     クラスが初期化された状態であること。スレッドが停止した状態であること。
     * @post    スレッドが開始した状態になる。
     */
    void StartThread() NN_NOEXCEPT;

    /**
     * @brief   イベント通知スレッドの停止。
     * @pre     クラスが初期化された状態であること。スレッドが開始した状態であること。
     * @post    スレッドが停止した状態になる。
     */
    void StopThread() NN_NOEXCEPT;

    void StartThreadForResuming() NN_NOEXCEPT;
    void StopThreadForSuspending() NN_NOEXCEPT;

    bool IsThreadRunning() NN_NOEXCEPT
    {
        return m_IsThreadRunning;
    }

#ifdef NN_DETAIL_UART_ENABLE_DMA
    nn::os::InterruptEventType* GetDmaEventType(int portIndex, int channelIndex) NN_NOEXCEPT
    {
        int eventIndex = portIndex * DmaChannelCountPerPort + channelIndex;
        NN_SDK_ASSERT(eventIndex >= 0 && eventIndex < DmaChannelCountMax);
        return &m_DmaInterruptEvent[eventIndex];
    }
#endif

private:
    static void InterruptThread(void* arg) NN_NOEXCEPT;
    void InterruptThreadBody() NN_NOEXCEPT;

private:
    bool                            m_IsInitialized;
    bool                            m_IsThreadRunning;

    // イベントを受けるスレッド本体と、それに使う多重待ちオブジェクト
    nn::os::ThreadType              m_WorkThread;
    nn::os::MultiWaitType           m_Waiter;

    // 割り込みイベント用
    nn::os::MultiWaitHolderType     m_InterruptEventHolder[InterruptCountMax];
    nn::os::InterruptEventType      m_InterruptEvent[InterruptCountMax];

    // 登録されたコールバック
    InterruptCallbackFuncType       m_Callback[PortCountMax];
    uintptr_t                       m_CallbackArg[PortCountMax];
    InterruptCallbackFuncType       m_CallbackForConfig[PortCountMax];
    uintptr_t                       m_CallbackArgForConfig[PortCountMax];

    // 登録された UnexpectedInterrupt コールバック
    InterruptCallbackFuncType       m_UnexpectedCallback[PortCountMax];
    uintptr_t                       m_UnexpectedCallbackArg[PortCountMax];

    // イベントを登録するためのイベント
    nn::os::MultiWaitHolderType      m_ConfigStartEventHolder;
    nn::os::EventType                m_ConfigStartEvent;
    nn::os::MultiWaitHolderType      m_DmaConfigStartEventHolder;
    nn::os::EventType                m_DmaConfigStartEvent;
    nn::os::SdkMutexType             m_ConfigMutex;
    nn::os::SdkConditionVariableType m_ConfigCond;
    int                              m_ConfigFinishedIndex;

    // イベント登録時に参照される、操作対象の Index
    int                             m_PortIndexForConfig;
    int                             m_ChannelIndexForConfig;

#ifdef NN_DETAIL_UART_ENABLE_DMA
    // DMA 用
    nn::os::MultiWaitHolderType     m_DmaInterruptEventHolder[DmaChannelCountMax];
    nn::os::InterruptEventType      m_DmaInterruptEvent[DmaChannelCountMax];
    InterruptCallbackFuncType       m_DmaCallback[DmaChannelCountMax];
    uintptr_t                       m_DmaCallbackArg[DmaChannelCountMax];
    InterruptCallbackFuncType       m_DmaCallbackForConfig[DmaChannelCountMax];
    uintptr_t                       m_DmaCallbackArgForConfig[DmaChannelCountMax];
    bool                            m_IsSetDmaHandler;
    bool                            m_IsUnsetDmaHandler;
    int                             m_SetDmaHandlerIndex;
#endif

    // スレッド終了イベント用
    nn::os::MultiWaitHolderType     m_FinalizeRequestEventHolder;
    nn::os::EventType               m_FinalizeRequestEvent;
    bool                            m_IsFinalizeRequested;
};

} // detail
} // driver
} // uart
} // nn
