﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/uart_Result.h>
#include <nn/uart/driver/uart_PortApiDev.h>
#include "uart_Interrupt.h"
#include "uart_TargetSpec.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

// 前方宣言
class PortSessionImpl;

/**
 * @brief   UART ドライバライブラリ全体の静的な状態とリソースを管理するクラス。
 * @details ポートへのセッションの割り当てと割り込みスレッドの設定・管理を主に行う。
 *          シングルトンパターンのため実体は常に唯一である。
 */
class Driver
{
    NN_DISALLOW_COPY(Driver);
    NN_DISALLOW_MOVE(Driver);

private: // Singleton
    Driver() NN_NOEXCEPT :
        m_LibInitCount(0),
        m_IsSuspended(false),
        m_DriverMutex(true,0),
        m_IsOpenPortExistBeforeSleeping(false)
        {}

public:
    static Driver& GetInstance() NN_NOEXCEPT
    {
        static Driver inst;
        return inst;
    }

    void Initialize() NN_NOEXCEPT;
    bool IsInitialized() const NN_NOEXCEPT
    {
        return (m_LibInitCount > 0);
    }
    void Finalize() NN_NOEXCEPT;

    bool IsSuspended() const NN_NOEXCEPT
    {
        return m_IsSuspended;
    }

    bool HasPort(int portIndex) const NN_NOEXCEPT
    {
        return IsAvailablePortIndex(portIndex);
    }

    bool IsPortOpen(int portIndex) const NN_NOEXCEPT
    {
        return GetPortSessionHolder(portIndex).IsPortSessionRegistered();
    }

    void SuspendOpenPorts() NN_NOEXCEPT;
    void ResumeOpenPorts() NN_NOEXCEPT;
    bool OpenPortSession(PortSession* pOutSession, int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT;
    void ClosePortSession(PortSessionImpl& session) NN_NOEXCEPT;
    bool IsPortSessionRegistered(const PortSessionImpl* pSession) const NN_NOEXCEPT;
    bool VerifyOpenPortSession(const PortSessionImpl& session) const NN_NOEXCEPT;

private:
    /**
     * @brief   オープンされているポートセッションを管理するためのクラス
     */
    class PortSessionHolder
    {
        NN_DISALLOW_COPY(PortSessionHolder);
        NN_DISALLOW_MOVE(PortSessionHolder);

    public:
        PortSessionHolder() NN_NOEXCEPT :
            m_pSessionImpl(nullptr)
            {}

        void Reset() NN_NOEXCEPT
        {
            m_pSessionImpl = nullptr;
        }

        bool RegisterPortSession(PortSessionImpl* pSession) NN_NOEXCEPT
        {
            if (m_pSessionImpl)
            {
                // すでにセッションがリンクされているため、オープン失敗
                return false;
            }
            m_pSessionImpl = pSession;
            return true;
        }
        void UnregisterPortSession(PortSessionImpl* pSession) NN_NOEXCEPT
        {
            // リンクされているポートセッションに対するクローズでなくてはならない
            NN_SDK_ASSERT(VerifyPortSession(pSession));
            m_pSessionImpl = nullptr;
        }

        const PortSessionImpl* GetPortSessionImpl() const NN_NOEXCEPT
        {
            return m_pSessionImpl;
        }
        PortSessionImpl* GetPortSessionImpl() NN_NOEXCEPT
        {
            return m_pSessionImpl;
        }

        bool IsPortSessionRegistered() const NN_NOEXCEPT
        {
            return m_pSessionImpl ? true : false;
        }
        bool VerifyPortSession(const PortSessionImpl* pSession) const NN_NOEXCEPT
        {
            return (m_pSessionImpl && (m_pSessionImpl == pSession));
        }
    private:
        PortSessionImpl*  m_pSessionImpl;
    };

private:
    const PortSessionHolder& GetPortSessionHolder(int portIndex) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(HasPort(portIndex));
        return m_PortSessionHolder[portIndex];
    }
    PortSessionHolder& GetPortSessionHolder(int portIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(HasPort(portIndex));
        return m_PortSessionHolder[portIndex];
    }

    void ClosePortSessionBody(PortSessionImpl& session) NN_NOEXCEPT;

private:
    int                     m_LibInitCount;
    bool                    m_IsSuspended;
    mutable nn::os::Mutex   m_DriverMutex;
    bool                    m_IsOpenPortExistBeforeSleeping;

    PortSessionHolder       m_PortSessionHolder[PortCountMax];
};

} // detail
} // driver
} // uart
} // nn
