﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_node_usb.h"
#if defined( TMIPC_TARGET_HORIZON )
#include "tmipc_node_usb_interface.h"
#include "tmipc_packet.h"
#include "tmipc_result.h"
#include "../DejaInsight.h"
#include "../thread_tracker.h"
#include "../Version.h"
#include "../performance_monitor/performance_monitor_service.h"
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>

// ============================================================================
namespace tmipc {
// ============================================================================

// Static data.
NN_OS_ALIGNAS_THREAD_STACK u8 NodeUSB::m_SendThreadStack[TMIPC_STACK_SIZE];    // The thread stacks send and receive.
NN_OS_ALIGNAS_THREAD_STACK u8 NodeUSB::m_ReceiveThreadStack[TMIPC_STACK_SIZE];
Thread NodeUSB::m_SendThread;
Thread NodeUSB::m_ReceiveThread;

// ============================================================================
// Initialize the USB interface.  ThreadPriority will persist until
// Finalize() is called - only then can ThreadPriority be changed.
Result NodeUSB::Initialize( s32 ThreadPriority )
{
    DEJA_TRACE( "NodeUSB::NodeUSB", "NodeUSB Create the USB interface" );
    const Result Result = { USBInterface::Initialize( ThreadPriority ) };
    return Result;
}

// ============================================================================
// Finalize the USB interface.
Result NodeUSB::Finalize()
{
    const Result Result = { USBInterface::Finalize() };
    return Result;
}

// ============================================================================
// Constructor.
NodeUSB::NodeUSB()
:   BaseClass()
{
    DEJA_TRACE( "NodeUSB::NodeUSB", "NodeUSB constructor" );
    m_SendQueue.Create( 64 );

    m_bAbortedBlockingCalls = false;
    m_bIsConnected          = false;

    // Set up the NodeEvent callback.
    USBInterface::SetCallbackInformation( EventCallback, this );
}

// ============================================================================
// Destructor.
NodeUSB::~NodeUSB()
{
    DEJA_TRACE( "NodeUSB::~NodeUSB", "NodeUSB destructor" );
    Disconnect();

    m_SendQueue.Destroy();
}

// ============================================================================
// Starts the Send/Receive threads.
void NodeUSB::StartThreads()
{
    // Cancel any existing blocking calls.
    VERIFY( m_ReceiveThread.Join() == Result::TMIPC_RESULT_OK );
    VERIFY( m_SendThread.Join() == Result::TMIPC_RESULT_OK );

    // Start Send/Receive threads.
    m_SendThread.Start( SendThread, this, m_SendThreadStack, TMIPC_STACK_SIZE, NN_SYSTEM_THREAD_PRIORITY(tma, TmaUsbSend), NN_SYSTEM_THREAD_NAME(tma, TmaUsbSend));
    m_ReceiveThread.Start( ReceiveThread, this, m_ReceiveThreadStack, TMIPC_STACK_SIZE, NN_SYSTEM_THREAD_PRIORITY(tma, TmaUsbRecv), NN_SYSTEM_THREAD_NAME(tma, TmaUsbRecv));

    // Register the threads.
    tma::ThreadTracker::RegisterThread( m_SendThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcSend );
    tma::ThreadTracker::RegisterThread( m_ReceiveThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcReceive );
}

// ============================================================================
// Stops the Send/Receive threads.
void NodeUSB::StopThreads()
{
    // Cancel any existing blocking calls.
    USBInterface::AbortBlockingCalls();
    VERIFY( m_ReceiveThread.Join() == Result::TMIPC_RESULT_OK );
    VERIFY( m_SendThread.Join() == Result::TMIPC_RESULT_OK );

    // Unregister the threads.
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcSend );
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcReceive );
}

// ============================================================================
// Callback function from the USBInterface.
void NodeUSB::EventCallback( USBEvent Event, void* lpThis )
{
    DEJA_TRACE( "NodeUSB::EventCallback", "Event callback" );
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );
    ASSERT( lpThis != nullptr );
    if( lpThis != nullptr )
    {
        NodeUSB* pThis = reinterpret_cast<NodeUSB*>(lpThis);
        switch( Event )
        {
        case USBEvent::Reset:
        {
            pThis->StopThreads();
        }
        break;
        case USBEvent::Start:
        {
            pThis->StartThreads();
        }
        break;
        case USBEvent::Quit:
        {
//TODO: What to do here?  The Quit means the USBInterface was shut down...
//      Should this node disconnect?
            pThis->StopThreads();
        }
        break;
        default:
        {
            DEJA_TRACE( "NodeUSB::EventCallback", "Unknown event number: %d", Event );
        }
        break;
        }
    }
TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
}

// ============================================================================
// Send thread.
s32 NodeUSB::SendThread( void* lpThis )
{
    DEJA_TRACE( "NodeUSB::SendThread", "SendThread( %p )", lpThis );
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    s32 Result = TMIPC_RESULT_OK;

    NodeUSB* pThis = reinterpret_cast<NodeUSB*>(lpThis);

    // Loop until the connection "closes".
    bool    NullPacketWasReceived = false;
    Packet* pPacket = nullptr;
    while( !NullPacketWasReceived && (Result == TMIPC_RESULT_OK) )
    {
        // Read a packet from the Message Queue.
        pPacket = (Packet*)pThis->m_SendQueue.Receive();
        NullPacketWasReceived = pPacket == nullptr;

        // Terminate thread if a nullptr packet.
        if( pPacket != nullptr)
        {
            // Send the packet over the Socket.
            if( pThis->IsConnected() )
            {
                // Reset this flag, to indicate that an abort was not issued.
                pThis->m_bAbortedBlockingCalls = false;
                Result = USBInterface::Write( pPacket );
                if( Result == TMIPC_RESULT_OK )
                {
                    ADD_SENT_PACKET( *pPacket );
                }
                else
                {
                    DEJA_TRACE( "NodeUSB::SendThread", "Failed to Send: %p, %d bytes. Result == %d", pPacket->GetBuffer(), pPacket->GetDataLen(), Result );
                    // It is expected behavior that the write command could fail if blocked.
                    // Removing this assert. This can trigger when a USB device is active in TM,
                    // then disconnected, then pinged by a target discovery beacon.
                    // ASSERT( 0 );
                }

                // An invalid Write code could be from an Abort operation.
                if( pThis->m_bAbortedBlockingCalls )
                {
                    Result = TMIPC_RESULT_OK;
                }
            }

            // Free the packet.
            pThis->FreePacket( pPacket );
            pThis->Tick();
        }
    }

    // Send Diconnected to the Work Thread.
    pThis->Disconnected();

    // Delete any pending packets.
    while( pThis->m_SendQueue.GetCount() > 0 )
    {
        Packet* pPacket = (Packet*)pThis->m_SendQueue.Receive();
        if( pPacket )
        {
            pThis->FreePacket( pPacket );
        }
    }

    // ASSERT no more packets.
    ASSERT( pThis->m_SendQueue.GetCount() == 0 );

    DEJA_TRACE( "NodeUSB::SendThread", "SendThread Stopped" );

TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
    // Thread exit.
    return Result;
}

// ============================================================================
// Receive thread.
s32 NodeUSB::ReceiveThread( void* lpThis )
{
    DEJA_THREAD_LABEL( "NodeUSB::ReceiveThread" );
    DEJA_TRACE( "NodeUSB::ReceiveThread", "ReceiveThread Started" );
    tmipc::Result Result = { TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    NodeUSB* pThis = reinterpret_cast<NodeUSB*>(lpThis);

    // Loop until the connection is shut down or disconnected.
    bool bDone = false;
    while( !bDone && (Result == TMIPC_RESULT_OK) )
    {
        // Allocate a packet.
        Packet* pPacket = pThis->AllocRecvPacket();
        ASSERT( pPacket );

        // Read the packet.
        if( Result == TMIPC_RESULT_OK )
        {
            Result = USBInterface::Read( pPacket );
            if( Result == TMIPC_RESULT_OK )
            {
                ADD_RECEIVED_PACKET( *pPacket );
            }
            else
            {
                DEJA_TRACE( "NodeUSB::ReceiveThread", "Failed to read packet: %p", pPacket->GetBuffer() );
            }
        }

        // Dispatch the packet as a Work Item.
        if( Result == TMIPC_RESULT_OK )
        {
            // What type of packet was this?
            const u32 ServiceId = pPacket->GetServiceId();
            const bool bNormalPacket
            {
                    (ServiceId != ServiceId_NodeUSB_BeaconQuery)
                &&  (ServiceId != ServiceId_NodeUSB_TMSInfo)
                &&  (ServiceId != ServiceId_NodeUSB_ConnectHandshake)
                &&  (ServiceId != ServiceId_NodeUSB_Disconnect)
            };

            if( bNormalPacket )
            {
                pThis->ProcessReceived( pPacket );
                pPacket = nullptr;
            }
            else
            {
                switch( ServiceId )
                {
                case ServiceId_NodeUSB_ConnectHandshake:
                {
                    // Send the "dummy nack" back to Target Manager - *BEFORE* acknowledging a successful connection.
                    // Sending this *after* the NodeEvent_Connected opens the floodgates for SentThread to start sending data.
                    // This should fix Siglo-62461.
                    Result = USBInterface::SendBeaconPacket(pPacket);

                    // This is a bit of a leap of faith to "become connected"
                    // before sending the Beacon response, but it must happen
                    // before the Target Manager gets it - the packet is the
                    // "connection confirmation".
                    pThis->m_bIsConnected = true;
                    if( Result == TMIPC_RESULT_OK )
                    {
                        pThis->OnEvent( NodeEvent_Connected );
                    }

                    // Respond to the handshake Packet.
                    pPacket->ResetCursor();
                }
                break;
                case ServiceId_NodeUSB_TMSInfo:
                {
TMA_POWER_TEST_PRINT( "[tmipc_node_usb] Received ServiceId_NodeUSB_TMSInfo.  Packet length: %d.\n", pPacket->GetDataLen() );
                    // Parse the TMSInfo.
                    s32 StringLength = { 0 };
                    const s32 TMSInfoLength = { TMIPC_TMSINFO_PACKET_SIZE - sizeof(StringLength) };
                    char TMSInfo[TMSInfoLength];
                    pPacket->ReadS32( StringLength );
                    if( StringLength <= TMSInfoLength )
                    {
                        pPacket->ReadData( TMSInfo, StringLength );
                        pThis->ProcessTMSInfoString( TMSInfo );
                    }
                }
                break;
                case ServiceId_NodeUSB_BeaconQuery:
                {
                    // Receiving a "Beacon request" means a connection does not exist.
                    pThis->m_bIsConnected = false;

                    // Read the TargetManager's version number.
                    if( Result == TMIPC_RESULT_OK )
                    {
                        u32 Version = 0;
                        pPacket->ReadU32( Version );
                    }

                    // Cancel any pending tasks...
                    if( Result == TMIPC_RESULT_OK )
                    {
                        // ...unless the Target has just woken up.
                        if (!tma::GetHasWokenUp())
                        {
TMA_POWER_TEST_PRINT( "[%s]!!! - Cancelling all: received ServiceId_NodeUSB_BeaconQuery.  Has woken up? '%s'.\n", _BestFunctionName_, tma::GetHasWokenUp() ? "Yes" : "No" );
                            pThis->CancelAllPendingTasks();
                        }
                    }

                    // Send the requested information back to TargetManager.
                    if( Result == TMIPC_RESULT_OK )
                    {
                        // Reset the packet cursor, for the response packet.
                        pPacket->ResetCursor();
                        Result = USBInterface::SendBeaconPacket( pPacket );
                    }
                }
                break;
                case ServiceId_NodeUSB_Disconnect:
                {
                    pThis->OnEvent( NodeEvent_Disconnected );

                    // Receiving a "Disconnect" means a connection does (will) not exist.
                    pThis->m_bIsConnected = false;

                    // Cancel any pending tasks.
                    if( Result == TMIPC_RESULT_OK )
                    {
TMA_POWER_TEST_PRINT( "[%s]!!! - Cancelling all: received ServiceId_NodeUSB_Disconnect.\n", _BestFunctionName_ );
                        pThis->CancelAllPendingTasks();
                    }
                }
                break;
                default:
                    ASSERT( 0 && "Unexpected ServiceId" );
                    break;
                }

                // There is no connection, therefore the packet must be
                // manually deallocated.
                pThis->FreePacket( pPacket );
            }
        }
        else
        {
            bDone = true;
            pThis->FreePacket( pPacket );
        }
    }

    // Send a nullptr to terminate the SendThread.
    pThis->m_SendQueue.Send( nullptr );

    // The send/receive threads will end, which means the connection be gone.
    pThis->m_bIsConnected = false;

    DEJA_TRACE( "NodeUSB::ReceiveThread", "ReceiveThread Stopped" );

TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
    // Thread exit.
    return Result;
}//NOLINT(impl/function_size)

// ============================================================================
// Disconnect from the current connection.
Result NodeUSB::Disconnect()
{
    DEJA_TRACE( "NodeUSB::Disconnect", "Starting thread" );
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );
    Result Result = TMIPC_RESULT_OK;

    // Remove all callbacks.
    USBInterface::SetCallbackInformation( nullptr, nullptr );

    // Abort blocking calls.
    USBInterface::AbortBlockingCalls();

    // Recv thread should terminate when socket closes.
    VERIFY( m_ReceiveThread.Join() == TMIPC_RESULT_OK );

    // The ReceiveThread sends a nullptr to terminate the Send Thread.
    VERIFY( m_SendThread.Join() == TMIPC_RESULT_OK );

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    DEJA_TRACE( "NodeUSB::Disconnect", "Exiting thread" );

    return Result;
}

// ============================================================================
// Returns 'true' if there is a valid connection.
bool NodeUSB::IsConnected()
{
    return m_bIsConnected;
}

// ============================================================================
// Sends a packet to the remote connection.
Result NodeUSB::Send( Packet* pPacket )
{
    DEJA_TRACE( "NodeUSB::Send", "Send( %p )", pPacket );
    Result Result = TMIPC_RESULT_OK;

    // Make sure a connection has been made.
    if( Result == TMIPC_RESULT_OK )
    {
        ScopedLock lock( m_Lock );

        if( IsConnected() )
        {
            // Add the packet to the Message Queue.
            m_SendQueue.Send( pPacket );
        }
        else
        {
            // No connection, just free the packet.
            FreePacket( pPacket );
            Result = TMIPC_RESULT_DISCONNECTED;

            // Need to Tick the WorkThread if we get here to process the FreePacket.
            Tick();
        }
    }

    return Result;
}

// ============================================================================
} // namespace tmipc
// ============================================================================

#endif // TMIPC_TARGET_HORIZON
