﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_node_tics_interface.h"
#include "tmipc_node_tics_plug.h"
#include "../DejaInsight.h"
#include "../Version.h"
#include <BridgeAPIInternal.h>
#include <HotBridgeAPI.h>
#include <nn/tics/tics_Api.h>
#include <nn/nn_SdkLog.h>
#include "../thread_tracker.h"
#include "../tmagent.h"

//==============================================================================
namespace tmipc {
//==============================================================================

// Local namespace.
namespace
{

// --------------------------------------------------------------------------
// This class is required, to receive events from tics::BridgeInstance.
class TicsBridgeInstance : public tics::BridgeInstance
{
    typedef tics::BridgeInstance BaseClass;

private:

static  bool                m_StaticInitialized;  // This is used to allocate "one time only" objects, namely the heap, etc.
static  const size_t        m_HeapSize;
static  uintptr_t           m_HeapAddress;
static  tmipc::Event        m_ShutdownEvent;
static  tmipc::Event        m_ConfigEvent;
static  TicsBridgeInstance* m_pTicsBridgeInstance;
static  tics::BridgeHandle  m_hBridge;
static  TICSPlug*           m_pPlug;
static  bool                m_ShuttingDown;
static  tics::BridgeConfigSectionAttrGetInfo m_ConfigAttrInfo;


private:

    // -------------
    // API callbacks

    // Callback that occurs when any channel operation (add, delete, update) has been executed.
static  int     OnChannelChange( void* lpThis, tics::EOperation Operation, const char* Name, bool bHIO, int Id, tics::UpdateChannelData* ChannelData );
    // Callback that occurs when the CrossBar is shutting down.
static  int     OnShutdown( void* lpNullptr );
    // Callback that occurs when the host app releases the bridge.
virtual int     OnReleaseRequest( const ReleaseRequest* pRequest, size_t Length );
    // Callback that occurs when the host takes over the bridge.
virtual void    OnConfigReset( void );
    // Process a remote "non-graceful shutdown".  This only seems to be called
    // when the Remote Host quits in a non-graceful session close.  Graceful
    // Remote Host shutdowns will end up calling OnReleaseRequest.
virtual int     OnRemoteDetach( tics::DisconnectReason reason );

    // -----------------------------------
    // TICSPlug callback "notifications".

    // A session (plug) has detached callback.
static  int     SessionDetachedCallback( void* lpThis );

    // -----------------------------------
    // Internal functions.

    // Returns the BridgeHandle.
static  tics::BridgeHandle  GetBridgeHandle();

    // Returns the CrossBar.
static  tics::CrossBar*     GetCrossBar();

// Keep the Constructor and Destructors private.  Use Initialize() to start TICS.
                TicsBridgeInstance();
virtual         ~TicsBridgeInstance();

public:

    // Initialize the TICS interface.
    static tmipc::Result Initialize();

    // Finalizes TICS.
    static tmipc::Result Finalize();

    // Returns true, if a connection exists.
    static bool          IsConnected();

    // Directly toggle the status LEDs, Cradle indicator, etc...
    static tmipc::Result SetTargetStatusFlag( tics::ETargetStatusFlag Flag, bool State );

    // Get Configuration from HostBridge
    static tmipc::Result ConfigSectionAttrGet( const char* sectionName, const char *attrName, char *attrValue, int attrValueLen  );
    static int GetConfigSectionAttrCallback( void* ctx, int operationResult, void* operationResources );

    // Starts a single TICS session.  Blocking call.
    static tmipc::Result StartSession();

    // Stops a single TICS session.
    static tmipc::Result StopSession();

    // Hack: Notifies this class to clean up a "completed" TICS session, to
    //       clean up memory allocations, etc.
    static tmipc::Result CleanupSession();

    // Reads data from the TICS connection.  Blocking call.
    // Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
    static tmipc::Result Read( tmipc::Packet& Packet );

    // Writes data to the TICS connection.  Blocking call.
    // Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
    static tmipc::Result Write( const tmipc::Packet& Packet );
};

// Static data.
bool                TicsBridgeInstance::m_StaticInitialized = false;
const size_t        TicsBridgeInstance::m_HeapSize = { 8 * 1024 * 1024 }; // 8MB.
uintptr_t           TicsBridgeInstance::m_HeapAddress = { 0 };
tmipc::Event        TicsBridgeInstance::m_ShutdownEvent;
tmipc::Event        TicsBridgeInstance::m_ConfigEvent;
TicsBridgeInstance* TicsBridgeInstance::m_pTicsBridgeInstance = { nullptr };
tics::BridgeHandle  TicsBridgeInstance::m_hBridge = { nullptr };
TICSPlug*           TicsBridgeInstance::m_pPlug = { nullptr };
bool                TicsBridgeInstance::m_ShuttingDown = { false };

tics::BridgeConfigSectionAttrGetInfo    TicsBridgeInstance::m_ConfigAttrInfo;

// --------------------------------------------------------------------------
// Constructor.
TicsBridgeInstance::TicsBridgeInstance()
: BaseClass( GetBridgeHandle(), true, nullptr )
{
    DEJA_TRACE( "TicsBridgeInstance::TicsBridgeInstance", "Constructor" );
}

// --------------------------------------------------------------------------
// Destructor.
TicsBridgeInstance::~TicsBridgeInstance()
{
    DEJA_TRACE( "TicsBridgeInstance::~TicsBridgeInstance", "Destructor" );
}

// --------------------------------------------------------------------------
// Returns the BridgeHandle.
inline tics::BridgeHandle TicsBridgeInstance::GetBridgeHandle()
{
    return m_hBridge;
}

// --------------------------------------------------------------------------
// Returns the CrossBar.
inline tics::CrossBar* TicsBridgeInstance::GetCrossBar()
{
    tics::CrossBar* pCrossBar = nullptr;

    if( m_pTicsBridgeInstance != nullptr )
    {
        // We cast the m_pTicsBridgeInstance pointer to the base class because GetCrossBar is
        // a static function and without the cast the compiler calls the static version even
        // though we have an object pointer resulting in recursion.
        // Ultimately, we want the CrossBar associated with the BridgeInstance returned here.
        pCrossBar = ((BridgeInstance*)m_pTicsBridgeInstance)->GetCrossBar();
    }

    return pCrossBar;
}

// --------------------------------------------------------------------------
// Callback that occurs when any channel operation (add, delete, update) has been executed.
int TicsBridgeInstance::OnChannelChange( void* lpThis, tics::EOperation Operation, const char* Name, bool bHIO, int Id, tics::UpdateChannelData* pChannelData )
{
    DEJA_TRACE( "TicsBridgeInstance::OnChannelChange", "OnChannelChange(..., %d, '%s', '%s', %d, ...)", Operation, (Name != nullptr) ? "Name" : "<null>", bHIO ? "true" : "false" );
    (void)bHIO;
    (void)Id;
    (void)pChannelData;
    // Using m_ShuttingDown is a race condition waiting to happen. :(
    if (!m_ShuttingDown)
    {
        if (Operation == tics::EOperation::EAdded)
        {
            // This name needs to match the tms's expected name:
            // .\Oasis\Source\TargetManager\tmipc\src\tmipc_node_tics_interface.cpp,
            // In function: TICSInterface::OnTakeoverComplete( ... ).
            static const char TMSChannelName[] = "tma_hb";

            // Make sure this is the TMS channel before accepting it.
            // Using Chris C's recommendation to weed out foreign chanels.
            // Oasis-925: TMA should check the id of the channel added.
            if ((Name != nullptr) && (strcmp(Name, TMSChannelName) == 0))
            {
                TicsBridgeInstance *pThis = reinterpret_cast<TicsBridgeInstance *>(lpThis);
                // Can multiple of "EAdded messages" be issued?  At least prevent
                // a memory leak (at the least)...
                // Again, from Oasis-925's example.
                if (pThis->m_pPlug == nullptr)
                {
                    pThis->m_pPlug = new TICSPlug(Name);

                    // Allow the user to register a callback when a send and/or receive has finished.
                    pThis->m_pPlug->RegisterOnSessionStartedCallback(TICSInterface::SessionStartedCallback, nullptr);
                    pThis->m_pPlug->RegisterOnSessionDetachedCallback(SessionDetachedCallback, pThis);

                    // Register Session Start.
                    GetCrossBar()->RegisterForSessionStart(Name, pThis->m_pPlug);
                }
            }
        }
    }
    else
    {
TMA_POWER_TEST_PRINT("[%s] !!! - Ignoring the OnChannelChange callback because m_ShuttingDown == true.\n", _BestFunctionName_);
    }

    return tics::BRIDGE_ERROR_SUCCESS;
}

// --------------------------------------------------------------------------
// Callback that occurs when the CrossBar is shutting down.
int TicsBridgeInstance::OnShutdown( void* lpNullptr )
{
    DEJA_TRACE( "TicsBridgeInstance::OnShutdown", "OnShutdown" );
    (void)lpNullptr;
    // Alert the StartSession that shutdown is done.
    m_ShutdownEvent.Set();

    return tics::BRIDGE_ERROR_SUCCESS;
}

// --------------------------------------------------------------------------
// Callback that occurs when the host app releases the bridge.
int TicsBridgeInstance::OnReleaseRequest( const ReleaseRequest* pRequest, size_t Length )
{
    DEJA_TRACE( "TicsBridgeInstance::OnReleaseRequest", "OnReleaseRequest( %p, %u).\n", pRequest, Length );

    GetCrossBar()->BeginShutdown( tics::CrossBar::ExitDone );
    return tics::BRIDGE_ERROR_SUCCESS;
}

// --------------------------------------------------------------------------
// Callback that occurs when the host takes over the bridge.
void TicsBridgeInstance::OnConfigReset()
{
    DEJA_TRACE( "TicsBridgeInstance::OnConfigReset().", "OnConfigReset" );

    tics::BridgeNotifications Notifications;
    memset( &Notifications, 0, sizeof(Notifications) );
    Notifications.OnChannelChange = OnChannelChange;
    const int Result = tics::BridgeRegister( GetBridgeHandle(), &Notifications, this );
    (void)Result;
}

// --------------------------------------------------------------------------
// Process a remote "non-graceful shutdown".  This only seems to be called
// when the Remote Host quits in a non-graceful session close.  Graceful
// Remote Host shutdowns will end up calling OnReleaseRequest.
int TicsBridgeInstance::OnRemoteDetach( tics::DisconnectReason reason )
{
    DEJA_TRACE( "TicsBridgeInstance::OnRemoteDetach().", "OnRemoteDetach" );

    // The Target will need to restart a new session, if needed.
    GetCrossBar()->BeginShutdown( tics::CrossBar::ExitDone );
    return tics::BRIDGE_ERROR_SUCCESS;
}

// --------------------------------------------------------------------------
// A session (plug) has detached callback.
int TicsBridgeInstance::SessionDetachedCallback( void* lpThis )
{
    DEJA_TRACE( "TicsBridgeInstance::SessionDetachedCallback", "SessionDetachedCallback" );
    int Result = { tics::BRIDGE_ERROR_SUCCESS };
    TicsBridgeInstance* pThis = reinterpret_cast<TicsBridgeInstance*>(lpThis);
    (void)pThis;
    (void)lpThis;

    // The Target will need to restart a new session, if needed.
    GetCrossBar()->BeginShutdown( tics::CrossBar::ExitDone );
    return Result;
}

// --------------------------------------------------------------------------
// Initialize the TICS interface.  ThreadPriority will persist until
// Finalize() is called - only then can ThreadPriority be changed.
tmipc::Result TicsBridgeInstance::Initialize()
{
    DEJA_TRACE( "TicsBridgeInstance::Initialize", "Initialize()" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );
    m_ShutdownEvent.Create();
    m_ConfigEvent.Create();

    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( (m_hBridge != nullptr) || (m_pTicsBridgeInstance != nullptr) )
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
            TMIPC_ERROR_LOG( "Bridge instance already initialized.\n" );
        }
//TODO: is this too strong of a failure?  If this is called twice without a
//      Finalize() in between them, then it will most likely not work anyways.
        NN_ABORT_UNLESS( m_hBridge == nullptr, "TICSInterface::Initialize: m_hBridge is not null - already initialized." );
        NN_ABORT_UNLESS( m_pTicsBridgeInstance == nullptr, "TICSInterface::Initialize: m_pTicsBridgeInstance is not null - already initialized." );
    }

    // Allocate a heap for tics.
    if( !m_StaticInitialized )
    {
        // This is the one time code in here should be run.
        m_StaticInitialized = true;
        m_HeapAddress = (uintptr_t)tma::s_Allocate( m_HeapSize );
        nn::tics::Initialize( m_HeapAddress, m_HeapSize );
    }


    // Create the Bridge.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        // Create a bridge handle.
        m_hBridge = tics::BridgeHandleCreateByName( "tma_hb" );

        // Create the Tics BridgeInstance.
        m_pTicsBridgeInstance = new TicsBridgeInstance();
    }

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    return Result;
}

// --------------------------------------------------------------------------
// Finalizes TICS.
tmipc::Result TicsBridgeInstance::Finalize()
{
    DEJA_TRACE( "TicsBridgeInstance::Finalize", "Finalize" );
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Deregister notifications.
    if( GetBridgeHandle() != nullptr )
    {
        const int DeregisterResult = tics::BridgeDeregister( GetBridgeHandle() );
        (void)DeregisterResult;
    }

    // Deallocate the Bridge Instance.
    // NOTE: This appears to only be safe after CrossBar()->Run() is called
    //       and returns.
    //       If it was allocated and CrossBar()->Run() was not called/returned,
    //       the deallocation seems to cause a crash.
    delete m_pTicsBridgeInstance;
    m_pTicsBridgeInstance = nullptr;

    // Release the bridge, if it was created.
    if( GetBridgeHandle() != nullptr )
    {
        const int FreeResult = tics::BridgeHandleFree( GetBridgeHandle() );
        (void)FreeResult;
    }
    m_hBridge = nullptr;

//TODO: This is not needed to support Sleep/Awake. Calling this will cause problems.
//    nn::tics::Finalize();

    // Deallocate the Heap given to tics.
// Actually, this causes a problem when Initialize() is called again, and it is
// not needed for the Sleep/Awake support.
//    nn::os::FreeMemoryBlock( m_HeapAddress, m_HeapSize );
//m_HeapAddress = 0;

    m_ShutdownEvent.Destroy();
    m_ConfigEvent.Destroy();
TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );

    return Result;
}

// --------------------------------------------------------------------------
// Returns true, if a connection exists.
bool TicsBridgeInstance::IsConnected()
{
    return m_pPlug != nullptr;
}

// --------------------------------------------------------------------------
// Directly toggle the status LEDs, Cradle indicator, etc...
tmipc::Result TicsBridgeInstance::SetTargetStatusFlag( tics::ETargetStatusFlag Flag, bool State )
{
    DEJA_TRACE( "TicsBridgeInstance::SetTargetStatusFlag", "Flag %lld", Flag );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    m_pTicsBridgeInstance->BridgeSetTargetStatus( Flag, State, NULL, NULL );

    return Result;
}

// --------------------------------------------------------------------------
//
int TicsBridgeInstance::GetConfigSectionAttrCallback( void* ctx, int operationResult, void* operationResources )
{
    ASSERT( ctx != nullptr );
    TicsBridgeInstance* tbi = (TicsBridgeInstance*)ctx;

    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( operationResult < 0)
    {
        const char* ipaddr = tics::BridgeHandleGetIP( tbi->m_hBridge );

        Result = tmipc::TMIPC_RESULT_FAILED;
    }
    else
    {
        memcpy(&m_ConfigAttrInfo, static_cast<tics::BridgeConfigSectionAttrGetInfo*>( operationResources ), sizeof(m_ConfigAttrInfo));
    }

    m_ConfigEvent.Set();

    return Result;
}

// --------------------------------------------------------------------------
//
tmipc::Result TicsBridgeInstance::ConfigSectionAttrGet( const char* sectionName, const char *attrName, char *attrValue, int attrValueLen )
{
    m_ConfigEvent.Reset();

    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( (m_hBridge != nullptr) || (m_pTicsBridgeInstance != nullptr) )
    {
        int result = m_pTicsBridgeInstance->BridgeConfigSectionAttrGet( sectionName, attrName, m_pTicsBridgeInstance, TicsBridgeInstance::GetConfigSectionAttrCallback );
        if (result == tics::BRIDGE_ERROR_SUCCESS)
        {
            m_ConfigEvent.Wait( TMIPC_INFINITE );

            strncpy(attrValue, m_ConfigAttrInfo.attrValue, attrValueLen);
        }
        else
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
        }
    }
    else
    {
        Result = tmipc::TMIPC_RESULT_FAILED;
    }

    return Result;
}

// --------------------------------------------------------------------------
// Starts a single TICS session.  Blocking call.
tmipc::Result TicsBridgeInstance::StartSession()
{
    DEJA_TRACE( "TicsBridgeInstance::StartSession", "StartSession - Start" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    // Someone is trying to start a new session.
    m_ShuttingDown = false;

    // Make sure there is a bridge instance before proceeding.  If it is null
    // at this point, then it is an indication that the Target is going to sleep.
    // The only way to recover from the nullptr is to call:
    // TicsBridgeInstance::Initialize().
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( m_pTicsBridgeInstance == nullptr )
        {
TMA_POWER_TEST_PRINT( "[%s] !!! - m_pTicsBridgeInstance was nullptr, do not start a new session.\n", _BestFunctionName_ );
            Result = tmipc::TMIPC_RESULT_NOT_INITIALIZED;
        }
    }

    // Keep TICS running in a loop until it is told to stop.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        // Start the Bridge.
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            m_pTicsBridgeInstance->Run();
        }

        // Register for the CrossBar's shutdown message.
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            m_ShutdownEvent.Reset();
            const int RegisterResult = GetCrossBar()->RegisterForShutdown( OnShutdown, nullptr );
            if( RegisterResult != tics::BRIDGE_ERROR_SUCCESS )
            {
                Result = tmipc::TMIPC_RESULT_FAILED;
                TMIPC_ERROR_LOG( "Register for shutdown failed.\n" );
            }
        }

        // Run CrossBar.
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            tics::CrossBar::Exit ExitResult = GetCrossBar()->Run();
            (void)ExitResult;
            if( IsConnected() )
            {
                m_pPlug->CancelBlockingOperations();
            }
            else
            {
                // The TICSPlug was not created, therefore the connection
                // attempt failed.
                Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
            }
        }

        // Wait until the shutdown is called.
        m_ShutdownEvent.Wait( TMIPC_INFINITE );

        // Cleanup the old Instance.
        delete m_pTicsBridgeInstance;
        m_pTicsBridgeInstance = nullptr;
        tics::BridgeHandleFree( GetBridgeHandle() );
        m_hBridge = nullptr;

        // Do not create a new BridgeHandle and TicsBridgeInstance, if we're trying to shut down.
        if( !m_ShuttingDown )
        {
            // Create a new Instance for the next connection
            m_hBridge = tics::BridgeHandleCreateByName( "tma_hb" );
            m_pTicsBridgeInstance = new TicsBridgeInstance();
        }
    }

    // Deallocation of some data members have been moved to the CleanupSession,
    // which is called by the "same caller that called StartSession(). The idea
    // is to allow enough time to occur that "in use" data members have time to
    // stop being accessed/used.
TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );

    return Result;
}

// --------------------------------------------------------------------------
// Stops a single TICS session.
tmipc::Result TicsBridgeInstance::StopSession()
{
    DEJA_TRACE( "TicsBridgeInstance::StopSession", "StopSession - Start" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    // A Shutdown is in progress, make a note of it.
    m_ShuttingDown = true;
    // Keep TICS running in a loop until it is told to stop.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( GetCrossBar() != nullptr )
        {
            GetCrossBar()->BeginShutdown( tics::CrossBar::ExitDone );
        }
    }

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    return Result;
}

// --------------------------------------------------------------------------
// Hack: Notifies this class to clean up a "completed" TICS session, to
//       clean up memory allocations, etc.
tmipc::Result TicsBridgeInstance::CleanupSession()
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    if( m_pPlug != nullptr )
    {
        m_pPlug->CancelBlockingOperations();
    }

    // Yes.  This is dangerous if called when it's still in use.
    delete m_pPlug;
    m_pPlug = nullptr;

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    return Result;
}

// --------------------------------------------------------------------------
// Reads data from the TICS connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result TicsBridgeInstance::Read( tmipc::Packet& Packet )
{
    DEJA_CONTEXT( "TicsBridgeInstance::Read", "Read\n" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Ensure that the Plug Interface was allocated.
    ASSERT( m_pPlug != nullptr );
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( m_pPlug == nullptr )
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
            TMIPC_ERROR_LOG( "Read failed: null plug instance.\n" );
        }
    }

    // Attempt to read a Packet.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        // Set the event, so that it can block when Waited on.
        Result = m_pPlug->Receive( Packet );
    }

    return Result;
} // NOLINT(readability/fn_size)

// --------------------------------------------------------------------------
// Writes data to the TICS connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result TicsBridgeInstance::Write( const tmipc::Packet& Packet )
{
    DEJA_CONTEXT( "TicsBridgeInstance::Write", "Write\n" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Ensure that the Plug Interface was allocated.
    ASSERT( m_pPlug != nullptr );
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( m_pPlug == nullptr )
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
            TMIPC_ERROR_LOG( "Write failed: null plug instance.\n" );
        }
    }

    // Send the Packet header.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = m_pPlug->Send( Packet );
    }

    return Result;
} // NOLINT(readability/fn_size)
}

// --------------------------------------------------------------------------
// TICSInterface
// --------------------------------------------------------------------------

static const int m_SessionThreadStackSize = { 32 * 1024 };
static  NN_OS_ALIGNAS_THREAD_STACK char s_SessionThreadStack[m_SessionThreadStackSize];
Thread* TICSInterface::m_pSessionThread = { nullptr };
Mutex*  TICSInterface::m_pReceiveMutex  = { nullptr };
Mutex*  TICSInterface::m_pSendMutex     = { nullptr };
Event*  TICSInterface::m_pStartedEvent  = { nullptr };
bool    TICSInterface::m_bStopSession   = { false };

// --------------------------------------------------------------------------
// Initialize the TICS interface.  ThreadPriority will persist until
// Finalize() is called - only then can ThreadPriority be changed.
tmipc::Result TICSInterface::Initialize()
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Create the send/receive mutexes.
    NN_ABORT_UNLESS( (m_pReceiveMutex == nullptr) && (m_pSendMutex == nullptr) );
    if( (m_pReceiveMutex == nullptr) && (m_pSendMutex == nullptr) )
    {
        m_pSessionThread = new Thread;
        m_pReceiveMutex  = new Mutex;
        m_pSendMutex     = new Mutex;
        m_pStartedEvent  = new Event;
        m_pReceiveMutex->Create();
        m_pSendMutex->Create();
        m_pStartedEvent->Create();
    }

    // Initialize the TicsBridgeInstance.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = TicsBridgeInstance::Initialize();
    }
    if( Result != tmipc::TMIPC_RESULT_OK )
    {
        TMIPC_ERROR_LOG( "Tics Bridge Instance Initialize failed.\n" );
    }

    return Result;
}

// --------------------------------------------------------------------------
// Finalize the TICS wrapper.
tmipc::Result TICSInterface::Finalize()
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    // TICS is being finalized, do not allow the SessionThread to continue running.
    m_bStopSession = true;
    if( m_pStartedEvent != nullptr )
    {
        m_pStartedEvent->Set();
    }

    if( m_pSessionThread != nullptr )
    {
        m_pSessionThread->Join( TMIPC_INFINITE );
        delete m_pSessionThread;
        m_pSessionThread = nullptr;
        tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcSession );
    }

    // Destroy the Send/Receive mutexes and Started event.
    delete m_pStartedEvent;
    m_pStartedEvent = nullptr;

    delete m_pSendMutex;
    m_pSendMutex = nullptr;

    delete m_pReceiveMutex;
    m_pReceiveMutex = nullptr;

    // Finalize the TicsBridgeInstance.
//TODO: Should this call Finalize anyways?  It seems like it should...
//    if( Result == tmipc::TMIPC_RESULT_OK )
//    {
//        Result = TicsBridgeInstance::Finalize();
//    }
    const tmipc::Result FinalizeResult = TicsBridgeInstance::Finalize();
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = FinalizeResult;
    }

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    return Result;
}

// --------------------------------------------------------------------------
// The TICS thread.
int TICSInterface::SessionThread( void* lpNull )
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    // Run the TICS session.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = TicsBridgeInstance::StartSession();
    }

    // The Session (connection) may not have started.  Also make sure the
    // connection is valid.
    if( (Result != tmipc::TMIPC_RESULT_OK) || !TicsBridgeInstance::IsConnected() )
    {
        Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
        // Set the "Session Started event" so that the StartSession's
        // m_pStartedEvent->Wait(...) returns.
        NN_ABORT_UNLESS( m_pStartedEvent != nullptr );
        m_pStartedEvent->Set();
    }

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
    return Result;
}

// --------------------------------------------------------------------------
// A session has started callback.
void TICSInterface::SessionStartedCallback( void* lpNull )
{
    DEJA_TRACE( "TICSInstance::SessionStartedCallback", "SessionStartedCallback" );
    (void)lpNull;
    // The session has started.
    // Set the event to allow StartSession(...)'s "Wait" to exit.
    m_pStartedEvent->Set();
}

// --------------------------------------------------------------------------
//
tmipc::Result TICSInterface::SetTargetStatusFlag( tics::ETargetStatusFlag Flag, bool State )
{
    DEJA_CONTEXT( "TicsBridgeInstance::SetTargetStatusFlag", "SetTargetStatusFlag" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = TicsBridgeInstance::SetTargetStatusFlag( Flag, State );
    }

    return Result;
}

// --------------------------------------------------------------------------
//
tmipc::Result TICSInterface::ConfigSectionAttrGet( const char* sectionName, const char *attrName, char *attrValue, int attrValueLen )
{
    DEJA_CONTEXT( "TicsBridgeInstance::ConfigSectionAttrGet", "ConfigSectionAttrGet" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = TicsBridgeInstance::ConfigSectionAttrGet( sectionName, attrName, attrValue, attrValueLen );
    }

    return Result;
}

// --------------------------------------------------------------------------
// Runs the TICS instance.
tmipc::Result TICSInterface::StartSession( s32 ThreadPriority )
{
    DEJA_CONTEXT( "TicsBridgeInstance::StartSession", "Run( %d )\n", ThreadPriority );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Start the SessionThread.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        m_bStopSession = false;
        while( !TicsBridgeInstance::IsConnected() && !m_bStopSession )
        {
            // Make sure the thread has completely shut down.
            {
                // This may be overkill, but if the m_pSessionThread is expected
                // to join (quit), then the event should be Set().
                m_pStartedEvent->Set();
                m_pSessionThread->Join( TMIPC_INFINITE );
            }

            // Set the event, so we can wait on it below.  The event will be reset
            // when the session has started.
            m_pStartedEvent->Reset();
            Result = m_pSessionThread->Start( SessionThread, nullptr, s_SessionThreadStack, m_SessionThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TICS_SESSION), NN_SYSTEM_THREAD_NAME(tma, TICS_SESSION) );

            // Wait for the TICS initialization to finish.
            if( Result == tmipc::TMIPC_RESULT_OK )
            {
                tma::ThreadTracker::RegisterThread( m_pSessionThread->GetThreadType(), tma::ThreadTracker::ThreadId::TmipcSession );

                m_pStartedEvent->Wait( TMIPC_INFINITE );
                if( !TicsBridgeInstance::IsConnected() )
                {
                    // The connection attempt failed.
                    Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
                }
            }
            else
            {
                TMIPC_ERROR_LOG( "Start session thread failed.\n" );
            }
        }
    }

    return Result;
}

// --------------------------------------------------------------------------
// Stops a single TICS session.  Blocking call.
tmipc::Result TICSInterface::StopSession()
{
    DEJA_CONTEXT( "TicsBridgeInstance::StartSession", "Run( %d )\n", ThreadPriority );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Stop the SessionThread.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        m_bStopSession = true;
        Result = TicsBridgeInstance::StopSession();
    }

    if( m_pStartedEvent != nullptr )
    {
        // This may be overkill, but if the m_pSessionThread is expected
        // to join (quit), then the event should be Set().
        m_pStartedEvent->Set();
        m_pSessionThread->Join( TMIPC_INFINITE );
    }

    return Result;
}

// --------------------------------------------------------------------------
// Hack: Notifies the TICS session to end.  Allows internal data to be
//       cleaned up.
tmipc::Result TICSInterface::CleanupSession()
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Allow the TicsBridgeInstance to clean up.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = TicsBridgeInstance::CleanupSession();
    }

    return Result;
}


// --------------------------------------------------------------------------
// Reads data from the TICS connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result TICSInterface::Read( tmipc::Packet& Packet )
{
    DEJA_CONTEXT( "TicsBridgeInstance::Read", "Read\n" );
    NN_ABORT_UNLESS( m_pReceiveMutex != nullptr );
    ScopedLock ScopedLock( *m_pReceiveMutex );

    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
    // Set the event, so that it can block when Waited on.
    Result = TicsBridgeInstance::Read( Packet );

    return Result;
}

// --------------------------------------------------------------------------
// Writes data to the TICS connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result TICSInterface::Write( const tmipc::Packet& Packet )
{
    const tmipc::Result Result = { TicsBridgeInstance::Write( Packet ) };
    return Result;
}

// --------------------------------------------------------------------------
// Sends a Beacon request back to the remote connection.
tmipc::Result TICSInterface::SendBeaconPacket( tmipc::Packet& Packet )
{
    tmipc::Result Result = tmipc::TMIPC_RESULT_OK;
    NN_ABORT_UNLESS( m_pSendMutex != nullptr );
    ScopedLock ScopedLock( *m_pSendMutex );

    // Retrieve the Beacon response.
    char YamlString[256] = "";
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        // NOTE: the "Host Bridge" string *must* match the one in
        //       TargetManager, or the the connection type (Host Bridge vs.
        //       USB) could be misinterpreted.
        if( !tma::GenerateBeaconResponse(YamlString, sizeof(YamlString), "HBX", false) )
        {
            Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
            TMIPC_ERROR_LOG( "Generate Beacon Response failed.\n" );
        }
    }

    // Reset the packet cursor, for the response packet.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Packet.ResetCursor();
        Packet.WriteString( YamlString );
        Result = TicsBridgeInstance::Write( Packet );
        if( Result != tmipc::TMIPC_RESULT_OK )
        {
            TMIPC_ERROR_LOG( "Send Beacon Packet failed.\n" );
        }
    }

    return Result;
}

// --------------------------------------------------------------------------

//==============================================================================
}
//==============================================================================
