﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\stdafx.h"
#include "tma_MQ.h"
#include <new>

//==============================================================================
namespace tma {
//==============================================================================

MQ::MQ()
:   m_Created   ( false ),
    m_pQueue    ( NULL  ),
    m_Allocated ( false ),
    m_Capacity  ( 0     ),
    m_Count     ( 0     ),
    m_iRead     ( 0     ),
    m_iWrite    ( 0     )
{
}

//==============================================================================

MQ::~MQ( )
{
    if( m_Created )
    {
        Destroy();
    }

    if( m_Allocated )
    {
        s_Deallocate( m_pQueue, sizeof( Msg ) * m_Capacity );
    }
}

//==============================================================================

void MQ::Create( s32 Capacity )
{
    ASSERT( !m_Created );
    ASSERT( Capacity > 0 );

    // Save the capacity
    m_Capacity = Capacity;

    // Allocate storage
    void* pMem = s_Allocate( sizeof( Msg ) * m_Capacity );
    m_pQueue = new (pMem) Msg[m_Capacity];
    ASSERT( m_pQueue );
    m_Allocated = true;

    // Create the message queue
    nn::os::InitializeMessageQueue( &m_q, (uintptr_t*)m_pQueue, Capacity );

    // Created
    m_Created = true;
}

//==============================================================================

void MQ::Create( s32 Capacity, Msg* pStorage )
{
    ASSERT( !m_Created );
    ASSERT( Capacity > 0 );

    // Save the capacity
    m_Capacity = Capacity;

    // Set the queue pointer to the provided storage
    ASSERT( pStorage );
    m_pQueue = pStorage;
    m_Allocated = false;

    // Create the message queue
    nn::os::InitializeMessageQueue( &m_q, (uintptr_t*)m_pQueue, Capacity );

    // Created
    m_Created = true;
}

//==============================================================================

void MQ::Destroy()
{
    ASSERT( m_Created );

    nn::os::FinalizeMessageQueue( &m_q );

    m_Created = false;
}

//==============================================================================

bool MQ::IsValid()
{
    return m_Created;
}

//==============================================================================

bool MQ::Send( void* pMessage, bool Block )
{
    ASSERT( m_Created );

    if( Block )
    {
        nn::os::SendMessageQueue( &m_q, (uintptr_t)pMessage );
        return( true );
    }
    else
    {
        return( nn::os::TrySendMessageQueue( &m_q, (uintptr_t)pMessage ) );
    }
}

//==============================================================================

bool MQ::Receive( void** ppMessage, bool Block )
{
    ASSERT( m_Created );
    ASSERT( ppMessage );

    if( Block )
    {
        uintptr_t outData;
        nn::os::ReceiveMessageQueue( &outData, &m_q );
        *ppMessage = (void*)outData;
        return( true );
    }
    else
    {
        uintptr_t outData;
        bool Success = nn::os::TryReceiveMessageQueue( &outData, &m_q );
        if( Success )
        {
            *ppMessage = (void*)outData;
        }
        return( Success );
    }
}

//==============================================================================

void* MQ::Receive()
{
    void* pMessage;

    bool result = Receive( &pMessage );
    (void)result;
    ASSERT( result );

    return( pMessage );
}

//==============================================================================

s32 MQ::GetCount()
{
    s32 Count;

    // Get the number of entries in the queue
    Count = m_Count;

    // Count
    return Count;
}

//==============================================================================
} // namespace tma
//==============================================================================
