﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hio_task.h"
#include "hio_service.h"
#include "hio_opcodes.h"
#include <nn/fs/fs_FileSystem.h>

//==============================================================================
namespace tma { namespace hio {
//==============================================================================

HostIOTask::HostIOTask( TMA_HOSTIO_FILE_HANDLE Handle )
{
    DEJA_TRACE( "HostIOTask::HostIOTask", "HostIOTask::HostIOTask, Handle = %lld", Handle );
    m_Handle            = Handle;
    m_Result            = 0;
    m_OperationValue    = 0;
}

//==============================================================================

HostIOTask::~HostIOTask()
{
}

//==============================================================================

s32 HostIOTask::GetResult()
{
    //===================================================================
    // If our task was cancelled, it was because we were disconnected.
    //===================================================================
    if( m_Status == StatusCanceled )
    {
        return TMA_IO_RESULT_DISCONNECTED;
    }
    return m_Result;
}

//==============================================================================

HostIOReadTask::HostIOReadTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
    DEJA_TRACE( "HostIOReadTask::HostIOReadTask", "HostIOReadTask::HostIOReadTask, Handle = %lld", Handle );
    m_NumberOfPacketsExpected = 0;
    m_pBytesRead = NULL;
}

//==============================================================================

HostIOReadTask::~HostIOReadTask()
{
}

//==============================================================================

HostIOWriteTask::HostIOWriteTask( TMA_HOSTIO_FILE_HANDLE Handle )
: HostIOTask( Handle )
{
    DEJA_TRACE( "HostIOWriteTask::HostIOWriteTask", "HostIOWriteTask::HostIOWriteTask, Handle = %lld", Handle );
    m_pBytesWritten         = NULL;
}

//==============================================================================

HostIOWriteTask::~HostIOWriteTask()
{
}

//==============================================================================

void HostIOReadTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOTask::OnInitiate", "HostIOTask::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOReadTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================
//Here's the size of the data we send at the start of each packet.
enum
{
    SIZE_OF_HOST_IO_HEADER  = ( sizeof(s32) + sizeof(u64) + sizeof(u64)  )
};

//==============================================================================

void HostIOReadTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Get our data
    s32 AmountRead = 0;

    pPacket->ReadS32( AmountRead );
    pPacket->ReadS32 ( m_Result );

    // NN_SDK_LOG("HostIOReadTask::OnRecvPacket - AmountRead = %d, Result = %d\n", AmountRead, m_Result);

    if (m_Result <= TMA_IO_RESULT_OK)
    {
        //Read our data
        pPacket->ReadData( m_WriteAddr, AmountRead );

        //Update where we write to.
        m_WriteAddr = (void*)( ((u64)m_WriteAddr) + AmountRead);
    }

    // Update our variables
    m_OperationValue += AmountRead;
    *m_pBytesRead = (int32_t)m_OperationValue;
    m_NumberOfPacketsExpected -= 1;

    //If we're through with our packets, or there was a problem...
    if ( m_NumberOfPacketsExpected <= 0 || m_Result >= TMA_IO_RESULT_OK || AmountRead == 0 )
    {
        // Done.
     //   NN_SDK_LOG("HostIOReadTask::OnRecvPacket - Complete\n");
        Complete();
    }
}

//==============================================================================

void HostIOReadTask::Read( int32_t* pBytesRead, int64_t Offset, void* pDataBuffer, int64_t AmountToRead )
{
enum
{
    SIZE_OF_RECV_BUFFER_HEADER = ( sizeof(s32) + sizeof(s32) ),
    SIZE_OF_RECV_DATA_BUFFER = ( (tmipc::MAX_PACKET_DATA) - (SIZE_OF_RECV_BUFFER_HEADER) )
};
    //NN_SDK_LOG("HostIOReadTask::Read - Offset: %lld; AmountToRead: %lld\n", Offset, AmountToRead);

    //Init these to known values.
    m_pBytesRead                = pBytesRead;
    m_NumberOfPacketsExpected   = (int32_t)((AmountToRead / SIZE_OF_RECV_DATA_BUFFER ) + 1);
    m_pDataBuffer               = pDataBuffer;
    m_WriteAddr                 = pDataBuffer;

    m_TaskType = tmipc::TaskType_HioFileRead;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteS64( Offset );
    p->WriteS64( AmountToRead );

    m_pServicesManager->SubmitTask( this, p );
}


//==============================================================================

void HostIOWriteTask::Write( int32_t* pBytesWritten, int64_t Offset, void* pDataBuffer, int64_t AmountToWrite )
{
#define SIZE_OF_SEND_BUFFER_HEADER ( (SIZE_OF_HOST_IO_HEADER) + sizeof(s64) + sizeof(s32) + sizeof(u8) )
#define SIZE_OF_SEND_DATA_BUFFER ( (tmipc::MAX_PACKET_DATA) - (SIZE_OF_SEND_BUFFER_HEADER) )

    //Init these.
    m_pBytesWritten     = pBytesWritten;
    m_Offset            = Offset;
    m_pReadFromBuffer   = (char*)pDataBuffer;
    m_AmountLeftToWrite = AmountToWrite;

    //Submit the packet
    m_TaskType = tmipc::TaskType_HioFileWrite;
    tmipc::Packet* p = AllocSendPacket( true );
    PopulatePacket( p );
    m_pServicesManager->SubmitTask( this, p );

    //Do we need to write more?
    if( m_AmountLeftToWrite > 0 )
    {
        SetNeedPackets();
    }
}

//==============================================================================

void HostIOWriteTask::PopulatePacket( tmipc::Packet* pPacket )
{
    s32 WriteThisPass = (s32)(m_AmountLeftToWrite > SIZE_OF_SEND_DATA_BUFFER ? SIZE_OF_SEND_DATA_BUFFER : m_AmountLeftToWrite);
    pPacket->WriteU64( m_Handle );
    pPacket->WriteS64( m_Offset );
    pPacket->WriteS32( WriteThisPass );

    //Last bite?
    m_AmountLeftToWrite -= WriteThisPass;
    u8 LastPacket = ( m_AmountLeftToWrite <= 0 );

    pPacket->WriteU8( LastPacket );

    //Pack out the rest of the buffer.
    pPacket->WriteData( m_pReadFromBuffer, WriteThisPass );

    //Get ready for next time.
    m_pReadFromBuffer     += WriteThisPass;
    m_Offset              += WriteThisPass;
}

//==============================================================================

void HostIOWriteTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOTask::OnInitiate", "HostIOTask::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOWriteTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Read how much we wrote
    s32 AmountWritten = 0;
    pPacket->ReadS32( AmountWritten );

    //Read the result
    pPacket->ReadS32 ( m_Result );

    //Update
    m_OperationValue = AmountWritten;
    *m_pBytesWritten = AmountWritten;

    DEJA_TRACE( "HostIOWriteTask::OnRecv", "Received result: %d", m_Result );

    // Done.
    Complete();
}

//==============================================================================

void HostIOWriteTask::OnSendPacket( tmipc::Packet* pPacket )
{
    PopulatePacket( pPacket );

    if( m_AmountLeftToWrite <= 0 )
    {
        // Done, so no more packets needed.
        SetNeedPackets( false );
    }
}

//==============================================================================

HostIOExistsTask::HostIOExistsTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
    DEJA_TRACE( "HostIOExistsTask::HostIOExistsTask", "HostIOExistsTask::HostIOExistsTask, Handle = %lld", Handle );
}

//==============================================================================

void HostIOExistsTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOExistsTask::OnInitiate", "HostIOExistsTask::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOExistsTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "HostIOExistsTask", "OnRecvPacket" );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );
    DEJA_TRACE( "HostIOTask::OnRecv", "Received tasktype: %d", pPacket->GetTaskType() );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOExistsTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOExistsTask::Exists( const char* pFullPath )
{
    DEJA_TRACE( "HostIOExistsTask::Exists", "Exists %s", pFullPath );

    m_TaskType = tmipc::TaskType_HioFileExists;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

bool HostIOExistsTask::GetExistsResult()
{
    return (m_OperationValue != 0);
}

//==============================================================================

HostIOOpenTask::HostIOOpenTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
    DEJA_TRACE( "HostIOOpenTask::HostIOOpenTask", "HostIOOpenTask::HostIOOpenTask, Handle = %lld", Handle );
}

//==============================================================================

void HostIOOpenTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOOpenTask::OnInitiate", "HostIOOpenTask::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOOpenTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOOpenTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOOpenTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOOpenTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOOpenTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOOpenTask::Open( uint32_t OpenFlags, const char* pFullPath )
{
    TMA_TRACE( "tma::hio::HostIOOpenTask", "Open(%d,%s)", OpenFlags, pFullPath );
    m_TaskType = tmipc::TaskType_HioFileOpen;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    p->WriteU32( OpenFlags );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

TMA_HOSTIO_FILE_HANDLE HostIOOpenTask::GetHandle()
{
    return m_Handle;
}

//==============================================================================

HostIOCloseTask::HostIOCloseTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
    DEJA_TRACE( "HostIOCloseTask::HostIOCloseTask", "HostIOCloseTask::HostIOCloseTask, Handle = %lld", Handle );
}

//==============================================================================

void HostIOCloseTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOCloseTask::OnInitiate", "HostIOCloseTask::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOCloseTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOCloseTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOCloseTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOCloseTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOCloseTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOCloseTask::Close()
{
    // tmipc::Packet* p = PreparePacket( TMA_OP_FILE_CLOSE, AgentHostIOService::TaskTypeIO );
    m_TaskType = tmipc::TaskType_HioFileClose;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

TMA_HOSTIO_FILE_HANDLE HostIOCloseTask::GetHandle()
{
    return m_Handle;
}

//==============================================================================

HostIOGetSizeTask::HostIOGetSizeTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOGetSizeTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOGetSizeTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOGetSizeTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOGetSizeTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOGetSizeTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOGetSizeTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOGetSizeTask::GetSize()
{
    m_TaskType = tmipc::TaskType_HioFileGetSize;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

TMA_HOSTIO_FILE_HANDLE HostIOGetSizeTask::GetHandle()
{
    return m_Handle;
}

//==============================================================================

int64_t HostIOGetSizeTask::GetSizeResult()
{
    return m_OperationValue;
}

//==============================================================================

HostIOCreateTask::HostIOCreateTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOCreateTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOCreateTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOCreateTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOCreateTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOCreateTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOCreateTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOCreateTask::Create( const char* pFullPath, s64 Size )
{
    m_TaskType = tmipc::TaskType_HioFileCreate;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    p->WriteS64( Size );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIODeleteTask::HostIODeleteTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIODeleteTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIODeleteTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIODeleteTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIODeleteTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIODeleteTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIODeleteTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIODeleteTask::Delete( const char* pFullPath )
{
    m_TaskType = tmipc::TaskType_HioFileDelete;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIORenameTask::HostIORenameTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIORenameTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIORenameTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIORenameTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIORenameTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIORenameTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIORenameTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIORenameTask::Rename( const char* pFrom, const char* pTo )
{
    m_TaskType = tmipc::TaskType_HioFileRename;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFrom );
    p->WriteString( pTo );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIOGetIOTypeTask::HostIOGetIOTypeTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOGetIOTypeTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOGetIOTypeTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOGetIOTypeTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOGetIOTypeTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOGetIOTypeTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOGetIOTypeTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOGetIOTypeTask::GetIOType( const char* pFullPath )
{
    m_TaskType = tmipc::TaskType_HioGetIoType;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

s32 HostIOGetIOTypeTask::GetIOTypeResult()
{
    //Our IO types
    if( m_OperationValue == IOType_Directory )
    {
        return nn::fs::DirectoryEntryType_Directory;
    }
    else if( m_OperationValue == IOType_File )
    {
        return nn::fs::DirectoryEntryType_File;
    }

    //Not sure what this is, so just return what we have.
    return (s32)m_OperationValue;
}

//==============================================================================

HostIOFlushTask::HostIOFlushTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOFlushTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOFlushTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOFlushTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOFlushTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOFlushTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOFlushTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOFlushTask::Flush()
{
    m_TaskType = tmipc::TaskType_HioFileFlush;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIOSetSizeTask::HostIOSetSizeTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOSetSizeTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOSetSizeTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOSetSizeTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOSetSizeTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOSetSizeTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOSetSizeTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOSetSizeTask::SetSize( s64 Size )
{
    m_TaskType = tmipc::TaskType_HioFileSetSize;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteS64( Size );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIOGetSizeFromPathTask::HostIOGetSizeFromPathTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOGetSizeFromPathTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOGetSizeFromPathTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOGetSizeFromPathTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::HostIOGetSizeFromPathTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOGetSizeFromPathTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOGetSizeFromPathTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOGetSizeFromPathTask::GetSize( const char* pFullPath )
{
    m_TaskType = tmipc::TaskType_HioFileGetSize;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

int64_t HostIOGetSizeFromPathTask::GetSizeResult()
{
    return m_OperationValue;
}

//==============================================================================

HostIOSetPriorityTask::HostIOSetPriorityTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOSetPriorityTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOSetPriorityTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::FileSetPriorityTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::FileSetPriorityTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::FileSetPriorityTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOSetPriorityTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOSetPriorityTask::SetPriority( s32 Priority )
{
    m_TaskType = tmipc::TaskType_HioFileSetPriority;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteS32( Priority );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostIOGetPriorityTask::HostIOGetPriorityTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
}

//==============================================================================

void HostIOGetPriorityTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOGetPriorityTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::FileGetPriorityTask", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    TMA_TRACE( "tma::hio::FileGetPriorityTask", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::FileGetPriorityTask", "*** m_OperationValue == %lld", m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOGetPriorityTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOGetPriorityTask::GetPriority()
{
    m_TaskType = tmipc::TaskType_HioFileGetPriority;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

s32 HostIOGetPriorityTask::GetPriorityResult()
{
    return (s32) m_OperationValue;
}

//==============================================================================

HostIOGetFileTimeStampTask::HostIOGetFileTimeStampTask( TMA_HOSTIO_FILE_HANDLE Handle )
:   HostIOTask( Handle )
{
    m_CreateTime    = 0;
    m_AccessTime    = 0;
    m_ModifyTime    = 0;
}

//==============================================================================

void HostIOGetFileTimeStampTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "HostIOGetFileTimeStamp::OnInitiate", "HostIOGetFileTimeStamp::OnInitiate" );

    //=====================================================================
    // FOR NOW - Silently ignore.  Eventually, we might want to
    // handle this case for file events.
    //=====================================================================

    Complete();
}

//==============================================================================

void HostIOGetFileTimeStampTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    TMA_TRACE( "tma::hio::HostIOGetFileTimeStamp", "OnRecvPacket()" );

    // Read our handle.
    pPacket->ReadU64( m_Handle );

    // Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadU64( m_CreateTime );
    pPacket->ReadU64( m_AccessTime );
    pPacket->ReadU64( m_ModifyTime );

    TMA_TRACE( "tma::hio::HostIOGetFileTimeStamp", "*** m_Result == %d", m_Result );
    TMA_TRACE( "tma::hio::HostIOGetFileTimeStamp", "*** m_CreateTime == %d", m_CreateTime );
    TMA_TRACE( "tma::hio::HostIOGetFileTimeStamp", "*** m_AccessTime == %d", m_AccessTime );
    TMA_TRACE( "tma::hio::HostIOGetFileTimeStamp", "*** m_ModifyTime == %d", m_ModifyTime );

    // Task complete.
    Complete();
}

//==============================================================================

void HostIOGetFileTimeStampTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;

    ASSERT( 0 );
}

//==============================================================================

void HostIOGetFileTimeStampTask::GetFileTimeStamp( const char* pFullPath )
{
    m_TaskType = tmipc::TaskType_HioGetFileTimeStamp;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

void HostIOGetFileTimeStampTask::GetTimeStampResults( u64* pCreateTile, u64* pAccessTime, u64* pModifyTime )
{
    if( pCreateTile != NULL )
    {
        *pCreateTile = m_CreateTime;
    }
    if( pAccessTime != NULL )
    {
        *pAccessTime = m_AccessTime;
    }
    if( pModifyTime != NULL )
    {
        *pModifyTime = m_ModifyTime;
    }
}

//==============================================================================

} } // namespace hio } namespace tma
//==============================================================================
