﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include "../tmagent.h"
#include "../tm_result.h"
#include "../tma/tma_Mutex.h"
#include <nn/osdbg.h>
#include "dbg_CommandArgs.h"
#include <nn/dmnt/dmnt_ThreadData.h>

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

class ThreadDefinitionCollection;

#define PRINTABLE_THREAD_ID(X) (u32)((X) & 0xFFFFFFFF)

class ThreadDefinition
{
friend class ThreadDefinitionCollection;

public:
    //Static call for getting the state of a given thread.
static s32      GetState            ( nn::svc::Handle   ProcessHandle,
                                      s32               ThreadId,
                                      u64*              pSP,
                                      u64*              pIP);

static s32      GetRegister         ( nn::svc::Handle   ProcessHandle,
                                      s32               ThreadId,
                                      s32               iRegister,
                                      u64*              pRegister);

static nn::Result GetThreadContext ( thread_context*    pThreadContext,
                                     nn::svc::Handle    ProcessHandle,
                                     nn::Bit64          ThreadId,
                                     nn::Bit32          ControlFlags );

static void       ClearThreadCache ( void );

static nn::Result UpdateAll        ( nn::svc::Handle ProcessHandle, ThreadDefinitionCollection* pThreadCollection, s32 NumberOfThreads );

        explicit ThreadDefinition   ( u64 ThreadId );
        ~ThreadDefinition           ( );
        tmapi::result  Update       ( nn::svc::Handle ProcessHandle, nn::dmnt::ThreadData& Data );

        void    SetInfo             ( nn::osdbg::ThreadInfo* pInfo );

        u64     GetIP               ( void );
        u64     GetSP               ( void );
        char    GetStatus           ( void );
        u64     GetThreadId         ( void );
        u8      GetCore             ( void );
        u32     GetIdealCore        ( void );
        u32     GetAffinityMask     ( void );
        u16     GetPriority         ( void );
        u16     GetPriorityBase     () { return nn::os::ThreadPriorityRangeSize; }
        u16     GetFlags            () { return 2;} //This is ignored in Target manager.  Setting it to 2 because that was what it set to in the code, historically.
        char*   GetName             ( void );
        u64     GetStackSize        ( void );
        u64     GetStackAddr        ( void );
        void    ToLog               ( void );

protected:
        //Linked list functionality
        ThreadDefinition* GetNext   ( void );
        void    SetNext             ( ThreadDefinition* pNext );
        void    SetName             ( char* pName );

        //Update functionality
nn::osdbg::ThreadInfo    GetThreadInfo    ( void );

private:
    u64     m_ThreadId;
    u64     m_IP;
    u64     m_SP;
    s8      m_Status;
    u8      m_Core;
    u8      m_IdealCore;
    u8      m_Priority;
    u32     m_AffinityMask;
    char    m_Name[nn::os::ThreadNameLengthMax];

    nn::osdbg::ThreadInfo   m_Info;
    ThreadDefinition*       m_pNext;
    per_thread_step_info    m_PTSI;

static std::map<nn::Bit64, nn::svc::ThreadContext> s_ThreadContextCache;
};

//==============================================================================

class ThreadDefinitionCollection
{

public:
            ThreadDefinitionCollection  ();
            ~ThreadDefinitionCollection ();

            //List management.
            void Add                ( u64 ThreadId,
                                      nn::osdbg::ThreadInfo* pInfo );

            void Remove             ( u64 ThreadId );
            void Clear              ( void );

            //Accessors
            void Update             ( int32_t NumberOfActiveThreads,
                                      uint64_t* pActiveThreadIds );

            //Accessors
            s32  Count                        ( void );
            ThreadDefinition* Get             ( s32 AtIndex );
            ThreadDefinition* Find            ( u64 ThreadId );
            ThreadDefinition* GetThreadList   ( void );
            u64  GetDefaultId                 ( void );
            void ToLog                        ( void );

            bool GetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI );
            bool SetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI );
private:
            tma::Mutex              m_Mutex;
            ThreadDefinition*       m_ThreadList;

            void Remove             ( ThreadDefinition* pRemove );
            void Add                ( ThreadDefinition* pAdd );
            bool Exists             ( u64 ThreadId );
};


//==============================================================================
// This is what we pass to TargetManager.
struct ThreadData
{
    nn::dmnt::ThreadData m_Data;
    NN_IMPLICIT ThreadData( ThreadDefinition* pThreadDef )
    {
        m_Data.m_ThreadId      = pThreadDef->GetThreadId();
        m_Data.m_IP            = pThreadDef->GetIP();
        m_Data.m_SP            = pThreadDef->GetSP();
        m_Data.m_Status        = pThreadDef->GetStatus();
        m_Data.m_Core          = pThreadDef->GetCore();
        m_Data.m_IdealCore     = pThreadDef->GetIdealCore();
        m_Data.m_AffinityMask  = pThreadDef->GetAffinityMask( );
        m_Data.m_Priority      = pThreadDef->GetPriority();
        m_Data.m_PriorityBase  = pThreadDef->GetPriorityBase();
        m_Data.m_Flags         = pThreadDef->GetFlags();

        memcpy(m_Data.m_Name, pThreadDef->GetName(), sizeof(m_Data.m_Name) - 1 );
    }
};

//==============================================================================
}} // namespace
//==============================================================================
