﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\stdafx.h"
#include "../dbghlp/dbghlp.h"

#define PRINT_STEPPING TMA_MACRO_VALUE(0)

#if PRINT_STEPPING
#define STEP_PRINTF( ... ) NN_SDK_LOG( __VA_ARGS__ )
#else
#define STEP_PRINTF( ... )
#endif

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

s32 SignExtendValue( u32 Value, u32 nBits )
{
    return( (s32)( Value << ( 32 - nBits ) ) >> ( 32 - nBits ) );
}

//==============================================================================

StepMgr::StepMgr()
:   m_pParent       ( NULL )
,   m_DataAddr      ( 0 )
,   m_TempCount     ( 0 )
,   m_MaxTempCount  ( 0 )
{
    break_point BP( 0 );
    m_SavedBP = BP;
}

//==============================================================================

void StepMgr::Init( Process* pProcess )
{
    m_pParent       = pProcess;
    m_SSMMutex.Create();
}

//==============================================================================

void StepMgr::Kill()
{
    m_pParent = NULL;
    m_SSMMutex.Destroy();
}

//==============================================================================

void StepMgr::ResumeExecution()
{
    nn::Result  result = m_pParent->ContinueDebugProcess( m_ThreadId ); // Continue as though nothing happened

    if( !result.IsSuccess() )
    {
        STEP_PRINTF( "ContinueDebugProcess( %lld, true ) failed, calling PurgeEventsAndContinue()\n", m_ThreadId );
        m_pParent->PurgeEventsAndContinue();
    }
    m_pParent->SetProcessState( PS_RUNNING ); // Mark us as running again
}

//==============================================================================

void StepMgr::GetLocalBP( break_point& LocalBP, ssm_exception* pSSME )
{
    break_point BP( (m_DataAddr != 0) ? m_DataAddr : m_IP, BP_NONE, SS_NONE, m_ThreadId );
    m_pParent->GetBreakpointManager()->GetBreakpoint( BP );

    if( BP.m_State != BP_NONE )
    {
        if( BP.m_State == BP_NORMAL ) // Unmanaged breakpoint
        {
            BP.m_Instruction = m_PTSI.m_Instr; // Use instruction passed by TargetManager
        }
        if( BP.m_State & BP_DATA_BOTH )
        {
            m_DataAddr  = BP.m_Address; // Update the local data address

            if( pSSME)
            {
                pSSME->exceptionAddress = BP.m_Address; // Update the exception information
            }
        }
    }
    else if( m_DataAddr != 0 ) // Unidentified hardware/data breakpoint
    {
        STEP_PRINTF( "StepMgr::GetLocalBP: Misaligned data breakpoint at %llx\n", m_DataAddr );
        BP.m_State      = BP_DATA_BOTH; // Default data breakpoint
        BP.m_Address    = m_DataAddr;
    }
    LocalBP = BP;
}

//==============================================================================

void StepMgr::SaveLocalBP( bool bSingleStep )
{
    break_point BP( 0 );
    GetLocalBP( BP );

    if( BP.m_State != BP_NONE )
    {
        m_SavedBP = BP;
        m_pParent->GetBreakpointManager()->DisableBreakpoint( BP ); // Disable the breakpoint

        SingleStep();
    }
    else if( bSingleStep )
    {
        SingleStep();
    }
}

//==============================================================================

bool StepMgr::OutsideSteppingBounds()
{
    if( (m_IP >= m_AddrLo) && (m_IP < m_AddrHi) ) // Inside current bounds
    {
        return( false );
    }

    for( s32 i = 0; (i < MAX_STEP_RANGES) && (m_PTSI.m_InstrRange[i].m_AddrLo != 0); i++ ) // Find current range
    {
        if( (m_IP >= m_PTSI.m_InstrRange[i].m_AddrLo )
            && ( m_IP < m_PTSI.m_InstrRange[i].m_AddrHi ) )
        {
            m_AddrLo = m_PTSI.m_InstrRange[i].m_AddrLo;
            m_AddrHi = m_PTSI.m_InstrRange[i].m_AddrHi;

            STEP_PRINTF( "StepMgr::OutsideSteppingBounds: IP:%llx, New Range:%d, %llx - %llx\n", m_IP, i, m_AddrLo, m_AddrHi );
            return( false ); // Inside new instruction range
        }
    }

    STEP_PRINTF( "StepMgr::OutsideSteppingBounds: IP:%llx, %llx - %llx\n", m_IP, m_AddrLo, m_AddrHi );
    return( true ); // Outside of all instruction ranges
}

//==============================================================================

step_state StepMgr::SetSteppingInfo( StepDebugCommandArgs* pArgs )
{
    tmapi::result Result = (tmapi::result)ThreadDefinition::GetState( m_Handle, pArgs->m_ThreadId, &m_SP, &m_IP );
    ASSERT( Result == tmapi::RESULT_OK );
    (void)Result;

    per_thread_step_info PTSI;

    PTSI.m_Instr    = pArgs->m_Instruction;
    PTSI.m_StepKind = pArgs->m_StepKind;

    memcpy( PTSI.m_InstrRange, pArgs->m_InstrRange, sizeof( PTSI.m_InstrRange ) );

    m_AddrLo = PTSI.m_InstrRange[0].m_AddrLo;
    m_AddrHi = PTSI.m_InstrRange[0].m_AddrHi;
    s32 i    = 0; // Instruction range selected

    step_state State = SS_OVER;

    if( pArgs->m_StepKind == STEP_CONT ) // Special handling of Continue
    {
        State    = SS_CONT;
    }
    else if( pArgs->m_StepKind == STEP_OUT )
    {
        State    = SS_OUT;
    }
    else
    {
        for( i = 0; i < MAX_STEP_RANGES; i++ ) // Find starting range
        {
            if( (m_IP >= PTSI.m_InstrRange[i].m_AddrLo )
                && ( m_IP < PTSI.m_InstrRange[i].m_AddrHi ) )
            {
                m_AddrLo = PTSI.m_InstrRange[i].m_AddrLo;
                m_AddrHi = PTSI.m_InstrRange[i].m_AddrHi;

                m_AddrHi = ( m_AddrHi + 3 ) & ~3; // Arm32 instructions are 4 bytes
                break;
            }
        }

        if( pArgs->m_StepKind == STEP_INTO )
        {
            State = SS_INTO;
        }
    }
    m_pParent->SetPerThreadStepInfo( pArgs->m_ThreadId, PTSI );

    STEP_PRINTF( "StepMgr::SetSteppingInfo: IP:%llx, Range:%d, %llx - %llx, Instr:%08x, %s\n", m_IP, i, m_AddrLo, m_AddrHi, PTSI.m_Instr, __TIMESTAMP__ );
    return( State );
}

//==============================================================================

void StepMgr::SteppingStateMachine( u64 ThreadId, step_state InitialState, ssm_exception* pSSME )
{
    // TODO We should not need to set the Debug Handle everytime.
    // This will require some work determining when to Init/Kill this class
    tma::ScopedLock Lock( m_SSMMutex ); // Make this call thread safe

    m_pParent->SetProcessState( PS_BKPT ); // Processing a breakpoint

    nn::Result  result;
    bool        bIgnore = false;

    m_ThreadId = ThreadId;
    m_Handle   = m_pParent->GetHandle();
    m_BPState  = BP_NONE;   // Assume not at a breakpoint
    break_point BPhc( 0 );  // BP from exception, if any

    ThreadDefinition::GetState( m_Handle, m_ThreadId, &m_SP, &m_IP );
    m_pParent->GetPerThreadStepInfo( m_ThreadId, m_PTSI );

    STEP_PRINTF( "==> SSM( %lld, %d ): StepState:%d, m_IP:%llx, m_SP:%llx, %d-bit\n", m_ThreadId, InitialState, m_StepState, m_IP, m_SP, m_pParent->Is64Bit() ? 64 : 32 );

    if( InitialState == SS_NONE ) // Called by the exception handler at a software breakpoint
    {
        m_DataAddr = 0;

        if( (pSSME != 0) && ( pSSME->exceptionCode == nn::svc::DebugException_BreakPoint ) )
        {
            m_DataAddr = pSSME->exceptionAddress;
        }
        break_point BP( 0 );
        GetLocalBP( BP );
        BPhc = BP;                          // Save for hit count testing

        if( BP.m_State == BP_NONE )         // The breakpoint no longer exists (cleared by another thread)
        {
            m_ThreadId  = 0;
            ResumeExecution();

            STEP_PRINTF( "<== SSM( %lld, %d ): Ignoring breakpoint already deleted by another thread\n\n", ThreadId, InitialState );
            return;
        }

        if( BP.m_State == BP_TEMP )
        {
            if( BP.m_Data != m_ThreadId )   // Temp breakpoint was not meant for this thread
            {
                if( m_TempCount < 1024 )    // Threshold of max Temp breakpoints not yet reached
                {
                    m_TempCount++;
                    bIgnore     = true;
                    m_StepState = SS_CONT;  // Avoid confusion when hitting Step breakpoint
                    STEP_PRINTF( "StepMgr::SSM - Stepping over a breakpoint on the wrong thread\n" );
                }
                else
                {
                    m_MaxTempCount = m_TempCount;
                    m_TempCount    = 0;   // Reset the count of skipped Temp breakpoints
                    NN_SDK_LOG( "StepMgr::SSM - Max number of skipped Temp breakpoints exceeded: %d\n", m_MaxTempCount );
                }
            }
            else
            {
                if( m_MaxTempCount < m_TempCount )
                {
                    m_MaxTempCount = m_TempCount;
                    STEP_PRINTF( "StepMgr::SSM - New MaxTempCount: %d\n", m_MaxTempCount );
                }
                m_TempCount     = 0;       // Reset the count of skipped Temp breakpoints
            }
        }

        else if( ( BP.m_State & BP_DATA_CHANGED ) == BP_DATA_CHANGED ) // Intel-style data breakpoint
        {
            ASSERT( pSSME != 0 );
            bIgnore = true;

            m_PrevSSME  = *pSSME; // Save exception information for Intel-style data breakpoints

            s32 Size    = 1 << (BP.m_Instruction & 0x1F);

            if( Size > MAX_DATA_SIZE ) Size = MAX_DATA_SIZE;

            m_pParent->ReadMemory( BP.m_Address, (void*)m_OldData, Size ); // Save the data for comparison
            STEP_PRINTF( "StepMgr::SSM - Saved data to implement Intel-style data breakpoint\n" );
        }

        if( bIgnore )
        {
            m_pParent->GetBreakpointManager()->ClearBreakpoints( BP_STEP ); // Guarantee all Step breakpoints removed

            SaveLocalBP( true );
            ResumeExecution();

            STEP_PRINTF( "<== SSM( %lld, %d ): StepState:%d, m_ThreadId:%lld\n\n", ThreadId, InitialState, m_StepState, m_ThreadId );
            return;
        }

        if( BP.m_State != BP_NONE ) // We hit a breakpoint, so try to get our former StepState
        {
            m_BPState   = BP.m_State;
            m_StepState = BP.m_StepState; // Get StepState when breakpoint was set
        }
    }
    else
    {
        m_StepState = InitialState;
    }

    if( m_SavedBP.m_Address > 1 ) // Single-stepped over a breakpoint
    {
        if( ( m_SavedBP.m_State & BP_DATA_CHANGED ) == BP_DATA_CHANGED ) // Checking phase of Intel-style data breakpoint
        {
            s32 Size    = 1 << (m_SavedBP.m_Instruction & 0x1F);

            if( Size > MAX_DATA_SIZE ) Size = MAX_DATA_SIZE;

            m_pParent->ReadMemory( m_SavedBP.m_Address, (void*)m_NewData, Size ); // Save the data for comparison

            for( s32 i = 0; i < Size; i++ )
            {
                if( m_OldData[i] != m_NewData[i] ) // Compare the old and new data
                {
                    *pSSME                  = m_PrevSSME;
                    pSSME->exceptionAddress = m_SavedBP.m_Address;
                    m_StepState             = SS_NONE;      // We are done
                    BPhc                    = m_SavedBP;    // Use the saved breakpoint for HitCount comparisons
                    STEP_PRINTF( "Stopping because change detected at %p\n", m_SavedBP.m_Address );
                    break;
                }
            }
        }
        m_pParent->GetBreakpointManager()->EnableBreakpoint( m_SavedBP ); // Restore the breakpoint that we single-stepped over
    }
    memset( &m_SavedBP, 0, sizeof(m_SavedBP) );

    m_pParent->GetBreakpointManager()->ClearBreakpoints( BP_STEP ); // Guarantee all Step breakpoints removed

    // Now that we have actually stopped, manage the HitCount if this is a PassCount breakpoint

    u32 PassCountType = BPhc.m_State & BP_COUNT_MOD;

    if( PassCountType ) // Currently stopped at a PassCount breakpoint
    {
        break_point& BP = BPhc;

        BP.m_HitCount++;
        BP.m_State = (bp_state)( BP.m_State | BP_MODIFIED );

        STEP_PRINTF( "\nUpdating BP for Addr:%llX, State:%d, HitCount:%d, PassCount:%d\n",
                    BP.m_Address, BP.m_State, BP.m_HitCount, BP.m_PassCount );

        m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Update the hit count

        bool bStop  = false;

        if( PassCountType == BP_COUNT_EQ )
        {
            bStop   = BP.m_HitCount == BP.m_PassCount;
        }
        else if( PassCountType == BP_COUNT_GE )
        {
            bStop   = BP.m_HitCount >= BP.m_PassCount;
        }
        else if( PassCountType == BP_COUNT_MOD )
        {
            bStop   = (BP.m_HitCount % BP.m_PassCount) == 0;
        }

        if( m_StepState != SS_BREAK )       // We are not required to stop here.
        {
            if( bStop                       // Check to see if we should stop.
                || ( m_PTSI.m_StepKind == STEP_OVER )
                || ( m_PTSI.m_StepKind == STEP_INTO )
                || ( m_PTSI.m_StepKind == STEP_OUT  ) )
            {
                STEP_PRINTF( "Stopping at BP Addr:%llX, State:%d, HitCount:%d, PassCount:%d\n", BP.m_Address, BP.m_State, BP.m_HitCount, BP.m_PassCount );
                m_StepState = SS_BREAK;     // Force it to Stop stepping.
            }
            else                            // Contine running.
            {
                m_BPState   = BP_NONE;
                m_StepState = SS_CONT;
                STEP_PRINTF( "Skipping BP at Addr:%llX, State:%d, HitCount:%d, PassCount:%d\n", BP.m_Address, BP.m_State, BP.m_HitCount, BP.m_PassCount );

                SaveLocalBP( true );
                ResumeExecution();

                STEP_PRINTF( "<== SSM( %lld, %d ): StepState:%d, m_ThreadId:%lld\n\n", ThreadId, InitialState, m_StepState, m_ThreadId );
                return;
            }
        }
    }

    STEP_PRINTF( "SSM Switch: StepState:%d, BPState:%x, DataAddr:%llx\n", m_StepState, m_BPState, m_DataAddr );

    switch( m_StepState )
    {
    case SS_NONE:       // Stopped at breakpoint that should be ignored
        if( m_BPState == BP_NONE )
        {
            m_StepState = SS_CONT;
        }
        break;

    case SS_BREAK:      // Forces stop after BreakAll or exception
        m_StepState = SS_NONE;
        break;

    case SS_CONT:       // Continue executing
    case SS_OUT:        // Continue executing until return address
        if( ( InitialState == SS_NONE ) && ( m_BPState > BP_STEP ) )
        {
            m_StepState = SS_NONE;
            break;
        }

        //if( InitialState == SS_CONT )
        //{
        //    m_pParent->GetBreakpointManager()->ClearBreakpoints( BP_TEMP, 0 ); // Clear all Temp breakponts on all threads
        //}
        //else
        if( InitialState == SS_OUT )
        {
            for( s32 i = 0; (i < MAX_STEP_RANGES) && (m_PTSI.m_InstrRange[i].m_AddrLo != 0); i++ ) // Valid return address
            {
                break_point BP( m_PTSI.m_InstrRange[i].m_AddrLo, BP_TEMP, m_StepState, m_ThreadId );
                STEP_PRINTF( "oTEMP breakpoint at %llx for thread %lld\n", BP.m_Address, BP.m_Data );
                m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Set TEMP BP at return address
            }
        }
        SaveLocalBP();  // Decrements process state and continues
        break;

    case SS_INTO: // Stepping until outside of statement bounds
    case SS_OVER: // Stepping over calls until outside of statement bounds
    {
        if( ( InitialState == SS_NONE ) && OutsideSteppingBounds() )
        {
            if( (m_PTSI.m_InstrRange[0].m_AddrLo + 1) != m_PTSI.m_InstrRange[0].m_AddrHi ) // Not instruction step
            {
                u32 Instruction = 0;
                m_pParent->ReadMemory( m_IP, &Instruction, sizeof(Instruction) );

                if( Instruction == ( m_pParent->Is64Bit() ? 0x14000001 : 0xEAFFFFFF ) ) // NOP
                {
                    STEP_PRINTF( "StepMgr::SSM - Stepping over NOP at 0x%016llx\n", m_IP );
                    SaveLocalBP( true );
                    ResumeExecution();      // Single step on this thread

                    STEP_PRINTF( "<== SSM( %lld, %d ): StepState:%d, m_ThreadId:%lld\n\n", ThreadId, InitialState, m_StepState, m_ThreadId );
                    return;
                }
            }
            m_StepState = SS_NONE;
            break;
        }

#if 0   // Enable this to leave Breadcrumb on every instruction during StepInto
        if( ProcessState == SS_INTO )
        {
            SaveLocalBP( true ); // Decrements process state and steps
            break;
        }
#endif

        m_pParent->GetBreakpointManager()->ClearBreakpoints( BP_TEMP, m_ThreadId ); // Guarantee all Temp breakpoints removed for this thread

        u64 Target  = 0;
        u64 CurNext = 0;

        // Scan forward until outside of remainder of statement

        bool    bSingleStep = ( m_StepState == SS_INTO ) && ( ( m_AddrLo & 1 ) == 0 ); // StepInto code not in relocation layer
        u64     CurAddr     = m_IP;

        while( CurAddr < m_AddrHi )
        {
            CurNext = CurAddr;
            Target  = GetBranchTarget( CurNext, bSingleStep );
            STEP_PRINTF( "GetBranchTarget( %llx ): ThreadId:%lld, Target:%llx, Next:%llx\n", CurAddr, m_ThreadId, Target, CurNext );

            if( Target != 0 )   // Change to IP found
            {
                if( bSingleStep                 // StepInto from this address range
                    || (Target < m_IP)          // OR Prior to portion of statement already scanned
                    || (Target == CurNext) )    // OR Temp breakpoint needed to guarantee values of registers
                {
                    break;      // Stop scanning
                }
                if( ( Target == m_AddrHi ) && ( CurNext != 0 ) )
                {
                    break_point BP( m_AddrHi, BP_TEMP, m_StepState, m_ThreadId );
                    STEP_PRINTF( "eTEMP breakpoint at %llx for thread %lld\n", BP.m_Address, BP.m_Data );
                    m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Set TEMP BP at end of statement
                }
                else if( m_AddrHi <= Target )   // Temp breakpoint for branch outside of statement
                {
                    break;      // Stop scanning
                }
                else if( CurAddr < Target )     // Branch within Remainder of statement
                {
                    break_point BP( Target, BP_TEMP, m_StepState, m_ThreadId );
                    STEP_PRINTF( "rTEMP breakpoint at %llx for thread %lld\n", BP.m_Address, BP.m_Data );
                    m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Set TEMP BP at Target

                    if( CurNext )               // Conditional branch
                    {
                        CurAddr = CurNext;      // Also set TEMP BP on CurNext
                    }
                    break;      // Stop scanning
                }
            }
            CurAddr += 4;
        }

        if( ( CurNext == m_AddrHi ) && ( m_AddrHi != 0 ) )
        {
            break_point BP( m_AddrHi, BP_TEMP, m_StepState, m_ThreadId );
            STEP_PRINTF( "hTEMP breakpoint at %llx for thread %lld\n", BP.m_Address, BP.m_Data );
            m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Set TEMP BP at end of statement
        }

        if( CurAddr == m_IP )
        {
            SaveLocalBP( true ); // Step over this instruction instead of setting a breakpoint
        }
        else
        {
            break_point BP( CurAddr, BP_TEMP, m_StepState, m_ThreadId );
            STEP_PRINTF( "cTEMP breakpoint at %llx for thread %lld\n", BP.m_Address, BP.m_Data );
            m_pParent->GetBreakpointManager()->SetBreakpoint( BP ); // Set TEMP BP at current address

            SaveLocalBP(); // Continue until some type of breakpoint
        }
        break;
    }

    default:
        ASSERT( 0 );
        m_StepState = SS_NONE;
        break;
    }

    //----------------------------------
    // Common cleanup logic
    //----------------------------------

    if( m_StepState == SS_NONE ) // Returning to the default state
    {
        m_pParent->GetBreakpointManager()->ClearBreakpoints( BP_TEMP, m_ThreadId );
    }
    else // Continue operation of Stepping State Machine
    {
        if( m_SavedBP.m_Address == 0 ) // No STEP breakpoint in use
        {
            m_ThreadId = 0; // Continue on all threads
        }
        ResumeExecution();
    }
    STEP_PRINTF( "<== SSM( %lld, %d ): StepState:%d, m_ThreadId:%lld\n\n", ThreadId, InitialState, m_StepState, m_ThreadId );

} // NOLINT(readability/fn_size)

//==============================================================================
// Compute the target address and next address for the specified instruction.
//      CurNext     - Address of the current/next instruction           (in/out)
//      bSingleStep - Extra processing for calls from SingleStep,       (in)
//                    such as targets of calls and "pop {...,pc}" instructions
// Returns: Target = 0 = Not a branch
//                  -1 = Max value to fail bounds check for StepOver
//                  -2 = Branch to Self (must be ignored)
//                  -3 = User coded breakpoint, requires special handling
//                 Else, actual target of branch for range checking or single step

u64 StepMgr::GetBranchTarget( u64& CurNext, bool bSingleStep )
{
    break_point BP( CurNext );
    u64 Target      = 0;
    m_bCallInst     = false;
    CurNext        += 4; // Address of next instruction

    m_pParent->GetBreakpointManager()->GetBreakpoint( BP );

    if( BP.m_State == BP_NORMAL )
    {
        if( BP.m_Address != m_IP )
        {
            // Special exit case for unmanaged breakpoint encountered after m_IP during stepping scan,
            // since instruction provided by TargetManager is only valid for the current stop location.
            return( m_IP - 1 );
        }
        BP.m_Instruction = m_PTSI.m_Instr; // Use instruction provided by TargetManager
    }
    STEP_PRINTF( "GetBranchTarget - BPAddr:%llx, BPState:%d, BPStepState:%d, Instr:%08x, Data:%lld\n", BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_Data );

    // ====================================
    // ======== ARM64 Instructions ========
    // ====================================

    if( m_pParent->Is64Bit() )
    {

////----------------------------------------------------------------------"x001 01xx xxxx xxxx xxxx xxxx xxxx xxxx" // Unconditional branch (immediate)
//    { 0, 0, "", "b",                "$Im26:00;",                            "0001 01 Im26" },
//    { 0, 0, "", "bl",               "$Im26:00;",                            "1001 01 Im26" },
        if( (BP.m_Instruction & 0x7C000000) == 0x14000000 )
        {
            if( BP.m_Instruction != 0x14000001 ) // Ignore NOP - Branch to Next
            {
                m_bCallInst = (BP.m_Instruction & 0x80000000) == 0x80000000;

                if( !m_bCallInst )
                {
                    CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
                }

                if( bSingleStep || !m_bCallInst ) // SingleStep or Not a call
                {
                    Target = SignExtendValue( ((BP.m_Instruction & 0x03FFFFFF) << 2), 28 ) + BP.m_Address;
                }
            }
        }

////----------------------------------------------------------------------"x011 010x xxxx xxxx xxxx xxxx xxxx xxxx" // Compare & branch (immediate)
//    { 0, 0, "", "cbz",              "$?sf?X?W?d;, $Im19:00;",               "sf 011 0100 Im19 Rd" },
//    { 0, 0, "", "cbnz",             "$?sf?X?W?d;, $Im19:00;",               "sf 011 0101 Im19 Rd" },
        else if( (BP.m_Instruction & 0x7E000000) == 0x34000000 )
        {
            Target = SignExtendValue( ((BP.m_Instruction & 0x00FFFFE0) >> 3), 21 ) + BP.m_Address;
        }

////----------------------------------------------------------------------"x011 011x xxxx xxxx xxxx xxxx xxxx xxxx" // Test & branch (immediate)
//    { 0, 0, "", "tbz",              "$?sf?X?W?d;, #$b31:M40;, $Im14:00;",   "sf 011 0110 M40 Im14 Rd" },
//    { 0, 0, "", "tbnz",             "$?sf?X?W?d;, #$b31:M40;, $Im14:00;",   "sf 011 0111 M40 Im14 Rd" },
        else if( (BP.m_Instruction & 0x7E000000) == 0x36000000 )
        {
            Target = SignExtendValue( ((BP.m_Instruction & 0x0007FFE0) >> 3), 16 ) + BP.m_Address;
        }

////----------------------------------------------------------------------"0101 010x xxxx xxxx xxxx xxxx xxxx xxxx" // Conditional branch (immediate)
//    { 0, 0, "", "b.$cond;",         "$Im19:00;",                            "0101 0100 Im19 0 cond" },
//    }
        else if( (BP.m_Instruction & 0xFF000010) == 0x54000000 )
        {
            if( (BP.m_Instruction & 0x0F) == 0x0E )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }
            Target = SignExtendValue( ((BP.m_Instruction & 0x00FFFFE0) >> 3), 21 ) + BP.m_Address;
        }

////----------------------------------------------------------------------"1101 011x xxxx xxxx xxxx xxxx xxxx xxxx" // Unconditional branch (register)
//    { 0, 0, "", "br",               "$Xn;",                                 "1101 0110 0001 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "blr",              "$Xn;",                                 "1101 0110 0011 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "ret",              "?Xn=11110??$Xn;?",                     "1101 0110 0101 1111 0000 00 Rn    00000" },
//    { 0, 0, "", "eret",             "",                                     "1101 0110 1001 1111 0000 00 11111 00000" },
//    { 0, 0, "", "drps",             "",                                     "1101 0110 1011 1111 0000 00 11111 00000" },
        else if( (BP.m_Instruction & 0xFF8FFC1F) == 0xD60F0000 )
        {
            m_bCallInst = (BP.m_Instruction & 0x00F00000) == 0x00300000;

            if( !m_bCallInst )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else if( bSingleStep || !m_bCallInst ) // SingleStep or Not a call
            {
                ThreadDefinition::GetRegister( m_Handle, m_ThreadId, (BP.m_Instruction & 0x03E0) >> 5, &Target );
            }
        }
    }

    // ====================================
    // ======== ARM32 Instructions ========
    // ====================================

    else
    {

//    { 0, 0, "", "blx",              "$Offset24:00+1000;",                   "1111 1010 Offset24" },                             // Branch to Thumb
//    { 0, 0, "", "blx",              "$Offset24:10+1000;",                   "1111 1011 Offset24" },                             // Branch to Thumb
//    { 0, 0, "", "b$Cond;",          "$Offset24:00+1000;",                   "Cond 1010 Offset24" },                             // Branch
//    { 0, 0, "", "bl$Cond;",         "$Offset24:00+1000;",                   "Cond 1011 Offset24" },                             // Branch with Link
        if( (BP.m_Instruction & 0x0E000000) == 0x0A000000 )          // B<c>/BL<c>/BLX imm
        {
            if( BP.m_Instruction != 0xEAFFFFFF ) // Ignore NOP - Branch to Next
            {
                m_bCallInst = ((BP.m_Instruction & 0xF0000000) == 0xF0000000) || ((BP.m_Instruction & 0x01000000) != 0);

                if( (!m_bCallInst) && ((BP.m_Instruction & 0xF0000000) == 0xE0000000) )
                {
                    CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
                }

                if( bSingleStep || !m_bCallInst ) // SingleStep or Not a call
                {
                    Target = SignExtendValue( (BP.m_Instruction & 0x00FFFFFF) << 2, 26 ) + 8 + BP.m_Address;

                    if( (BP.m_Instruction & 0xFF000000) == 0xFB000000 )
                    {
                        Target += 2; // Adjust for Thumb target
                    }
                }
            }
        }

//    { 0, 0, "", "bx$Cond;",         "$Rm;",                                 "Cond 0001 0010 1111 1111 1111 0001 Rm" },          // Branch and Exchange
//    { 0, 0, "", "blx$Cond;",        "$Rm;",                                 "Cond 0001 0010 1111 1111 1111 0011 Rm" },          // Branch with Link and Exchange
        else if( (BP.m_Instruction & 0x0FFFFFD0) == 0x012FFF10 )     // BX<c>/BLX<c> reg
        {
            m_bCallInst = ((BP.m_Instruction & 0x00000020) != 0);

            if( (!m_bCallInst) && ((BP.m_Instruction & 0xF0000000) == 0xE0000000) )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target = BP.m_Address;  // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else if( bSingleStep || !m_bCallInst ) // SingleStep or Not a call
            {
                if( (BP.m_Instruction & 0x0F) == 15 ) // Special handling for ARM32 PC
                {
                    Target  = BP.m_Address + 8;
                }
                else
                {
                    ThreadDefinition::GetRegister( m_Handle, m_ThreadId, BP.m_Instruction & 0x0F, &Target );
                }
            }
        }

//    { 0, 0, "", "mov$S;$Cond;",     "$Rd;, $Rm;, $ST; $Rs;",                "Cond 0001 101 S Rn Rd Rs 0 ST 1 Rm" },             // Rd = Op2
        else if( (BP.m_Instruction & 0x0FFFFFF0) == 0x01A0F000 )     // MOV PC, reg
        {
            if( (BP.m_Instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else if( (BP.m_Instruction & 0x0F) == 15 ) // Special handling for ARM32 PC
            {
                Target  = BP.m_Address + 8;
            }
            else
            {
                ThreadDefinition::GetRegister( m_Handle, m_ThreadId, BP.m_Instruction & 0x0F, &Target );
            }
        }

//    { 0, 0, "", "ldr$B;$Cond;",     "$Rd;, [$Rn;]$W;",                      "Cond 0101 U B W 1 Rn Rd 000000000000" },
//    { 0, 0, "", "ldr$B;$Cond;",     "$Rd;, [$Rn;, #$U;$Offset12;]$W;",      "Cond 0101 U B W 1 Rn Rd Offset12" },
        else if( (BP.m_Instruction & 0x0F10F000) == 0x0510F000 )    // LDR PC, [Rx, #Imm]{!}
        {
            if( (BP.m_Instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else
            {
                u64 Addr    = 0;
                s64 Offset  = BP.m_Instruction & 0x00000FFF;
                s32 RegN    = ( BP.m_Instruction >> 16 ) & 0x0F;
                u32 Value   = 0;

                if( RegN == 15 ) // Special handling for ARM32 PC
                {
                    Addr = BP.m_Address + 8;
                }
                else
                {
                    ThreadDefinition::GetRegister( m_Handle, m_ThreadId, RegN, &Addr );
                }
                STEP_PRINTF( "\nGetBranchTarget( %llx ): Instr:%08x, RegN:%d, ValueN:0x%llx, Offset:0x%llx\n",
                             Addr, BP.m_Instruction, RegN, Addr, Offset );

                if( (BP.m_Instruction & 0x01000000) != 0 ) // Offset or Pre-Indexed
                {
                    if( (BP.m_Instruction & 0x00800000) == 0 ) // Subtract offset
                    {
                        Offset = 0 - Offset;
                    }
                    Addr += Offset;
                }
                m_pParent->ReadMemory( Addr, (void*)&Value, sizeof(u32) ); // Read new PC from memory
                Target      = (u64)Value;
                STEP_PRINTF( "Addr:0x%llx, Value:0x%x\n\n", Addr, Value );
            }
        }

//    { 0, 0, "", "add$S;$Cond;",     "$Rd;, $Rn;, $Rm;$ShiftST;",            "Cond 0000 100 S Rn Rd ShiftST 0 Rm" },             // Rd = Op1  +  Op2
        else if( (BP.m_Instruction & 0x0FF0F060) == 0x0080F000 )    // ADD PC, Rn, Rm, LSL #Shift
        {
            if( (BP.m_Instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
           }
            else
            {
                u64 Addr    = 0;
                s64 Shift   = ( BP.m_Instruction >>  7 ) & 0x0000001F;
                s32 RegN    = ( BP.m_Instruction >> 16 ) & 0x0F;
                s32 RegM    = BP.m_Instruction & 0x0F;
                u64 ValueN  = 0;
                u64 ValueM  = 0;

                if( RegN == 15 ) // Special handling for ARM32 PC
                {
                    ValueN = BP.m_Address + 8;
                }
                else
                {
                    ThreadDefinition::GetRegister( m_Handle, m_ThreadId, RegN, &ValueN );
                }

                if( RegM == 15 ) // Special handling for ARM32 PC
                {
                    ValueM = BP.m_Address + 8;
                }
                else
                {
                    ThreadDefinition::GetRegister( m_Handle, m_ThreadId, RegM, &ValueM );
                }

                Target      = (u64)( (s32)ValueN + (s32)( ValueM << Shift ) );

                STEP_PRINTF( "\nGetBranchTarget( %llx ): Instr:%08x, RegN:%d, RegM:%d, ValueN:0x%llx, ValueM:0x%llx, Shift:%d, Target:0x%llx\n",
                             Addr, BP.m_Instruction, RegN, RegM, ValueN, ValueM, Shift, Target );

                if( Target & 0x03 )
                {
                    Target  = ( Target >> 2 ) << 2; // Branch to Thumb or misaligned ARM32
                }
            }
        }

//    { 0, 0, "", "pop$Cond;",        "{$Rd;}",                               "Cond 0100 1001 1101 Rd 0000 0000 0100" },
        else if( (BP.m_Instruction & 0x0FFFFFFF) == 0x049DF004 )     // POP<c> {PC}
        {
            if( (BP.m_Instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else
            {
                u32  Value;
                m_pParent->ReadMemory( m_SP, (void*)&Value, sizeof(u32) ); // Read new PC from stack
                Target = (u64)Value;
            }
        }

//    { 0, 0, "", "pop$Cond;",        "{$RegList;}",                          "Cond 1000 1011 1101 RegList" },
        else if( (BP.m_Instruction & 0x0FFF8000) == 0x08BD8000 )     // POP<c> {...,PC}
        {
            if( (BP.m_Instruction & 0xF0000000) == 0xE0000000 )
            {
                CurNext = 0; // Unconditional branch needs no STEP breakpoint on next instruction
            }

            if( m_IP != BP.m_Address )
            {
                Target  = BP.m_Address; // Place STEP breakpoint here to guarantee correct values of registers
                CurNext = Target;       // Special case to force placement of breakpoint
            }
            else
            {
                u32  RegList = BP.m_Instruction & 0x0000FFFF;
                u64  Addr    = m_SP;

                for( s32 i = 0; i < 14; i++ ) // Advance address to point to PC by skipping all other regs
                {
                    if( RegList & 1 )
                    {
                        Addr += 4;
                    }
                    RegList >>= 1;
                }

                u32 Value;
                m_pParent->ReadMemory( Addr, (void*)&Value, sizeof(u32) ); // Read new PC from stack
                Target = (u64)Value;
            }
        }
    }
    return( Target );
} // NOLINT(readability/fn_size)

//==============================================================================
// Set breakpoints to simulate SingleStep

void StepMgr::SingleStep()
{
    u64 CurNext = m_IP;
    u64 Target  = GetBranchTarget( CurNext, true ); // Update Target and Next for single step
    STEP_PRINTF( "GetBranchTarget( %llx ): ThreadId:%lld, Target:%llx, Next:%llx, SingleStep\n", m_IP, m_ThreadId, Target, CurNext );

    if( ( Target != 0 ) && ( Target != m_IP ) )
    {
        step_state StepState = m_StepState;

        if( m_bCallInst && (StepState == SS_OVER) )
        {
            StepState = SS_CONT; // Do not stop inside subroutine
        }
        break_point BP( Target, BP_STEP, StepState );
        STEP_PRINTF( "tSTEP breakpoint at %llx in state %d\n", BP.m_Address, BP.m_StepState );
        m_pParent->GetBreakpointManager()->SetBreakpoint( BP );

        if( m_SavedBP.m_Address == 0 )
        {
            m_SavedBP.m_Address = 1; // Continue only on this one thread
        }
    }
    if( CurNext != 0 )
    {
        break_point BP( CurNext, BP_STEP, m_StepState ); // Use new BP object to avoid corruption by previous call
        STEP_PRINTF( "nSTEP breakpoint at %llx in state %d\n", BP.m_Address, BP.m_StepState );
        m_pParent->GetBreakpointManager()->SetBreakpoint( BP );

        if( m_SavedBP.m_Address == 0 )
        {
            m_SavedBP.m_Address = 1; // Continue only on this one thread
        }
    }
}

//==============================================================================
}} // namespace
//==============================================================================
