﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dbg_opcodes.h"
#include "dbg_SimTask.h"
#include "dbg_Command.h"
#include "dbg_CommandArgs.h"
#include "dbg_SimService.h"
#include "../tm_result.h"

// TODO: this is temporary, until a proper #define (or whatnot) is
//       created and [ideally] centrally located.  For now, 0 means
//       "No Process"
static const u64 NoProcessId(0);

enum
{
    DEFAULT_RETURN_BUFFER_SIZE = 2048
};

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

char SimDebugTask::m_ApplicationFileName[1024];

//==============================================================================

void GetApplicationFileName( char* FileName, s32 FileNameBufferLength )
{
    bool Success = true;

    // Get the HMODULE from "this" process.
    HMODULE hModule = nullptr;
    if( Success )
    {
        hModule = GetModuleHandle( nullptr );
        Success = hModule != nullptr;
    }

    // Retrieve the full path file name from the HMODULE.
    if( Success )
    {
        // Retrieve the file name.
        const u32 BufferSize( 1024 );
        char fileName[BufferSize];
        const DWORD StringLength = GetModuleFileNameA( hModule, fileName, BufferSize );
        Success = StringLength > 0;
        if( Success )
        {
            // Copy the file name to the returned parameter.
            strcpy_s( FileName, FileNameBufferLength, fileName );
        }
    }

    // Make sure the application name is not empty.
    if( FileName[0] == '\0' )
    {
        strcpy_s( FileName, FileNameBufferLength, "Unknown" );
    }
}


void SimDebugTask::Initialize()
{
    //TMA_TRACE("SimDebugTask", "Install");

    // Attempt to get the application name, so it can be returned in the
    // beacon response.
    m_ApplicationFileName[0] = '\0';
    GetApplicationFileName( m_ApplicationFileName, sizeof(m_ApplicationFileName) );
}

//==============================================================================

void SimDebugTask::Finalize()
{
    //TMA_TRACE("SimDebugTask", "Remove");
    m_ApplicationFileName[0] = '\0';
}

//==============================================================================

SimDebugTask::SimDebugTask()
{
}

//==============================================================================

SimDebugTask::~SimDebugTask()
{
}

//==============================================================================

void SimDebugTask::HandleStandardCommand( u32 CommandCode, u64 ProcessId, u64 ParameterVal )
{
    (void)ParameterVal;
    TMA_TRACE( "SimDebugTask", "HandleStandardCommand( %llx )", ParameterVal );

    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    char ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    // This is to make the code look like the original in dbg_ProcessMgr.cpp.
    char* pReturnBuffer = ReturnBuffer;
    memset( ReturnBuffer, 0, DEFAULT_RETURN_BUFFER_SIZE );

    TMAgent_message_type MessageType = TMAGENT_MESSAGE_TYPE_UNKNOWN;
    switch( CommandCode )
    {
        case TMA_OP_NOP:
            ResultCode = tmapi::RESULT_OK;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_WRITE_DUMP:
            ResultCode  = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_DISCONNECT:
        case TMA_OP_STOP_DEBUG:
        case TMA_OP_KILL_PROCESS:
            ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_NUM_PROCESSES:
            *reinterpret_cast<s32*>(pReturnBuffer) = 1;
            ResultCode  = tmapi::RESULT_OK;
            MessageType = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
            break;

        case TMA_OP_PROCESS_INFO:
        {
            ResultCode  = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
            MessageType = TMAGENT_MESSAGE_TYPE_PROCESS;
            break;
        }

        default:
        {
            //This process is dead.  Tell the guy pinging us.
            TMA_TRACE("ProcessMgr::HandleCommand", "HandleCommand:  Rejecting message to dead process" );
            ResultCode  = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;
        }
    }

    Packetize( MessageType, ProcessId, ResultCode, ReturnBuffer, 0 );
}

//==============================================================================

void SimDebugTask::HandleLoadElfCommand( tmipc::Packet* pPacket )
{
    //TMA_TRACE( "SimDebugTask", "HandleLoadElfCommand" );

    char FileName[1024];
    char Args[1024];

    s32 Result = pPacket->ReadString( FileName, sizeof(FileName), NULL );
    ASSERT( Result == tmapi::RESULT_OK );

    Result = pPacket->ReadString( Args, sizeof(Args), NULL );
    ASSERT( Result == tmapi::RESULT_OK );

    //TMA_TRACE( "SimDebugTask::HandleLoadElfCommand", "Created process %d", ProcessId  );
    u64 ProcessId = NoProcessId;
    Packetize( tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64, ProcessId, Result, (void*)&ProcessId, 0 );
}

//==============================================================================

void SimDebugTask::HandleReadMemCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    ReadMemCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(Args) );

    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( Args.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem = tma::s_Allocate( sizeof( u8 ) * Args.m_DataSize );
        pReturnBuffer = new (pMem) u8[Args.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)Args.m_DataSize );

    TMA_TRACE( "SimDebugTask", "HandleReadMemCommand( %llx )", Args.m_Address );

    tma::dbg::Command Command( TMA_OP_READ_MEM, ProcessId, 0, sizeof(ReadMemCommandArgs), &Args );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RAW;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, pReturnBuffer, Args.m_DataSize );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        s_Deallocate( pReturnBuffer, sizeof( u8 ) * Args.m_DataSize );
    }
}

//==============================================================================

void SimDebugTask::HandleReadRegDefsCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    ReadRegDefsCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(Args) );

    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( Args.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem = tma::s_Allocate( sizeof( u8 ) * Args.m_DataSize );
        pReturnBuffer = new (pMem) u8[(size_t)Args.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)Args.m_DataSize );

    TMA_TRACE( "SimDebugTask", "HandleReadRegDefsCommand( %d )", Args.m_DataSize );

    tma::dbg::Command Command( TMA_OP_READ_REG_DEFS, ProcessId, 0, sizeof(ReadRegDefsCommandArgs), &Args );
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RAW;

    s32 DataSize = Args.m_DataSize;

    if( ResultCode != tmapi::result::RESULT_OK )
    {
        DataSize = 0;

        if( ResultCode == tmapi::result::RESULT_BUFFER_TOO_SMALL )
        {
            DataSize = sizeof(u32); // Enough to return correct size
        }
    }
    Packetize( Type, ProcessId, ResultCode, pReturnBuffer, DataSize );

    TMA_TRACE( "SimDebugTask", "HandleReadRegDefsCommand( %d ), ResultCode = %d, DataSize = %d", Args.m_DataSize, ResultCode, DataSize );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        s_Deallocate( pReturnBuffer, sizeof( u8 ) * Args.m_DataSize );
    }
}

//==============================================================================

void SimDebugTask::HandleReadRegDataCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    ReadRegDataCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(Args) );

    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( Args.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem = tma::s_Allocate( sizeof( u8 ) * Args.m_DataSize );
        pReturnBuffer = new (pMem) u8[(size_t)Args.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)Args.m_DataSize );

    TMA_TRACE( "SimDebugTask", "HandleReadRegDataCommand" );

    tma::dbg::Command Command( TMA_OP_READ_REG_DATA, ProcessId, 0, sizeof(ReadRegDataCommandArgs), &Args );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RAW;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, pReturnBuffer, Args.m_DataSize );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        s_Deallocate( pReturnBuffer, sizeof( u8 ) * Args.m_DataSize );
    }
}

//==============================================================================

void SimDebugTask::HandleSetBkptCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    break_point BP;
    pPacket->ReadData( &BP, sizeof(break_point) );

    TMA_PRINTF( "HandleBreakpointCommand - Addr: %llx, State: %d, Instr: %08x\n", BP.m_Address, BP.m_State, BP.m_Instruction );

    tma::dbg::Command Command( TMA_OP_SET_BKPT, ProcessId, 0, sizeof(BP), &BP );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RAW;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, &BP, sizeof(BP) );

    TMA_PRINTF( "HandleBreakpointCommand - Addr: %llx, State: %d, Instr: %08x\n", BP.m_Address, BP.m_State, BP.m_Instruction );
}

//==============================================================================

void SimDebugTask::HandleStepCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    StepDebugCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(StepDebugCommandArgs) );
    tma::dbg::Command Command( TMA_OP_STEP_DEBUG, ProcessId, 0, sizeof(StepDebugCommandArgs), &Args );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RESULT;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void SimDebugTask::HandleWriteMemCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    WriteMemoryCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(WriteMemoryCommandArgs) );
    tma::dbg::Command Command( TMA_OP_WRITE_MEM, ProcessId, 0, sizeof(WriteMemoryCommandArgs), &Args );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RESULT;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void SimDebugTask::HandleWriteRegCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    WriteRegisterCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(WriteRegisterCommandArgs) );
    tma::dbg::Command Command( TMA_OP_WRITE_REG, ProcessId, 0, sizeof(WriteRegisterCommandArgs), &Args );
    TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RESULT;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void SimDebugTask::HandleGetNumProcesses( tmipc::Packet* pPacket )
{
    (void)pPacket;
    s32 ProcessCount = 0;
    s32 ResultCode = tmapi::RESULT_OK;
    tma::dbg::Command Command( TMA_OP_NUM_PROCESSES, NoProcessId, 0, 0, NULL );
    TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
    ResultCode = tmapi::RESULT_OK;
    // If this code was reached, then the process is running.
    ProcessCount = 1;
    Packetize( Type, NoProcessId, ResultCode, &ProcessCount, 0 );
}

//==============================================================================

void SimDebugTask::HandleWriteDumpCommand( u64 ProcessId, tmipc::Packet* pPacket )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    WriteDumpCommandArgs Args;
    pPacket->ReadData( &Args, sizeof(WriteDumpCommandArgs) );
    tma::dbg::Command Command( TMA_OP_WRITE_DUMP, ProcessId, 0, sizeof(WriteDumpCommandArgs), &Args );
    tma::dbg::TMAgent_message_type Type = TMAGENT_MESSAGE_TYPE_RESULT;
    ResultCode = tmapi::RESULT_EXECUTABLE_NOT_COMPATIBLE;
    Packetize( Type, ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void SimDebugTask::Packetize( tma::dbg::TMAgent_message_type MessageType, u64 ProcessId, s32 ResultCode, void* Data1, s32 DataSize )
{
    //TMA_TRACE( "SimDebugTask", "Packetize" );

    m_TaskType = tmipc::TaskType_DebugTask;
    tmipc::Packet* pPacket = AllocSendPacket();
    pPacket->WriteS32( MessageType );
    pPacket->WriteU64( ProcessId );
    pPacket->WriteS32( ResultCode );
    switch( MessageType )
    {
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_S32:
            pPacket->WriteS32( *(s32*)Data1 );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U32:
            pPacket->WriteU32( *(u32*)Data1 );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_S64:
            pPacket->WriteS64( *(s64*)Data1 );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64:
            pPacket->WriteU64( *(u64*)Data1 );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_MODULE:
        {
            u64 BaseAddressForCode  = 0;
            u64 TextSize            = 0;
            char Name[]             = "";
            pPacket->WriteU64( BaseAddressForCode );
            pPacket->WriteU64( TextSize );
            pPacket->WriteString( Name );
        }
        break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_PROCESS:
        {
            // copied from fs_Parameters.h (and dbg_Process.h)
            static const size_t MAX_ARCHIVE_NAME_LENGTH = 7;
            static const size_t MAX_SUB_PATH_LENGTH = 253;
            static const size_t MAX_FILE_PATH_LENGTH = MAX_ARCHIVE_NAME_LENGTH + 1 + MAX_SUB_PATH_LENGTH;
            static const size_t PROCESS_NAME_MAX_LENGTH = MAX_FILE_PATH_LENGTH + 1;  //Need enough space for the terminating 0
            s32  NumberOfModules = 0;
            u64  PID             = 0;
            u64  Handle          = 0;
            u8   Is64BitProcess  = 0;
            u8   Is64BitAddress  = 0;
            u32  NumberOfThreads = 0;
            char Data2[MAX_FILE_PATH_LENGTH];
            memset( Data2, 0, sizeof(Data2) );
            strcpy( Data2, m_ApplicationFileName );
            pPacket->WriteS32 ( NumberOfModules );
            pPacket->WriteU64 ( PID );
            pPacket->WriteU64 ( Handle );
            pPacket->WriteU8  ( Is64BitProcess );
            pPacket->WriteU8  ( Is64BitAddress );
            pPacket->WriteU32 ( NumberOfThreads );
            pPacket->WriteData( Data2, MAX_FILE_PATH_LENGTH );
        }
        break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_THREAD:
        {
            // From dbg_Process.h:
            static const s32 THREAD_NAME_MAX_LENGTH = 32;   //nn::os::ThreadNameLengthMax = 32;
            static const s32 ThreadId     = 0;
            static const u64 IP           = 0;
            static const u64 SP           = 0;
            static const u8  Status       = 0;
            static const u8  Core         = 0;
            static const u16 Priority     = 0;
            static const u16 PriorityBase = 0;
            static const u16 Flags        = 0;
            char Name[THREAD_NAME_MAX_LENGTH] = "";
            memset( Name, 0, sizeof(Name) );
            pPacket->WriteS32( ThreadId );
            pPacket->WriteU64( IP );
            pPacket->WriteU64( SP );
            pPacket->WriteU8( Status );
            pPacket->WriteU8( Core );
            pPacket->WriteU16( Priority );
            pPacket->WriteU16( PriorityBase );
            pPacket->WriteU16( Flags );
            pPacket->WriteData( Name, THREAD_NAME_MAX_LENGTH );
        }
        break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_RAW:
            pPacket->WriteData( Data1, DataSize );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_RESULT:
            break;
        default:
            pPacket->WriteString( (const char*)Data1 );
            break;
    }

    m_pServicesManager->Send( pPacket );
}

//==============================================================================

void SimDebugTask::OnInitiate( tmipc::Packet* pPacket )
{
    u32 CommandCode = 0;
    u32 u32Param = 0;
    u64 u64Param = 0;
    u64 ProcessId = NoProcessId;
    pPacket->ReadU32( CommandCode );
    pPacket->ReadU64( ProcessId );

    TMA_TRACE( "SimDebugTask", "OnInitiate( %d )", CommandCode );

    //Report on this command.
    DEJA_TRACE( "SimDebugTask::OnInitiate", "Received command: %d", CommandCode );
    DEJA_TRACE( "SimDebugTask::OnInitiate", "Got Command: %(DebugOpcodes)", CommandCode );
    // See if we have any additional parameters
    switch( CommandCode )
    {
        case TMA_OP_CONT_DEBUG:
        case TMA_OP_MODULE_INFO:
        case TMA_OP_THREAD_INFO:
            pPacket->ReadU32( u32Param );
            HandleStandardCommand( CommandCode, ProcessId, u32Param );
            break;
        case TMA_OP_SET_BKPT:
            HandleSetBkptCommand( ProcessId, pPacket );
            break;
        case TMA_OP_CLR_BKPT:
        case TMA_OP_START_DEBUG:
            pPacket->ReadU64( u64Param );
            HandleStandardCommand( CommandCode, ProcessId, u64Param );
            break;
        case TMA_OP_LOAD_ELF:
            HandleLoadElfCommand( pPacket );
            break;
        case TMA_OP_STEP_DEBUG:
            HandleStepCommand( ProcessId, pPacket );
            break;
        case TMA_OP_WRITE_MEM:
            HandleWriteMemCommand( ProcessId, pPacket );
            break;
        case TMA_OP_READ_MEM:
            HandleReadMemCommand( ProcessId, pPacket );
            break;
        case TMA_OP_READ_REG_DEFS:
            HandleReadRegDefsCommand( ProcessId, pPacket );
            break;
        case TMA_OP_READ_REG_DATA:
            HandleReadRegDataCommand( ProcessId, pPacket );
            break;
        case TMA_OP_WRITE_REG:
            HandleWriteRegCommand( ProcessId, pPacket );
            break;
        case TMA_OP_WRITE_DUMP:
            HandleWriteDumpCommand( ProcessId, pPacket );
            break;
        case TMA_OP_NUM_PROCESSES:
            HandleGetNumProcesses( pPacket );
            break;

        default:
            HandleStandardCommand( CommandCode, ProcessId, 0 );
            break;
    }

    // Task complete.
    Complete();
}

//==============================================================================

void SimDebugTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;
    //TMA_TRACE( "SimDebugTask", "OnRecvPacket %p", GetTaskId() );

    // Task complete.
    Complete();
}

//==============================================================================

void SimDebugTask::OnSendPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;
    ASSERT( 0 );
}

//==============================================================================

void SimDebugTask::Start( TMAgent_message_type MessageType, process_id ProcessId,  void* pData, s32 SizeOfData )
{
    //TMA_TRACE( "SimDebugTask", "Start %p", GetTaskId() );

    m_TaskType = tmipc::TaskType_DebugTask;
    tmipc::Packet* pPacket = AllocSendPacket( true );
    pPacket->WriteS32( MessageType );
    pPacket->WriteU64( ProcessId );
    switch( MessageType )
    {
        case tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION:
            pPacket->WriteS32( ((tma::dbg::Exception*)pData)->m_ThreadId );
            pPacket->WriteU64( ((tma::dbg::Exception*)pData)->m_IP );
            pPacket->WriteU64( ((tma::dbg::Exception*)pData)->m_SP );
            pPacket->WriteS32( ((tma::dbg::Exception*)pData)->m_ExceptionId );
            pPacket->WriteU8( ((tma::dbg::Exception*)pData)->m_ModulesChanged );
            break;

        default:
            pPacket->WriteData( pData, SizeOfData );
            break;
    }

    m_pServicesManager->SubmitTask( this, pPacket );
}

//==============================================================================

//==============================================================================
} }
//==============================================================================
