﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "..\tmagent.h"
#include "..\tma\tma_Mutex.h"
#include "..\tma\tma_Thread.h"
#include "dbg_Command.h"
#include "dbg_Message.h"
#include "dbg_Process.h"
#include "dbg_Service.h"

//==============================================================================

namespace tma { namespace dbg {

class ProcessMgr;

//==============================================================================
// Simple struct to make a linked list out of processes.
struct ProcessListNode
{
    ProcessListNode     ( ProcessMgr*            pOwner,
                          const char*            pSource,
                          const char*            pArgs,
                          nn::svc::Handle        processHandle );
    ProcessListNode     ( ProcessMgr*            pOwner,
                          process_id             PID );
    ProcessListNode     ( ProcessMgr*            pOwner,
                          process_id             PID,
                          nn::svc::Handle        processHandle );
    ~ProcessListNode    ();

    Process* GetProcess ();

    //Static list functionality
    static void AddToList( struct ProcessListNode* pNew, struct ProcessListNode** pAddToList );
    static void RemoveFromList( struct ProcessListNode* pRemove, struct ProcessListNode** pFromList );
    static void ClearList( struct ProcessListNode** pDeleteList );
    static s32  GetCountList( const struct ProcessListNode* pList );
    static void RemoveStale( int32_t NumProcesses, nn::Bit64* pProcessIds, struct ProcessListNode** pFromList );

    //Find functionality
    static struct ProcessListNode* FindInList( int AtIndex, struct ProcessListNode* pList );
    static struct ProcessListNode* FindInList( process_id ProcessId, struct ProcessListNode* pList );
    static struct ProcessListNode* FindInList( Process* pProcess, struct ProcessListNode* pList );
    static struct ProcessListNode* FindByHandleInList( s32 hProcessHandle, struct ProcessListNode* pList );

    Process*            m_pProcess;
    ProcessListNode*    m_pNext;
};

//==============================================================================

class ProcessMgr
{
public:
        ProcessMgr();
        ~ProcessMgr();

        void                    Init                    ( void );
        void                    Kill                    ( void );
        void                    OnSleep                 ( void );

        //Command-handling calls
        s32                     HandleLoadElf           ( const char*       pFileName,
                                                          const char*       pArgs,
                                                          process_id&       ProcessId );

        nn::Result              LoadExe                 ( const char*       pFileName,
                                                          const char*       pArgs,
                                                          process_id&       ProcessId );

        s32                     HandleAttachOnLaunch    ( u64               ProgramId,
                                                          process_id&       ProcessId );

        s32                     HandleAttachByProgramId ( u64               ProgramId,
                                                          process_id&       ProcessId );

        void                    HandleKillProcess       ( Command*          pCommand );

        s32                     HandleDetachProcess     ( Command*          pCommand );

        void                    HandleWriteDump         ( Command*          pCommand,
                                                          bool              WriteMiniDump = false );

        // Cancel-able CoreDump functionality
        s32                     InitiateCoreDump        ( process_id        ProcessId,
                                                          char*             pOutputName,
                                                          bool              QuickDump );
        s32                     ContinueCoreDump        ( process_id        ProcessId,
                                                          s32*              pPercentDone );
        s32                     AddTTYToCoreDump        ( process_id        ProcessId,
                                                          char*             pData,
                                                          u32               SizeOfData );
        void                    CloseCoreDump           ( process_id        ProcessId );


        void                    ProcessHasExited        ( process_id        ProcessId );

        //Called by the process
        void                    OnProcessKilled         ( Process*  pKilled );

        //Called by interested parties.
        TMAgent_message_type    HandleCommand           ( Command*          pCommandPacket,
                                                          s32&              ResultCode,
                                                          void*             pReturnBuffer );

        //Query calls
        s32                     HandleGetNumberOfProcesses( void );
        s32                     HandleProcessInfo       ( s32 iIndex, ProcessDefinition& ProcessDefinition );

        s32                     HandleGetNumberOfJITProcesses( void );
        s32                     HandleGetJITProcessInfo ( s32 iIndex, ProcessDefinition& ProcessDefinition );

protected:
        Process*                FindTargetProcess       ( Command*          pCommandPacket );
        Process*                FindTargetProcess       ( process_id        ProcessId );
        Process*                CreateTargetProcess     ( process_id        ProcessId );
        void                    Kill                    ( Process*          pProcess );

        void                    NotifyUser              ( process_id        ProcessId,
                                                          const char*       pMessage );

static  void*                   HandleException         ( Process*          pProcess,
                                                          nn::svc::DebugEventInfo& ExceptionInfo );

        void                    UpdateProcessList       (  );

        bool                    GetJITProcessList       ( int32_t*          pNumProcesses,
                                                          nn::os::ProcessId pProcessIds[],
                                                          int32_t           arraySize );
protected:
        ProcessListNode*        m_ProcessList;
        Mutex                   m_Mutex;                // Mutex to protect process list.
};

//==============================================================================
}} // namespace
//==============================================================================
