﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "dbg_BreakpointMgr.h"
#include "dbg_Process.h"
#include "../dbghlp/dbghlp.h"

#define PRINT_BREAKPOINTS TMA_MACRO_VALUE(0)

#if PRINT_BREAKPOINTS
#define BREAK_PRINTF( ... ) NN_SDK_LOG( __VA_ARGS__ )
#define RETURN              DumpBreakpoints(); return
#else
#define BREAK_PRINTF( ... )
#define RETURN              return
#endif

#if 0
#define BC_INFO( Bp, Info ) m_pParent->DropBreadcrumb( ( ( ( Bp.m_Address + Info ) | ( (u64)Bp.m_State << 28 ) ) << 32 ) | (u64)Bp.m_Instruction );
#define BC_DATA( Bp, Data ) m_pParent->DropBreadcrumb( ( Bp.m_Address << 32 ) | (u64)( Data ) );
#else
#define BC_INFO( Bp, Info )
#define BC_DATA( Bp, Data )
#endif

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

BreakpointMgr::BreakpointMgr()
:   m_ContextControlReg ( 0 )
,   m_ContextValueReg   ( 0 )
,   m_ContextReg        ( -1 )
,   m_WatchRegsMask     ( 0 )
,   m_NumWatchRegs      ( 0 )
,   m_nActive           ( 0 )
,   m_MaxActive         ( 0 )
,   m_pParent           ( NULL )
{
}

//==============================================================================

void BreakpointMgr::Init( Process* Process )
{
    m_pParent   = Process;
    //m_SBPMutex.Create();
    Reset();
}

//==============================================================================

void BreakpointMgr::Kill()
{
    m_pParent = NULL;
    //m_SBPMutex.Destroy();
}

//==============================================================================

void BreakpointMgr::Reset()
{
    m_nActive           = 0;
    m_MaxActive         = 0;
    m_NumWatchRegs      = 0;
    m_WatchRegsMask     = 0;

    m_ContextControlReg = 0x00300001; // Assign this register as a linked-contxt breakpoint
    m_ContextValueReg   = (u64)( static_cast<nnHandle>(m_pParent->GetHandle()).value ); // Context to be used for comparisons

    nn::Result  Res;

    // Find a context-aware breakpoint register to use as a linked context breakpoint for use with our watchpoint registers.
    // The context-aware register are the highest numbered breakpoint registers, so we will search backwards to find the last one.

    for( s32 reg = nn::svc::HardwareBreakPointRegisterName_I15; reg >= (s32)nn::svc::HardwareBreakPointRegisterName_I0; reg-- )
    {
        BREAK_PRINTF( "SetBreakpoint - SetHardwareBreakPoint( %2d, %016llx, %016llx )\n", reg, m_ContextControlReg, m_ContextValueReg );
        Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)reg, (nn::Bit64)m_ContextControlReg, (nn::Bit64)m_ContextValueReg ); // Enable context breakpoint

        if( Res.IsSuccess() )
        {
            m_ContextReg = reg;
            break;
        }
    }

    // Reset all of the watchpoint registers while counting how many are available.

    for( s32 reg = nn::svc::HardwareBreakPointRegisterName_D0; reg <= (s32)nn::svc::HardwareBreakPointRegisterName_D15; reg++ )
    {
        BREAK_PRINTF( "SetBreakpoint - SetHardwareBreakPoint( %2d, %016llx, %016llx )\n", reg, 0, 0 );
        Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)reg, (nn::Bit64)0, (nn::Bit64)0 ); // Disable Watchpoint

        if( ! Res.IsSuccess() )
        {
            break;
        }
        m_NumWatchRegs++;
    }

    NN_SDK_LOG( "[tma] BreakpointMgr::Reset - ContextReg: %d, NumWatchRegs: %d\n", m_ContextReg, m_NumWatchRegs );
}

void BreakpointMgr::DumpBreakpoints()
{
    NN_SDK_LOG( "[tma] Total Breakpoints: %d\n", m_nActive );

    for( s32 i = 0; i < m_nActive; i++ ) // Dump entire list of breakpoints
    {
        break_point& BP = m_Breakpoints[ i ];
        NN_SDK_LOG( "BP[ %2d ]: Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
            i, BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_HitCount, BP.m_PassCount, BP.m_Data );
    }
}

//==============================================================================
// Get break_point object at this Address and optional ThreadId

void BreakpointMgr::GetBreakpoint( break_point& BP )
{
    BC_INFO( BP, 0 );

    BP.m_State       = BP_NONE;
    BP.m_StepState   = SS_NONE;
    BP.m_Instruction = 0;

    s32 iBest = -1;

    for( s32 i = 0; i < m_nActive; i++ )
    {
        break_point& r = m_Breakpoints[i];

        if( r.m_State & BP_DATA_BOTH ) // Data breakpoint
        {
            u64 Mask = ~( ( 1 << ( BP.m_Instruction & 0x1F ) ) - 1 );
            if( r.m_Address == ( BP.m_Address & Mask ) ) // Matches start of data area
            {
                BP = r;
                break;
            }

            if( (BP.m_Address <= r.m_Address) && (r.m_Address < (BP.m_Address + 8)) )
            {
                iBest = i; // Mark as best candidate if near hit detected
            }
        }
        else if( r.m_Address == BP.m_Address )
        {
            if( ( r.m_State == BP_TEMP ) && ( r.m_Data == BP.m_Data )  ) // Thread-specific temp breakpoint
            {
                BP = r;
                break;
            }
            iBest = i; // Mark as best candidate if no thread-specific breakpoint found
        }
    }

    if( BP.m_State == BP_NONE )
    {
        if( iBest >= 0 )
        {
            BP = m_Breakpoints[ iBest ]; // Use best candidate found
        }
        else
        {
            BP.m_Data    = 0;
            m_pParent->ReadMemory( BP.m_Address, &BP.m_Instruction, sizeof(BP.m_Instruction) );

            if( IsBreakpoint( BP.m_Instruction ) ) // We are at an unmanaged breakpoint
            {
                BP.m_State = BP_NORMAL;
            }
        }
    }

    BC_INFO( BP, 0 );
    BREAK_PRINTF( "\nGetBreakpoint - Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
                    BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_HitCount, BP.m_PassCount, BP.m_Data );
}

//==============================================================================

void BreakpointMgr::SetBreakpoint( break_point& BP )
{
    //tma::ScopedLock Lock( m_SBPMutex ); // Make this call thread safe

    //DEJA_TRACE( "BreakpointMgr::SetBreakpoint", "Address: %p, State: %(bp_state)", (void*)Address, State );
    BREAK_PRINTF( "\nSetBreakpoint - Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
        BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_HitCount, BP.m_PassCount, BP.m_Data );

    // Use this old error case to get one modified breakpoint, if any

    if( BP.m_Address == 0 ) // Must not set a breakpoint at address 0
    {
        if( BP.m_State == BP_MODIFIED )
        {
            // Return ULEB pairs of Address and HitCount from breakpoints with updated hit counts,
            // where: Address == 0 implies end of complete list.
            //        Address == 1 implies end of incomplete list.

            u8* pWrite  = (u8*)&BP;
            u8* pEnd    = pWrite + sizeof(break_point);

            for( s32 i = 0; i < m_nActive; i++ )
            {
                break_point& r = m_Breakpoints[i];

                if( r.m_State & BP_MODIFIED )
                {
                    s32 Size = GetSizeULEB( r.m_Address ) + GetSizeULEB( r.m_HitCount ) + 1;

                    if( (pWrite + Size) < pEnd )
                    {
                        r.m_State   = (bp_state)( r.m_State & ~BP_MODIFIED ); // Order improves thread safety

                        EncodeULEB( pWrite, pEnd, r.m_Address );
                        EncodeULEB( pWrite, pEnd, r.m_HitCount );
                    }
                    else
                    {
                        EncodeULEB( pWrite, pEnd, 1 ); // More modified breakpoints
                        RETURN;
                    }
                }
            }
            EncodeULEB( pWrite, pEnd, 0 ); // No more modified breakpoints
            RETURN;
        }
        BP.m_State = BP_NONE;
        RETURN;
    }

    BC_INFO( BP, 1 );

    //---------------------------------
    // Removing an existing breakpoint
    //---------------------------------

    if( BP.m_State == BP_NONE ) // Removing a breakpoint
    {
        bool bTempFound = false; // Other Temp breakpoints at this address

        for( s32 i = 0; i < m_nActive; i++ ) // Search for a managed breakpoint
        {
            break_point& r      = m_Breakpoints[ i ];

            if( BP.m_Address == r.m_Address )
            {
                BP = r; // Return the breakpoint that is being deleted, to preserve any updated hit count

                if( (r.m_State & BP_DATA_BOTH) != 0 ) // Data breakpoint
                {
                    u32 WatchReg = r.m_Instruction >> 8;

                    BREAK_PRINTF( "SetBreakpoint - SetHardwareBreakPoint( %d, %016llx, %016llx )\n", WatchReg, 0, 0 );
                    nn::Result Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)WatchReg, 0, 0 ); // Disable WatchReg

                    if( ! Res.IsSuccess() )
                    {
                        NN_SDK_LOG( "SetBreakpoint - SetHardwareBreakPoint failed: %x, %08x\n", Res.GetDescription(), Res.GetInnerValueForDebug() );
                    }

                    m_WatchRegsMask &= ~( 1 << (u32)(WatchReg - nn::svc::HardwareBreakPointRegisterName_D0) );  // Mark WatchReg as available
                    BREAK_PRINTF( "SetBreakpoint - m_WatchRegsMask: %08x\n", m_WatchRegsMask );

                    if( i < (m_nActive - 1) ) // Not the last breakpoint in the array
                    {
                        r = m_Breakpoints[m_nActive - 1]; // Fill this slot with last record
                    }
                    --m_nActive; // Decrease count of active records

                    BC_DATA( BP, 0x20000 | (u64)m_nActive );
                    RETURN;
                }

                if( r.m_State == BP_TEMP ) // Multiple thread-specific TEMP breakpoints may exist at this address
                {
                    if( r.m_Data != BP.m_Data )
                    {
                        bTempFound = true;
                        continue; // Not the correct Temp breakpoint, keep searching
                    }

                    if( ! bTempFound ) // No other Temp breakpoints found yet at this address
                    {
                        for( s32 j = i + 1; j < m_nActive; j++ ) // Search for another after this one
                        {
                            break_point& s = m_Breakpoints[ j ];

                            if( ( BP.m_Address == s.m_Address ) && ( s.m_State == BP_TEMP ) )
                            {
                                bTempFound = true;

                                BC_DATA( BP, 0x30000 | (u64)j );
                                break;
                            }
                        }
                    }
                }

                if( ! bTempFound ) // No other Temp breakpoints still exist at this address
                {
                    DisableBreakpoint( r );
                }

                if( i < (m_nActive - 1) ) // Not the last breakpoint in the array
                {
                    r = m_Breakpoints[m_nActive - 1]; // Fill this slot with last record
                }
                --m_nActive; // Decrease count of active records
                BREAK_PRINTF( "SetBreakpoint - Breakpoint deleted\n" );

                BC_DATA( BP, 0x40000 | (u64)m_nActive );
                RETURN;
            }
        }

        // No managed breakpoint found

        if( BP.m_Instruction )
        {
            DisableBreakpoint( BP );
            BREAK_PRINTF( "SetBreakpoint - Unmanaged breakpoint deleted\n" );

            BC_INFO( BP, 1 );
            RETURN;
        }

        BC_DATA( BP, 0xFFFFF );

        NN_SDK_LOG( "ERROR: SetBreakpoint - Attempt to delete a non-existent breakpoint at %llx\n", BP.m_Address );
        RETURN; //TODO: What action needs to be taken?
    }

    //-------------------------------
    // Check for managed breakpoints
    //-------------------------------

    for( s32 i = 0; i < m_nActive; i++ ) // Search for an existing managed breakpoint
    {
        break_point& r = m_Breakpoints[ i ];

        if( BP.m_Address == r.m_Address )
        {
            if( BP.m_State < BP_NORMAL) // Stepping breakpoint
            {
                if( BP.m_State < r.m_State ) // Temp vs Normal or Step vs Anything
                {
                    BP = r; // Return existing stronger breakpoint state
                    BREAK_PRINTF( "SetBreakpoint - Stronger breakpoint already exists: %d\n", BP.m_State );

                    BC_DATA( BP, 0x50000 | (u64)BP.m_State );
                    RETURN;
                }

                if( r.m_State == BP_STEP ) // Apply anything over Step breakpoint
                {
                    BREAK_PRINTF( "ERROR: SetBreakpoint - Attempt to replace Step breakpoint at %llx\n", BP.m_Address );

                    BC_DATA( BP, 0x60000 );
                    RETURN;
                }

                // Look for duplicate Temp breakpoints

                for( s32 j = i; j < m_nActive; j++ )
                {
                    break_point& s = m_Breakpoints[ j ];

                    if( ( BP.m_Address == s.m_Address ) && ( s.m_State == BP_TEMP ) && ( s.m_Data == BP.m_Data ) )
                    {
                        BC_DATA( BP, 0x70000 );
                        RETURN; // Temp breakpoint already exists
                    }
                }

                if( m_nActive >= USR_BREAKPOINTS )
                {
                    BREAK_PRINTF( "ERROR: SetBreakpoint - Too many breakpoints\n" );
                    BP.m_State = BP_NONE; // Discard this request and return BP_NONE as state

                    BC_DATA( BP, 0xF0000 );
                    RETURN;
                }

                break_point& s = m_Breakpoints[ m_nActive ]; // Get the next available breakpoint from the array

                if( ++m_nActive > m_MaxActive )
                {
                    m_MaxActive = m_nActive;
                    BREAK_PRINTF( "SetBreakpoint - New MaxActive: %d\n", m_MaxActive );
                }
                s = r;
                s.m_Data = BP.m_Data; // Clone the existing breakpoint

                BC_DATA( BP, 0xC0000 );
                RETURN;
            }
            else // Update existing special breakpoint
            {
                BREAK_PRINTF( "SetBreakpoint - Replacing BP[%d] - Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
                    i, r.m_Address, r.m_State, r.m_StepState, r.m_Instruction, r.m_HitCount, r.m_PassCount, r.m_Data );
                r = BP;
            }

            BC_DATA( BP, 0x80000 );
            RETURN;
        }
    }

    //---------------------------------
    // Check for unmanaged breakpoints
    //---------------------------------

    if( (BP.m_State & BP_DATA_BOTH) == 0 ) // Not a data breakpoint
    {
        m_pParent->ReadMemory( BP.m_Address, &BP.m_Instruction, sizeof(BP.m_Instruction) ); // Get original instruction for disable

        if( IsBreakpoint( BP.m_Instruction ) ) // Existing unmanaged breakpoint found (either BP_NORMAL or user-coded)
        {
            if( BP.m_State < BP_NORMAL) // Stepping breakpoint
            {
                BP.m_State = BP_NORMAL; // Return existing stronger breakpoint state
                BREAK_PRINTF( "SetBreakpoint - Stronger breakpoint already exists: %d\n", BP.m_State );

                BC_DATA( BP, 0x90000 + (u64)BP.m_State );
                RETURN;
            }
            else if( BP.m_State == BP_NORMAL )
            {
                BREAK_PRINTF( "ERROR: SetBreakpoint - Breakpoint already exists: %d\n", BP.m_State );

                BC_DATA( BP, 0xA0000 + (u64)BP.m_State );
                RETURN;
            }
            // Continue processing to apply special breakpoint
        }

        if( BP.m_State == BP_NORMAL )
        {
            EnableBreakpoint( BP ); // Just write the breakpoint, TargetManager will manage it
            BREAK_PRINTF( "SetBreakpoint - New unmanaged breakpont - Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
                BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_HitCount, BP.m_PassCount, BP.m_Data );

            BC_INFO( BP, 1 );
            RETURN;
        }
    }

    //---------------------------------
    // Create a new managed breakpoint
    //---------------------------------

    if( m_nActive >= USR_BREAKPOINTS )
    {
        if( BP.m_State != BP_STEP )
        {
            NN_SDK_LOG( "ERROR: SetBreakpoint - Too many managed breakpoints\n" );
            BP.m_State = BP_NONE; // Discard this request and return BP_NONE as state

            BC_DATA( BP, 0xD0000 );
            RETURN;
        }
        ASSERT( m_nActive < MAX_BREAKPOINTS );
    }

    nn::Result  Res;
    u32         WatchReg    = nn::svc::HardwareBreakPointRegisterName_D0;
    u64         ControlReg  = 0;
    u64         ValueReg    = 0;

    if( (BP.m_State & BP_DATA_BOTH) != 0 ) // Data breakpoint
    {
        if( m_ContextReg == -1 ) // Context reg not initialized, no data breakpoint support
        {
            NN_SDK_LOG( "ERROR: SetBreakpoint - Context breakpoint register not found\n" );
            BP.m_State = BP_NONE; // Discard this request and return BP_NONE as state

            BC_DATA( BP, 0xD1000 );
            RETURN;
        }

        if( m_WatchRegsMask == ((1 << m_NumWatchRegs) - 1) )
        {
            NN_SDK_LOG( "ERROR: SetBreakpoint - All watchpoint registers already in use\n" );
            BP.m_State = BP_NONE; // Discard this request and return BP_NONE as state

            BC_DATA( BP, 0xD2000 );
            RETURN;
        }
    }

    break_point& r = m_Breakpoints[ m_nActive ]; // Get the next available breakpoint from the array

    if( ++m_nActive > m_MaxActive )
    {
        m_MaxActive = m_nActive;
        BREAK_PRINTF( "SetBreakpoint - New MaxActive: %d\n", m_MaxActive );
    }
    r = BP;

    if( (BP.m_State & BP_DATA_BOTH) != 0 ) // Data breakpoint
    {
        u32 Selector        = 1;
        u32 Log2DataSize    = BP.m_Instruction & 0x001F;

        for( s32 i = 0; i < MAX_WATCH_REGS; i++ )
        {
            if( (m_WatchRegsMask & Selector) == 0 ) // Watch register is available
            {
                m_WatchRegsMask    |= Selector; // Claim use of watch register
                WatchReg            = i + (u32)nn::svc::HardwareBreakPointRegisterName_D0;
                BP.m_Instruction    = ( WatchReg << 8 ) | Log2DataSize;
                BREAK_PRINTF( "SetBreakpoint - m_WatchRegsMask: %08x\n", m_WatchRegsMask );

                r = BP; // Update breakpoint in array of active breakpoints
                break;
            }
            Selector <<= 1;
        }

        u64 Mask    = Log2DataSize; // Set values for powers of 2 >= 8
        u64 BAS     = 0x00FF;
        u64 AMask   = (1 << Log2DataSize) - 1;

        if( Log2DataSize < 3 ) // Less than 8 bytes (could also be used for 8)
        {
            Mask    = 0;
            BAS     = ( 1 << (1 << Log2DataSize) ) - 1; // Get Byte Adddress Select field

            BAS   <<= BP.m_Address & 0x07; // Adjust for offset from start of dword

            BAS    &= 0x00FF; // Ensure BAS will fit in field

            AMask   = 0x0007; // Force to 8-bit boundary
        }

        ControlReg  = (Mask << 24) | (m_ContextReg << 16) | (BAS << 5) | ( (BP.m_State & BP_DATA_BOTH) >> 1 ) | 1; // Place fields in reg
        ValueReg    = BP.m_Address & ~AMask; // Ensure correct alignment

        BREAK_PRINTF( "SetBreakpoint - SetHardwareBreakPoint( %d, %016llx, %016llx )\n", WatchReg, ControlReg, ValueReg );
        Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)WatchReg, (nn::Bit64)ControlReg, (nn::Bit64)ValueReg ); // Enable data breakpoint

        if( ! Res.IsSuccess() )
        {
            NN_SDK_LOG( "SetBreakpoint - SetHardwareBreakPoint failed: %x, %08x\n", Res.GetDescription(), Res.GetInnerValueForDebug() );
            BP.m_State = BP_NONE; // Discard this request and return BP_NONE as state
            SetBreakpoint( BP );  // Remove this breakpoint from the list

            BC_DATA( BP, 0xD3000 );
            RETURN;
        }
    }
    else
    {
        EnableBreakpoint( BP );
    }

    BC_INFO( BP, 1 );
    BREAK_PRINTF( "SetBreakpoint - New managed breakpont - Addr:%llX, State:%02X, StepState:%d, Instr:%08X, HitCount:%d, PassCount:%d, Data:%lld\n",
        BP.m_Address, BP.m_State, BP.m_StepState, BP.m_Instruction, BP.m_HitCount, BP.m_PassCount, BP.m_Data );
    RETURN;

} // NOLINT(readability/fn_size)

//==============================================================================

void BreakpointMgr::ClearBreakpoints( bp_state State, u64 ThreadId )
{
    //DEJA_TRACE( "BreakpointMgr::ClearBreakpoints", "State: %(bp_state)", State );
    BREAK_PRINTF( "ClearBreakpoints( %d, %lld )\n", State, ThreadId );

    s32 i = 0;

    while( i < m_nActive )
    {
        break_point& r = m_Breakpoints[i];

        if( r.m_State == State )
        {
            if( (State != BP_TEMP) || (ThreadId == 0) || (ThreadId == r.m_Data) )
            {
                r.m_State = BP_NONE;

                if( State == BP_TEMP )
                {
                    r.m_Data = 0; // Make sure this Temp breakpoint is deleted
                }
                SetBreakpoint( r );
            }
            else
            {
                i++;
            }
        }
        else
        {
            i++;
        }
    }
}

//==============================================================================

bool BreakpointMgr::IsBreakpoint( u32 Instruction )
{
    if(    ( Instruction == ARM_BKPT )
        || ( Instruction == SDK_BKPT )
        || ( m_pParent->Is64Bit()
          && ( ( ( Instruction & A64_BRK_MASK  ) == A64_BRK  )
            || ( ( Instruction & A64_HLT_MASK  ) == A64_HLT  ) ) )
        || ( ! m_pParent->Is64Bit()
          && ( ( ( Instruction & A32_BKPT_MASK ) == A32_BKPT )
            || ( ( Instruction & T16_BKPT_MASK ) == T16_BKPT ) ) ) )
    {
        return( true );
    }
    return( false );
}

//==============================================================================

void BreakpointMgr::DisableBreakpoint( break_point& BP )
{
    BC_INFO( BP, 2 );

    BREAK_PRINTF( "DisableBreakpoint( %llx, %x, %08x )\n", BP.m_Address, BP.m_State, BP.m_Instruction );

    if( (BP.m_State & BP_DATA_BOTH) != 0 ) // Data breakpoint
    {
        BREAK_PRINTF( "DisableBreakpoint - SetHardwareBreakPoint( %d, %016llx, %016llx )\n", m_ContextReg, (nn::Bit64)0, (nn::Bit64)0 );
        nn::Result Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)m_ContextReg, (nn::Bit64)0, (nn::Bit64)0 ); // Disable all data breakpoints

        if( ! Res.IsSuccess() )
        {
            NN_SDK_LOG( "DisableBreakpoint - SetHardwareBreakPoint failed: %x, %08x\n", Res.GetDescription(), Res.GetInnerValueForDebug() );
        }
    }
    else
    {
        u32 Instruction = BP.m_Instruction;

        if( IsBreakpoint( Instruction ) )
        {
            NN_SDK_LOG( "DisableBreakpoint: Replacing breakpoint %08X at %llx with NOP\n", BP.m_Instruction, BP.m_Address );
            Instruction = m_pParent->Is64Bit() ? (u32)A64_NOP : (u32)A32_NOP;
        }
        m_pParent->WriteMemory( BP.m_Address, &Instruction, sizeof(Instruction) );
    }
    BC_INFO( BP, 2);
}

//==============================================================================

void BreakpointMgr::EnableBreakpoint( break_point& BP )
{
    BC_INFO( BP, 3 )

    BREAK_PRINTF( "EnableBreakpoint( %llx, %x )\n", BP.m_Address, BP.m_State );

    if( (BP.m_State & BP_DATA_BOTH) != 0 ) // Data breakpoint
    {
        BREAK_PRINTF( "EnableBreakpoint - SetHardwareBreakPoint( %d, %016llx, %016llx )\n", m_ContextReg, m_ContextControlReg, m_ContextValueReg );
        nn::Result Res = dbghlp::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)m_ContextReg, (nn::Bit64)m_ContextControlReg, (nn::Bit64)m_ContextValueReg ); // Enable all data breakpoints

        if( ! Res.IsSuccess() )
        {
            NN_SDK_LOG( "EnableBreakpoint - SetHardwareBreakPoint failed: %x, %08x\n", Res.GetDescription(), Res.GetInnerValueForDebug() );
        }
    }
    else
    {
        u32 Bkpt =  (u32)( m_pParent->Is64Bit() ? SDK_BKPT : ARM_BKPT );

        m_pParent->WriteMemory( BP.m_Address, &Bkpt, sizeof(Bkpt) );
    }
    BC_INFO( BP, 3 )
}

//==============================================================================

void BreakpointMgr::PatchReadMemory( void* pBuffer, u64 address, u32 nBytes )
{
    u8* p      = (u8*)pBuffer;
    u64 aStart = address;
    u64 aEnd   = aStart + nBytes;

    for( s32 i = 0; i < m_nActive; i++ )
    {
        break_point& r = m_Breakpoints[i];

        if( (r.m_State & BP_DATA_BOTH) == 0 )
        {
            u64 bStart = r.m_Address;
            u64 bEnd   = bStart + sizeof(r.m_Instruction);

            if( (bStart < aEnd) && (aStart < bEnd) )
            {
                s32 nInstBytes   = sizeof(r.m_Instruction);
                u64 a            = r.m_Address;
                u8* pInstruction = (u8*)&r.m_Instruction;

                for( s32 i=0; i < nInstBytes; i++ )
                {
                    if( (a >= aStart) && (a < aEnd) )
                    {
                        s64 offset = (s64)(a - address);
                        ASSERT( (offset >= 0) && (offset < nBytes) );
                        p[offset] = pInstruction[i];
                    }
                    a++;
                }
            }
        }
    }
}

//==============================================================================
}} // namespace
//==============================================================================
